package ru.yandex.chemodan.app.lentaloader.cool.model;

import lombok.Data;
import org.junit.Test;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.inside.utils.ConstLocalizedString;
import ru.yandex.inside.utils.DynamicLocalizedString;
import ru.yandex.inside.utils.Language;
import ru.yandex.inside.utils.LocalizedString;
import ru.yandex.misc.bender.BenderMapper;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.bender.config.BenderConfiguration;
import ru.yandex.misc.bender.config.BenderSettings;
import ru.yandex.misc.bender.config.CustomMarshallerUnmarshallerFactoryBuilder;
import ru.yandex.misc.random.Random2;
import ru.yandex.misc.test.Assert;

/**
 * @author tolmalev
 */
public class LocalizedStringMarshallerUnmarshallerTest {

    private final BenderMapper mapper = new BenderMapper(new BenderConfiguration(
            new BenderSettings(),
            CustomMarshallerUnmarshallerFactoryBuilder
                    .cons()
                    .add(LocalizedString.class,
                            new LocalizedStringMarshallerUnmarshaller(),
                            new LocalizedStringMarshallerUnmarshaller())
                    .build()));

    @Test
    public void constStrings() {
        testParseSerialize(new WithLocalized(
                new ConstLocalizedString("aaa"),
                Option.of(new ConstLocalizedString("bbb")),
                Cf.list(new ConstLocalizedString("ccc1"), new ConstLocalizedString("ccc2")),
                Cf.map(
                        "a", new ConstLocalizedString("aaa1"),
                        "b", new ConstLocalizedString("aaa2"),
                        "c", new ConstLocalizedString("aaa3")
                )
        ));
    }

    @Test
    public void dynamicStrings() {
        testParseSerialize(new WithLocalized(
                randomDynString(),
                Option.of(randomDynString()),
                Cf.list(randomDynString(), randomDynString()),
                Cf.map(
                        "a", randomDynString(),
                        "b", randomDynString(),
                        "c", randomDynString()
                )
        ));
    }

    @Test
    public void fullRandom() {
        for (Integer i : Cf.range(0, 100)) {
            testParseSerialize(new WithLocalized(
                    randomString(),
                    Option.when(Random2.R.nextBoolean(), randomString()),
                    Cf.range(0, Random2.R.nextInt(4)).map(ii -> randomString()),
                    Cf.range(0, Random2.R.nextInt(4)).toMap(ii ->
                            Tuple2.tuple(Random2.R.nextAlnum(3), randomString())
                    )
            ));
        }
    }

    private LocalizedString randomString() {
        return Random2.R.nextBoolean()
                ? randomDynString()
                : new ConstLocalizedString(Random2.R.nextAlnum(10));
    }

    private DynamicLocalizedString randomDynString() {
        MapF<Language, String> values = Random2.R
                .randomElements(Cf.x(Language.values()), 1 + Random2.R.nextInt(Language.values().length - 1))
                .zipWith(language -> Random2.R.nextAlnum(10))
                .toMap();
        return new DynamicLocalizedString(values);
    }

    private void testParseSerialize(WithLocalized o) {
        byte[] serialized = mapper.serializeJson(o);
        WithLocalized parsed = mapper.parseJson(WithLocalized.class, serialized);

        Assert.equals(o, parsed);
    }

    @BenderBindAllFields
    @Data
    private static class WithLocalized {
        public final LocalizedString text1;
        public final Option<LocalizedString> text2;
        public final ListF<LocalizedString> text3;
        public final MapF<String, LocalizedString> texts;
    }
}
