package ru.yandex.chemodan.app.lentaloader.cool.model;

import java.util.Arrays;

import lombok.Data;
import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.lentaloader.cool.utils.GenerationInterval;
import ru.yandex.chemodan.app.lentaloader.cool.utils.IntervalType;
import ru.yandex.chemodan.app.lentaloader.cool.utils.TimeIntervalUtils;
import ru.yandex.chemodan.mpfs.MpfsResourceId;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;

/**
 * @author tolmalev
 */
@Data
@BenderBindAllFields
public class MinimalCoolLentaBlock {
    public final String id;
    public final String generationType;

    public final ListF<MpfsResourceId> resourceIds;
    public final MpfsResourceId bestResourceId;

    public final String userTimezoneId;

    public final Option<Instant> photosliceDate;
    public final Instant minDate;
    public final Instant maxDate;

    public final Option<Instant> lastShowDate;
    public final Option<Instant> legacyLentaLastShowDate;
    public final Option<Instant> legacyLentaLastPushDate;

    public final boolean pushSent;

    /** checks only part of fields
     * - resourceIds
     * - bestResourceIds
     * - userTimezoneId
     * - photosliceDate
     * - minDate
     * - maxDate
     *
     * id and titles are ignored
     */
    public boolean hasSameData(MinimalCoolLentaBlock other) {
        return bestResourceId.equals(other.resourceIds)
                && photosliceDate.equals(other.photosliceDate)
                && minDate.equals(other.minDate)
                && maxDate.equals(other.maxDate)
                && userTimezoneId.equals(other.userTimezoneId)
                && Arrays.equals(resourceIds.toArray(), other.resourceIds.toArray());
    }

    public DateTime getUserMinDate() {
        return new DateTime(minDate, DateTimeZone.forID(userTimezoneId));
    }

    public DateTime getUserMaxDate() {
        return new DateTime(maxDate, DateTimeZone.forID(userTimezoneId));
    }

    public MinimalCoolLentaMordaBlock toMordaBlock(String id, Instant mtime) {
        return new MinimalCoolLentaMordaBlock(id, generationType, this.id, resourceIds, bestResourceId, userTimezoneId, photosliceDate,
                minDate, maxDate, mtime, lastShowDate, legacyLentaLastShowDate, legacyLentaLastPushDate, pushSent);
    }

    public MinimalCoolLentaBlock withLastShowDates(Option<Instant> lastShowDate, Option<Instant> legacyLentaLastShowDate) {
        return new MinimalCoolLentaBlock(id, generationType, resourceIds, bestResourceId, userTimezoneId, photosliceDate, minDate, maxDate,
                lastShowDate, legacyLentaLastShowDate, legacyLentaLastPushDate, pushSent);
    }

    public MinimalCoolLentaBlock withPushSent(boolean pushSent) {
        return new MinimalCoolLentaBlock(id, generationType, resourceIds, bestResourceId, userTimezoneId, photosliceDate, minDate, maxDate,
                lastShowDate, legacyLentaLastShowDate, legacyLentaLastPushDate, pushSent);
    }

    public GenerationInterval getGenerationInterval() {
        return new GenerationInterval(getUserMinDate(), getUserMaxDate());
    }

    public IntervalType getIntervalType() {
        return TimeIntervalUtils.getMinimalInterval(getUserMinDate(), getUserMaxDate());
    }

    public DateTime getIntervalStart() {
        return getIntervalType().getIntervalStart(getUserMinDate());
    }

    public DateTime getIntervalEnd() {
        return getIntervalType().getIntervalEnd(getIntervalStart());
    }
}
