package ru.yandex.chemodan.app.lentaloader.cool.worker;

import lombok.Data;
import org.jetbrains.annotations.NotNull;
import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.joda.time.LocalDate;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.function.Function1V;
import ru.yandex.chemodan.app.lentaloader.cool.utils.IntervalType;
import ru.yandex.chemodan.app.lentaloader.cool.utils.Season;
import ru.yandex.chemodan.app.lentaloader.cool.utils.TimeIntervalUtils;
import ru.yandex.chemodan.util.blackbox.UserTimezoneHelper;
import ru.yandex.chemodan.util.yt.YqlHelper;
import ru.yandex.chemodan.util.yt.YtHelper;
import ru.yandex.commune.bazinga.BazingaTaskManager;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.commune.util.RetryUtils;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.tables.YTableEntryTypes;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.env.EnvironmentType;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.random.Random2;
import ru.yandex.misc.time.TimeUtils;

/**
 * @author tolmalev
 */
public class CoolLentaRoutines {
    private static final Logger logger = LoggerFactory.getLogger(CoolLentaRoutines.class);

    private static final SetF<IntervalType> IMMEDIATELY_SCHEDULE_TYPES = Cf.set(IntervalType.ONE_DAY, IntervalType.WEEKEND, IntervalType.WEEK);
    private static final int WAIT_FOR_TABLE_DURATION_MINUTES = 30;

    private final YtHelper yt;
    private final YqlHelper yqlHelper;
    private final UserTimezoneHelper userTimezoneHelper;
    private final BazingaTaskManager bazingaTaskManager;

    private final YPath ytHomePath;

    private final int scheduleBatchSize;

    private final DynamicProperty<Double> scheduleProcessOneDayPercent = new DynamicProperty<>("cool-lenta-regenerate-blocks-percent", 0.0);
    private final DynamicProperty<ListF<String>> scheduleProcessOneDayUids = new DynamicProperty<>("cool-lenta-regenerate-blocks-uids", Cf.list());

    public CoolLentaRoutines(YtHelper yt, YqlHelper yqlHelper,
            UserTimezoneHelper userTimezoneHelper, BazingaTaskManager bazingaTaskManager, YPath ytHomePath,
            int scheduleBatchSize)
    {
        this.yt = yt;
        this.yqlHelper = yqlHelper;
        this.userTimezoneHelper = userTimezoneHelper;
        this.bazingaTaskManager = bazingaTaskManager;
        this.ytHomePath = ytHomePath;
        this.scheduleBatchSize = scheduleBatchSize;
    }

    public void processYesterdayChanges() {
        processImagesChanges(LocalDate.now().minusDays(1), false);
    }

    public void processImagesChanges(LocalDate date, boolean forceYql) {
        String inputPath = "//logs/ydisk-event-history-log/1d/" + date;
        String outputPath = getUserChangedDaysPath(date);

        processChangesInDay(inputPath, outputPath, forceYql);
    }

    public void scheduleFinishedIntervalTasks() {
        scheduleFinishedIntervalTasks(DateTime.now());
    }

    public void scheduleFinishedIntervalTasks(DateTime now) {
        DateTime yesterday = now.minusDays(1);
        if (now.getDayOfMonth() == 1) {
            logger.info("Schedule month {}.{} processing", yesterday.getYear(), yesterday.getMonthOfYear());
            DateTime timeInInterval = new DateTime(yesterday.getYear(), yesterday.getMonthOfYear(), 15, 0, 0);
            bazingaTaskManager.schedule(new ScheduleOldIntervalGeneration(IntervalType.MONTH, timeInInterval.toInstant()));
        } else if (now.getDayOfMonth() == 2) {
            Cf.x(Season.values()).find(season -> season.getMonths().last() == now.minusDays(15).getMonthOfYear())
                    .forEach(season -> {
                        logger.info("Schedule season {} {} processing", season, yesterday.getYear());
                        DateTime timeInInterval = new DateTime(yesterday.getYear(), season.getMonths().last(), 15, 0, 0);
                        bazingaTaskManager.schedule(new ScheduleOldIntervalGeneration(IntervalType.SEASON, timeInInterval.toInstant()));
                    });
        } else if (now.getDayOfMonth() == 5) {
            if (now.getMonthOfYear() == 1) {
                logger.info("Schedule year {} processing", yesterday.getYear());
                DateTime timeInInterval = new DateTime(yesterday.getYear(), 6, 1, 0, 0);
                bazingaTaskManager.schedule(new ScheduleOldIntervalGeneration(IntervalType.YEAR, timeInInterval.toInstant()));
            }
        } else {
            logger.info("Don't schedule anything");
        }
    }

    public void scheduleOldIntervalGeneration(IntervalType type, Instant timeInInterval) {
        DateTime inIntervalDt = timeInInterval.toDateTime(TimeUtils.EUROPE_MOSCOW_TIME_ZONE);

        DateTime intervalStart = type.getIntervalStart(inIntervalDt);
        DateTime intervalEnd = type.getIntervalEnd(inIntervalDt);

        YPath tmpOutputPath = ytHomePath.child("tmp").child("old-intervals-" + type + "-" + timeInInterval.getMillis());
        String yql = getYqlForOldIntervals(intervalStart, intervalEnd, tmpOutputPath.toString());

        yqlHelper.execute(yql);

        long rowCount = yt.getRowCount(tmpOutputPath);
        logger.debug("Result row count = {}. Going to schedule {} image processing tasks for old interval", rowCount, rowCount / 1000);

        long startRow = 0;
        while (startRow < rowCount) {
            long endRow = Math.min(startRow + scheduleBatchSize, rowCount);

            bazingaTaskManager.schedule(new ScheduleProcessOneDayTask(tmpOutputPath, startRow, endRow, type));

            startRow = endRow;
        }
    }

    @NotNull
    String getYqlForOldIntervals(DateTime intervalStart, DateTime intervalEnd, String tmpOutputPath) {
        long minHour = intervalStart.getMillis() / 1000 / 3600;
        long maxHour = intervalEnd.getMillis() / 1000 / 3600;

        long midHour = (minHour + maxHour) / 2;

        return "PRAGMA yt.DefaultOperationWeight=\"50.0\";\n" +
                "USE hahn;\n" +
                "\n" +
                "INSERT INTO `" + tmpOutputPath + "` WITH TRUNCATE\n" +
                "SELECT \n" +
                "   uid, \n" +
                "   " + midHour + " as hour \n" +
                "FROM RANGE(" +
                    "`" + getUserChangedDaysRoot() + "`, " +
                    "`" + intervalStart.minusDays(1).toLocalDate() + "`, " +
                    "`" + intervalEnd.plusDays(1).toLocalDate() + "`" +
                ")\n" +
                "WHERE hour >= " + minHour + "\n" +
                "AND hour <= " + maxHour + "\n" +
                "GROUP BY uid;\n";
    }

    public void processChangesInDay(String inputPath, String outputPath, boolean forceYql) {
        yt.waitForTableForMinutes(YPath.simple(inputPath), WAIT_FOR_TABLE_DURATION_MINUTES);

        String yql = "PRAGMA yt.ForceInferSchema=\"1000\";\n" +
                "PRAGMA yt.DefaultOperationWeight=\"50.0\";\n" +
                "USE hahn;\n" +
                "\n" +
                "INSERT INTO {{output_path}}\n" +
                "SELECT \n" +
                "    uid, hour, count(*) as cnt\n" +
                "FROM {{input_path}} \n" +
                "WHERE WeakField(uid, \"String\", null) is not null \n" +
                "AND WeakField(tgt_etime, \"String\", null) is not null\n" +
                "GROUP BY \n" +
                "    WeakField(uid, \"String\") as uid, \n" +
                "    CAST(WeakField(tgt_etime, \"String\", null) as Int64) / 3600 as hour\n" +
                "HAVING hour > 0\n" +
                "ORDER BY uid, hour;";

        yql = yql
                .replace("{{input_path}}", "`" + inputPath + "`")
                .replace("{{output_path}}", "`" + outputPath + "`");

        YPath outputYPath = YPath.simple(outputPath);

        // Если таблица уже есть - незачем заново ее генерировать
        if (forceYql || !yt.existsWithRetries(outputYPath)) {
            yqlHelper.execute(yql);
        }

        long rowCount = yt.getRowCount(outputYPath);
        logger.debug("Result row count = {}. Going to schedule {} image processing tasks", rowCount, rowCount / 1000);

        long startRow = 0;
        while (startRow < rowCount) {
            long endRow = Math.min(startRow + scheduleBatchSize, rowCount);

            bazingaTaskManager.schedule(new ScheduleProcessOneDayTask(outputYPath, startRow, endRow));

            startRow = endRow;
        }
    }

    String buildCurrentCoolLentaSnapshotYql(String logsDateFrom, String currentSnapshotPath, ListF<String> outputPaths) {
        String yql = "PRAGMA yt.ForceInferSchema=\"1000\";\n" +
                "PRAGMA yt.DefaultOperationWeight=\"50.0\";\n" +
                "USE hahn;\n" +
                "\n" +
                "$input = (\n" +
                "   SELECT * FROM RANGE(`//logs/ydisk-lenta-cool-events-log/1d`, `" + logsDateFrom + "`)\n" +
                "   UNION ALL SELECT * from `" + currentSnapshotPath + "`\n" +
                ");\n" +
                "\n" +
                "$snapshot = (SELECT \n" +
                "   uid,\n" +
                "   block_collection,\n" +
                "   block_id,\n" +
                "   MAX(revision) as revision,\n" +
                "   MAX_BY(block, revision) as block,\n" +
                "   MAX_BY(event_type, revision) as event_type\n" +
                "   FROM $input\n" +
                "   WHERE revision IS NOT NULL\n" +
                "   GROUP BY uid, block_collection, block_id\n" +
                "   HAVING MAX_BY(event_type, revision) NOT IN ('all-blocks-delete', 'morda-blocks-delete')\n" +
                ");\n\n";

        for (String outputPath : outputPaths) {
            yql += "INSERT INTO `" + outputPath + "` WITH TRUNCATE SELECT * FROM $snapshot ORDER BY uid, block_collection, block_id; \n";
        }
        return yql;
    }

    public void scheduleProcessOneDayTasks(YPath pathWithRange, Option<IntervalType> intervalTypeO) {
        SetF<Tuple2<Long, Long>> scheduled = Cf.hashSet();
        MapF<Long, DateTimeZone> timezoneByUser = Cf.hashMap();

        Function1V<UidAndHour> createPhotoTaskF = uidAndHour -> {
            RetryUtils.retryE(logger, 2, () -> {
                long uid = uidAndHour.uid;
                long hour = uidAndHour.hour;

                if (Random2.R.nextDouble() * 100 < scheduleProcessOneDayPercent.get()
                        || scheduleProcessOneDayUids.get().containsTs(Long.toString(uid))) {
                    try {
                        DateTimeZone userTimezone = timezoneByUser.getOrElseUpdate(uid, () -> userTimezoneHelper.getUserTimezone(uid));
                        DateTime userTime = new DateTime(hour * 3600L * 1000, userTimezone);
                        Instant timeToProcess = TimeIntervalUtils.getDayStart(userTime).toInstant();

                        if (userTime.isAfter(Instant.now().plus(Duration.standardDays(5)))) {
                            logger.debug("Skip file from future: uid={}, hour={}, userTime={}", uid, hour, userTime);
                        }

                        if (EnvironmentType.getActive() != EnvironmentType.PRODUCTION) {
                            logger.debug("Processing line uid={}, hour={}, instantMs={}, userTimezone={}, userTime={}, timeToProcess={}, userTimeMs={}, timeToProcessMs={}",
                                    uid, hour, userTimezone, userTime, timeToProcess, userTime.getMillis(), timeToProcess.getMillis());
                        }

                        // исходная таблица тасков отсортирована по uid, hour поэтому скорее всго один день для юзера попадет в одну пачку
                        // и эта оптимизация поможет не ходить в базу лишний раз
                        Tuple2<Long, Long> sTuple = new Tuple2<>(uid, timeToProcess.getMillis());
                        if (scheduled.containsTs(sTuple)) {
                            if (EnvironmentType.getActive() != EnvironmentType.PRODUCTION) {
                                logger.debug("Skip already scheduled, uid={}, hour={}, userTime={}, timeToProcess={}",
                                        uid, hour, userTime, timeToProcess.getMillis());
                            }
                            return;
                        }

                        // Если передали intervalTypeO - процессим только его
                        // Иначе все возможные варианты для этой даты
                        ListF<IntervalType> intervalTypes = intervalTypeO.isPresent() ? intervalTypeO : IntervalType.getTypesForDateTime(userTime);

                        for (IntervalType intervalType : intervalTypes) {
                            // Для месяца и больше шедулинг через YT чтобы не забивать qdb горой тасок
                            // Но только для тасков, котрые хотели поставить на будущее
                            Instant timeToSchedule = timeToSchedule(userTime, intervalType);
                            if (timeToSchedule.isAfter(Instant.now().plus(Duration.standardHours(1)))) {
                                if (!IMMEDIATELY_SCHEDULE_TYPES.containsTs(intervalType)) {
                                    continue;
                                }
                            }
                            // 2 разных таска нужно для того чтобы видеть реальную очередь того, что мы не успеваем обрабатывать
                            if (timeToSchedule.isAfter(Instant.now().plus(Duration.standardHours(1)))) {
                                bazingaTaskManager.schedule(
                                        new ProcessOneFutureIntervalTask(new PassportUid(uid), timeToProcess, intervalType),
                                        timeToSchedule
                                );
                            } else {
                                timeToSchedule = Random2.R.nextInstant(Instant.now(), Instant.now().plus(Duration.standardHours(6)));
                                bazingaTaskManager.schedule(
                                        new ProcessOneIntervalTask(new PassportUid(uid), timeToProcess, intervalType),
                                        timeToSchedule
                                );
                            }
                        }

                        scheduled.add(sTuple);

                    } catch(Exception e) {
                        logger.error("Can't schedule ProcessOneIntervalTask for {} at {}", uid, hour);
                        throw e;
                    }
                }
            });
        };

        yt.runWithRetries(
                () -> yt.tables().read(pathWithRange, YTableEntryTypes.bender(UidAndHour.class), createPhotoTaskF));
    }

    static Instant timeToSchedule(DateTime userTime, IntervalType intervalType) {
        return timeToSchedule(Instant.now(), userTime, intervalType);
    }

    static Instant timeToSchedule(Instant now, DateTime userTime, IntervalType intervalType) {
        if (TimeIntervalUtils.isIntervalFinished(intervalType, intervalType.getIntervalStart(userTime), now)) {
            // размазываем в прошлое потому что при фетчинге задач идет сортировка по priority, schedule_time
            // таким образом мы для одного юзера размажем запросы и будем получать меньше 529
            return Random2.R.nextInstant(now.minus(Duration.standardHours(1)), now);
        }
        DateTime result = intervalType.getIntervalEnd(userTime);
        Duration scheduleOffset;
        Duration randomizationDuration;

        switch (intervalType) {
            case ONE_DAY:
            case WEEKEND:
                scheduleOffset = Duration.ZERO;
                randomizationDuration = Duration.standardHours(1); break;
            case WEEK:
                scheduleOffset = Duration.ZERO;
                randomizationDuration = Duration.standardDays(1); break;
            case MONTH:
                scheduleOffset = Duration.ZERO;
                randomizationDuration = Duration.standardDays(7); break;
            case SEASON:
                scheduleOffset = Duration.standardDays(2);
                randomizationDuration = Duration.standardDays(7); break;
            case YEAR:
                scheduleOffset = Duration.standardDays(6);
                randomizationDuration = Duration.standardDays(30); break;
            default:
                scheduleOffset = Duration.ZERO;
                Duration intervalDuration = new Duration(intervalType.getIntervalStart(userTime), intervalType.getIntervalStart(userTime));
                randomizationDuration = intervalDuration.dividedBy(12);
        }

        Instant low = result.toInstant().plus(scheduleOffset);
        Instant high = result.toInstant().plus(randomizationDuration);

        return Random2.R.nextInstant(low, high);
    }

    private String getUserChangedDaysPath(LocalDate date) {
        return getUserChangedDaysRoot().child(date.toString()).toString();
    }

    private YPath getUserChangedDaysRoot() {
        return ytHomePath.child("user_changed_days");
    }

    private String getUserWithBlocksPath(LocalDate date) {
        return ytHomePath.child("user_with_blocks").child(date.toString()).toString();
    }

    public void buildCurrentCoolLentaSnapshot(int logsDays) {
        String logsDateFrom = LocalDate.now().minusDays(logsDays).toString();

        YPath currentSnapshot = ytHomePath.child("snapshots").child("current");
        YPath currentSnapshotNew = ytHomePath.child("snapshots").child("current_new");
        String currentSnapshotWithDate = ytHomePath.child("snapshots").child(LocalDate.now().minusDays(1).toString()).toString();

        String yql = buildCurrentCoolLentaSnapshotYql(logsDateFrom, currentSnapshot.toString(), Cf.list(currentSnapshotNew.toString(), currentSnapshotWithDate));
        yqlHelper.execute(yql);
        yt.cypress().move(currentSnapshotNew, currentSnapshot, true);
    }

    public void cleanOldTables(int daysAgo) {
        Cf.wrap(yt.cypress().get(ytHomePath).asMap()).filterValues(YTreeNode::isMapNode).keys().forEach(name -> {
            logger.info("Processing node {}", name);
            YPath folder = ytHomePath.child(name);
            yt.cypress().get(folder, Cf.set("creation_time")).asMap().forEach((k, v) -> {
                if (k.equals("current")) {
                    // never delete current table
                    return;
                }
                String createTimeStr = v.getAttribute("creation_time").get().stringValue();
                DateTime createTime = DateTime.parse(createTimeStr);

                logger.info("node = {}, create_time={}", folder.child(k), createTime);

                if (new Duration(createTime, DateTime.now()).getStandardDays() > daysAgo) {
                    YPath pathToRemove = folder.child(k);
                    logger.info("Going to remove node: {}", pathToRemove);
                    yt.cypress().remove(pathToRemove);
                }
            });
        });
    }

    @BenderBindAllFields
    @Data
    private static class UidAndHour {
        public final long uid;
        public final long hour;
    }
}
