package ru.yandex.chemodan.app.lentaloader.cool.worker;

import org.joda.time.DateTime;
import org.joda.time.Instant;
import org.junit.Test;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.chemodan.app.lentaloader.YtPathsUtils;
import ru.yandex.chemodan.app.lentaloader.cool.utils.IntervalType;
import ru.yandex.misc.test.Assert;

/**
 * @author tolmalev
 */
public class CoolLentaRoutinesTest {
    CoolLentaRoutines coolLentaRoutines = new CoolLentaRoutines(null, null, null, null, YtPathsUtils.getCoolLentaYPath(), 0);

    @Test
    public void timeToSchedule() {
        for (Integer i : Cf.range(0, 100000)) {
            checkTimeToScheduleOnce();
        }
    }

    @Test
    public void getYqlForOldIntervals() {

        Assert.equals("PRAGMA yt.DefaultOperationWeight=\"50.0\";\n" +
                "USE hahn;\n" +
                "\n" +
                "INSERT INTO `//tmp/tolmalev/ooo` WITH TRUNCATE\n" +
                "SELECT \n" +
                "   uid, \n" +
                "   430232 as hour \n" +
                "FROM RANGE(`//home/mpfs-stat/testing/cool-lenta/user_changed_days`, `2018-12-31`, `2019-03-01`)\n" +
                "WHERE hour >= 429536\n" +
                "AND hour <= 430928\n" +
                "GROUP BY uid;\n", coolLentaRoutines.getYqlForOldIntervals(
                DateTime.parse("2019-01-01T11:00:00.00+0300"),
                DateTime.parse("2019-02-28T11:00:00.00+0300"),
                "//tmp/tolmalev/ooo"));
    }

    @Test
    public void buildCurrentCoolLentaSnapshotYql() {
        Assert.equals(
                "PRAGMA yt.ForceInferSchema=\"1000\";\n" +
                        "PRAGMA yt.DefaultOperationWeight=\"50.0\";\n" +
                        "USE hahn;\n" +
                        "\n" +
                        "$input = (\n" +
                        "   SELECT * FROM RANGE(`//logs/ydisk-lenta-cool-events-log/1d`, `2019-03-01`)\n" +
                        "   UNION ALL SELECT * from `//home/mpfs-stat/testing/cool-lenta/snapshots/current`\n" +
                        ");\n" +
                        "\n" +
                        "$snapshot = (SELECT \n" +
                        "   uid,\n" +
                        "   block_collection,\n" +
                        "   block_id,\n" +
                        "   MAX(revision) as revision,\n" +
                        "   MAX_BY(block, revision) as block,\n" +
                        "   MAX_BY(event_type, revision) as event_type\n" +
                        "   FROM $input\n" +
                        "   WHERE revision IS NOT NULL\n" +
                        "   GROUP BY uid, block_collection, block_id\n" +
                        "   HAVING MAX_BY(event_type, revision) NOT IN ('all-blocks-delete', 'morda-blocks-delete')\n" +
                        ");\n" +
                        "\n" +
                        "INSERT INTO `//home/mpfs-stat/testing/cool-lenta/snapshots/current_new` WITH TRUNCATE SELECT * FROM $snapshot ORDER BY uid, block_collection, block_id; \n" +
                        "INSERT INTO `//home/mpfs-stat/testing/cool-lenta/snapshots/2018-03-05` WITH TRUNCATE SELECT * FROM $snapshot ORDER BY uid, block_collection, block_id; \n",
                coolLentaRoutines.buildCurrentCoolLentaSnapshotYql("2019-03-01", "//home/mpfs-stat/testing/cool-lenta/snapshots/current",
                        Cf.list("//home/mpfs-stat/testing/cool-lenta/snapshots/current_new", "//home/mpfs-stat/testing/cool-lenta/snapshots/2018-03-05"))
        );
    }

    public void checkTimeToScheduleOnce() {
        DateTime dtNow = DateTime.parse("2019-02-12T11:00:00.00+0300");
        Instant now = dtNow.toInstant();

        assertBetween(
                dtNow.minusHours(1),
                dtNow,
                CoolLentaRoutines.timeToSchedule(now, dtNow.minusDays(1), IntervalType.ONE_DAY)
        );
        assertBetween(
                dtNow.minusHours(1),
                dtNow,
                CoolLentaRoutines.timeToSchedule(now, dtNow.minusDays(7), IntervalType.WEEK)
        );
        assertBetween(
                dtNow.minusHours(1),
                dtNow,
                CoolLentaRoutines.timeToSchedule(now, dtNow.minusDays(70), IntervalType.MONTH)
        );
        assertBetween(
                dtNow.minusHours(1),
                dtNow,
                CoolLentaRoutines.timeToSchedule(now, dtNow.minusDays(150), IntervalType.SEASON)
        );

        assertBetween(
                DateTime.parse("2019-02-13T00:00:00.00+0300"),
                DateTime.parse("2019-02-13T01:00:00.00+0300"),
                CoolLentaRoutines.timeToSchedule(now, dtNow, IntervalType.ONE_DAY)
        );

        assertBetween(
                DateTime.parse("2019-02-18T00:00:00.00+0300"),
                DateTime.parse("2019-02-19T00:00:00.00+0300"),
                CoolLentaRoutines.timeToSchedule(now, dtNow, IntervalType.WEEK)
        );

        assertBetween(
                DateTime.parse("2019-02-18T00:00:00.00+0300"),
                DateTime.parse("2019-02-18T01:00:00.00+0300"),
                CoolLentaRoutines.timeToSchedule(now, DateTime.parse("2019-02-16T00:00:00.00+0300"), IntervalType.WEEKEND)
        );
        assertBetween(
                DateTime.parse("2019-02-18T00:00:00.00+0300"),
                DateTime.parse("2019-02-18T01:00:00.00+0300"),
                CoolLentaRoutines.timeToSchedule(now, DateTime.parse("2019-02-17T00:00:00.00+0300"), IntervalType.WEEKEND)
        );
        assertBetween(
                DateTime.parse("2019-02-18T00:00:00.00+0300"),
                DateTime.parse("2019-02-18T01:00:00.00+0300"),
                CoolLentaRoutines.timeToSchedule(now, DateTime.parse("2019-02-17T23:59:59.00+0300"), IntervalType.WEEKEND)
        );

        assertBetween(
                DateTime.parse("2019-03-01T00:00:00.00+0300"),
                DateTime.parse("2019-03-08T00:00:00.00+0300"),
                CoolLentaRoutines.timeToSchedule(now, dtNow, IntervalType.MONTH)
        );
        assertBetween(
                DateTime.parse("2019-03-03T00:00:00.00+0300"),
                DateTime.parse("2019-03-08T00:00:00.00+0300"),
                CoolLentaRoutines.timeToSchedule(now, dtNow, IntervalType.SEASON)
        );

        assertBetween(
                DateTime.parse("2020-01-06T00:00:00.00+0300"),
                DateTime.parse("2020-01-31T00:00:00.00+0300"),
                CoolLentaRoutines.timeToSchedule(now, dtNow, IntervalType.YEAR)
        );
    }

    private void assertBetween(DateTime from, DateTime to, Instant checkTime) {
        if (from.isEqual(checkTime) || to.isEqual(checkTime)) {
            return;
        }
        Assert.isTrue(checkTime.isAfter(from), checkTime + " must be after " + from);
        Assert.isTrue(checkTime.isBefore(to), checkTime + " must be before " + to);
    }
}
