package ru.yandex.chemodan.app.lentaloader.cool.worker;

import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;

import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.chemodan.app.lentaloader.cool.utils.IntervalType;
import ru.yandex.chemodan.app.lentaloader.reminder.DiskSearchClient;
import ru.yandex.chemodan.util.blackbox.UserTimezoneHelper;
import ru.yandex.commune.bazinga.BazingaTaskManager;
import ru.yandex.commune.bazinga.scheduler.OnetimeTask;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.random.Random2;
import ru.yandex.misc.thread.ThreadUtils;

/**
 * @author tolmalev
 */
public class CoolLentaTasksScheduler {
    private final BazingaTaskManager bazingaTaskManager;
    private final DiskSearchClient diskSearchClient;
    private final UserTimezoneHelper userTimezoneHelper;

    private final DynamicProperty<ListF<Integer>> sleepDelays = new DynamicProperty<>("CoolLentaTasksScheduler-retry-delays", Cf.list(1, 2, 3, 5, 10));

    public CoolLentaTasksScheduler(BazingaTaskManager bazingaTaskManager, DiskSearchClient diskSearchClient,
            UserTimezoneHelper userTimezoneHelper)
    {
        this.bazingaTaskManager = bazingaTaskManager;
        this.diskSearchClient = diskSearchClient;
        this.userTimezoneHelper = userTimezoneHelper;
    }

    public void scheduleTasksForUser(PassportUid uid) {
        scheduleTasksForUser(uid, 0, Option.empty(), Option.empty());
    }

    public List<Future> scheduleTasksForUser(PassportUid uid, ExecutorService executor) {
        return scheduleTasksForUser(uid, 0, Option.of(executor), Option.empty());
    }

    public void scheduleTasksForUserHighPriority(PassportUid uid) {
        scheduleTasksForUser(uid, 1, Option.empty(), Option.empty());
    }

    public List<Future> scheduleTasksForUserLowPriority(PassportUid uid, ExecutorService executor, Duration schedulingInterval) {
        return scheduleTasksForUser(uid, -1, Option.of(executor), Option.of(schedulingInterval));
    }

    public void scheduleTasksForUserLowPriority(PassportUid uid) {
        scheduleTasksForUser(uid, -1, Option.empty(), Option.empty());
    }

    public List<Future> scheduleTasksForUser(PassportUid uid, int priority, Option<ExecutorService> executor, Option<Duration> schedulingInterval) {
        DateTimeZone userTimezone = userTimezoneHelper.getUserTimezone(uid);
        SetF<ProcessOneIntervalTask.Parameters> tasks = Cf.hashSet();
        for (Tuple2<Integer, Integer> t2 : diskSearchClient.getCountPhotosByDay(uid).entries()) {
            if (t2._1 == 16417 && t2._2 == 6) {
                // с очень большой вероятностью это дефолтные файлы
                continue;
            }

            int day = t2._1;
            for (int i = -1; i <= 1; i++) {
                if (day + i >= 0) {
                    Instant time = new Instant(0).plus(Duration.standardHours(12)).plus(Duration.standardDays(day + i));
                    tasks.addAll(tasksForTime(uid, time.toDateTime(userTimezone)));
                }
            }
        }

        return scheduleAll(tasks.toList(), priority, executor, schedulingInterval);
    }

    private ListF<ProcessOneIntervalTask.Parameters> tasksForTime(PassportUid uid, DateTime userTime) {
        return IntervalType.getTypesForDateTime(userTime).map(type -> new ProcessOneIntervalTask.Parameters(
                uid, type.getIntervalStart(userTime).toInstant(), type
        ));
    }

    private List<Future> scheduleAll(ListF<ProcessOneIntervalTask.Parameters> parameters, int priority, Option<ExecutorService> executor,
                                     Option<Duration> schedulingInterval)
    {
        if (priority > 0) {
            return parameters.map(ProcessOneIntervalHighPriorityTask::new).map(task -> this.schedule(task, executor, schedulingInterval));
        } else if (priority < 0) {
            return parameters.map(ProcessOneIntervalLowPriorityTask::new).map(task -> this.schedule(task, executor, schedulingInterval));
        } else {
            return parameters.map(ProcessOneIntervalTask::new).map(task -> this.schedule(task, executor, schedulingInterval));
        }
    }

    private Future schedule(OnetimeTask task, Option<ExecutorService> executor, Option<Duration> schedulingInterval) {
        if (executor.isPresent()) {
            return executor.get().submit(() -> scheduleImpl(task, schedulingInterval));
        } else {
            scheduleImpl(task, schedulingInterval);
            return CompletableFuture.completedFuture(null);
        }
    }

    private void scheduleImpl(OnetimeTask task, Option<Duration> schedulingInterval) {
        List<Integer> delays = sleepDelays.get();
        RuntimeException lastException = null;
        for(int i = 0; i < 5; i++) {
            try {
                Instant time = Random2.R.nextInstant(Instant.now(),
                        Instant.now().plus(schedulingInterval.getOrElse(Duration.standardSeconds(1))));
                bazingaTaskManager.schedule(task, time);
                return;
            } catch (RuntimeException e) {
                lastException = e;
                Duration delay = Duration.standardSeconds(delays.get(Math.min(i, delays.size() - 1)));
                ThreadUtils.sleep(delay);
            }
        }
        throw lastException;
    }
}
