package ru.yandex.chemodan.app.lentaloader.cool.worker;

import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;

import lombok.Data;
import org.joda.time.Duration;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.function.Function1V;
import ru.yandex.chemodan.app.lentaloader.worker.LentaTaskQueueName;
import ru.yandex.chemodan.util.yt.YtHelper;
import ru.yandex.commune.bazinga.impl.JobStatus;
import ru.yandex.commune.bazinga.impl.TaskId;
import ru.yandex.commune.bazinga.pg.storage.PgBazingaStorage;
import ru.yandex.commune.bazinga.scheduler.CronTask;
import ru.yandex.commune.bazinga.scheduler.ExecutionContext;
import ru.yandex.commune.bazinga.scheduler.TaskQueueName;
import ru.yandex.commune.bazinga.scheduler.schedule.Schedule;
import ru.yandex.commune.bazinga.scheduler.schedule.SchedulePeriodic;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.tables.YTableEntryTypes;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.thread.ThreadUtils;
import ru.yandex.misc.thread.factory.ThreadNameIndexThreadFactory;

/**
 * @author tolmalev
 */
public class ScheduleUserBlocksGenerationCron extends CronTask {
    private static final Logger logger = LoggerFactory.getLogger(ScheduleUserBlocksGenerationCron.class);

    private static final String ATTRIBUTE_NAME = "processed_count";

    private final CoolLentaTasksScheduler tasksScheduler;
    private final PgBazingaStorage bazingaStorage;

    private final YtHelper yt;
    private final YPath tableWithUidsQueue;
    private final YPath processedCountAttributePath;

    private final ExecutorService smallExecutor;
    private final ExecutorService schedulerExecutor;

    private final DynamicProperty<Integer> bulkSize = new DynamicProperty<>("cool-lenta-reindex-scheduler-bulk-size", 1000);
    private final DynamicProperty<Integer> maxTasksCount = new DynamicProperty<>("cool-lenta-reindex-scheduler-max-tasks-count", 5000000);
    private final DynamicProperty<Integer> delayForScheduling = new DynamicProperty<>("cool-lenta-reindex-scheduler-delay-s", 300);

    public ScheduleUserBlocksGenerationCron(CoolLentaTasksScheduler tasksScheduler, YPath tableWithUidsQueue, YtHelper yt,
                                            PgBazingaStorage bazingaStorage, int threads)
    {
        this.yt = yt;
        this.tasksScheduler = tasksScheduler;
        this.tableWithUidsQueue = tableWithUidsQueue;
        this.bazingaStorage = bazingaStorage;
        this.smallExecutor = Executors.newFixedThreadPool(10, new ThreadNameIndexThreadFactory("ScheduleUserBlocksGenerationCron-small-pool-"));
        this.schedulerExecutor = Executors.newFixedThreadPool(threads, new ThreadNameIndexThreadFactory("ScheduleUserBlocksGenerationCron-pool-"));
        this.processedCountAttributePath = tableWithUidsQueue.attribute(ATTRIBUTE_NAME);
    }

    @Override
    public Schedule cronExpression() {
        return new SchedulePeriodic(Duration.standardMinutes(1));
    }

    @Override
    public Duration timeout() {
        return Duration.standardHours(1);
    }

    @Override
    public TaskQueueName queueName() {
        return LentaTaskQueueName.LENTA_CRON;
    }

    @Override
    public void execute(ExecutionContext executionContext) throws Exception {
        if (!yt.existsWithRetries(tableWithUidsQueue)) {
            logger.info("No table with users to reindex");
            return;
        }

        int currentQueueSize = bazingaStorage.findOnetimeJobCounters(
                new TaskId("processOneIntervalLowPriority"),
                Cf.list(JobStatus.READY, JobStatus.STARTING, JobStatus.RUNNING)
        );

        if (currentQueueSize > maxTasksCount.get()) {
            logger.info("Queue is over limit: {} > {}", currentQueueSize, maxTasksCount.get());
            return;
        }
        logger.info("Queue is lower limit: {} <= {}. Going to schedule {} users", currentQueueSize, maxTasksCount.get(), bulkSize.get());

        long totalUids = yt.getRowCount(tableWithUidsQueue);
        long processedUids = yt.getWithRetries(() -> {
            if (!yt.existsWithRetries(processedCountAttributePath)) {
                return 0L;
            }
            return yt.cypress().get(processedCountAttributePath).longValue();
        });
        if (processedUids >= totalUids) {
            logger.info("All users are already reindexed");
            return;
        }

        // сохраняем текущее значение просто для проверки, что все ок с записью
        saveProcessedCount(processedUids);

        long newProcessedUids = Math.min(totalUids, processedUids + bulkSize.get());
        YPath pathWithRange = tableWithUidsQueue.withRange(processedUids, newProcessedUids);

        Tuple2List<Long, List<Future>> allFutures = Tuple2List.arrayList();
        ListF<Future> firstFutures = Cf.arrayList();

        Function1V<UidToSchedule> submitScheduling = uid -> {
            Future<?> future = smallExecutor.submit(() -> {
                for (int i = 0; i < 5; i++) {
                    try {
                        List<Future> futuresForUser = tasksScheduler.scheduleTasksForUserLowPriority(PassportUid.cons(uid.uid),
                                schedulerExecutor, Duration.standardSeconds(delayForScheduling.get()));
                        allFutures.add(Tuple2.tuple(uid.uid, futuresForUser));
                        logger.debug("Submitted tasks for {}", uid);
                        return;
                    } catch (Exception e) {
                        int delaySeconds = 1 + i * 2;
                        logger.debug("Failed to submit for user {}, try: {}, delay: {}", uid, delaySeconds);
                        ThreadUtils.sleep(Duration.standardSeconds(delaySeconds));
                    }
                }
                logger.error("Complete failed to uid: {}", uid);
            });
            firstFutures.add(future);
        };

        logger.info("Going to reindex users from rows {}-{}", processedUids, newProcessedUids);
        yt.runWithRetries(() -> yt.tables().read(pathWithRange, YTableEntryTypes.bender(UidToSchedule.class), submitScheduling));

        for (Future firstFuture : firstFutures) {
            try {
                firstFuture.get();
            } catch (RuntimeException e) {
                logger.error("Failed to submit for user: {}", e);
            }
        }

        int ok = 0;
        int failed = 0;
        int uids = 0;
        for (Tuple2<Long, List<Future>> tuple2 : allFutures) {
            List<Future> realSubmitFutures = tuple2._2;
            for (int i = 0; i < realSubmitFutures.size(); i++) {
                try {
                    realSubmitFutures.get(i).get();
                    ok++;
                } catch (Exception e) {
                    failed++;
                }

                if (ok + failed > 0 && (ok + failed) % 1000 == 0) {
                    logger.info("Processed uids: {}, total_tasks: {}, ok: {}, failed: {}", uids, (ok + failed), ok, failed);
                }
            }
            uids++;
        }

        saveProcessedCount(newProcessedUids);
    }

    private void saveProcessedCount(long processedUids) {
        yt.runWithRetries(() -> yt.cypress().set(processedCountAttributePath, processedUids));
    }

    @Data
    @BenderBindAllFields
    private static class UidToSchedule {
        public final long uid;
    }
}
