package ru.yandex.chemodan.app.lentaloader.lenta;

import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.function.Function0;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataField;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataRecord;
import ru.yandex.chemodan.app.dataapi.api.deltas.FieldChange;
import ru.yandex.chemodan.app.dataapi.support.DataFieldsSource;
import ru.yandex.chemodan.app.dataapi.support.RecordField;
import ru.yandex.chemodan.app.dataapi.support.RecordFieldUtils;
import ru.yandex.misc.lang.DefaultObject;
import ru.yandex.misc.lang.Validate;

/**
 * @author dbrylev
 */
public class LentaBlockRecord extends DefaultObject implements DataFieldsSource {
    public final String id;
    public final long order;

    public final LentaRecordType type;
    public final String groupKey;

    public final Option<Instant> mFrom;
    public final Instant mTime;
    public final Option<Instant> mTill;

    public final MapF<String, DataField> specific;

    public LentaBlockRecord(
            String id, long order,
            LentaRecordType type, String groupKey,
            Option<Instant> mFrom, Instant mTime, Option<Instant> mTill,
            MapF<String, DataField> specific)
    {
        Validate.notEquals(LentaRecordType.NEXT_INDEX, type);

        this.id = id;
        this.order = order;
        this.mFrom = mFrom;
        this.mTime = mTime;
        this.mTill = mTill;
        this.type = type;
        this.groupKey = groupKey;
        this.specific = specific.filterKeys(Fields.NAMES.containsF().notF());
    }

    public static LentaBlockRecord fromDataRecord(DataRecord rec) {
        return new LentaBlockRecord(
                rec.getRecordId(),
                Fields.ORDER.get(rec), Fields.TYPE.get(rec), Fields.GROUP_KEY.get(rec),
                Fields.MFROM.getO(rec), Fields.MTIME.get(rec), Fields.MTILL.getO(rec), rec.getData());
    }

    private final transient Function0<MapF<String, DataField>> toData = Function0.wrap(this::calculateData).memoize();

    private MapF<String, DataField> calculateData() {
        ListF<Tuple2<String, DataField>> data = Cf.list(
                Fields.ORDER.toData(order), Fields.TYPE.toData(type),
                Fields.GROUP_KEY.toData(groupKey), Fields.MTIME.toData(mTime));

        data = data
                .plus(mFrom.map(Fields.MFROM::toData))
                .plus(mTill.map(Fields.MTILL::toData));

        return Cf.toMap(data).plus(specific);
    }

    @Override
    public MapF<String, DataField> toData() {
        return toData.apply();
    }

    public ListF<FieldChange> diffFrom(MapF<String, DataField> previous) {
        return RecordFieldUtils.diff(previous, toData());
    }

    public LentaBlockRecord withSpecific(MapF<String, DataField> specific) {
        return new LentaBlockRecord(id, order, type, groupKey, mFrom, mTime, mTill, specific);
    }

    public LentaBlockIdAndType getIdAndType() {
        return new LentaBlockIdAndType(id, type);
    }

    public interface Fields {
        RecordField<Long> ORDER = RecordField.longNumber("order");
        RecordField<LentaRecordType> TYPE = RecordField.stringEnum("type", LentaRecordType.class);
        RecordField<String> GROUP_KEY = RecordField.string("group_key");
        RecordField<Instant> MFROM = RecordField.instant("mfrom");
        RecordField<Instant> MTIME = RecordField.instant("mtime");
        RecordField<Instant> MTILL = RecordField.instant("mtill");

        SetF<String> NAMES = RecordFieldUtils.fieldNames(Fields.class);
    }
}
