package ru.yandex.chemodan.app.lentaloader.lenta;

import java.util.function.Supplier;

import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Either;
import ru.yandex.bolts.collection.IteratorF;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.function.Function2V;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataField;
import ru.yandex.chemodan.app.dataapi.api.data.filter.RecordsFilter;
import ru.yandex.chemodan.app.dataapi.api.data.filter.condition.CollectionIdCondition;
import ru.yandex.chemodan.app.dataapi.api.data.filter.condition.DataCondition;
import ru.yandex.chemodan.app.dataapi.api.data.filter.condition.RecordIdCondition;
import ru.yandex.chemodan.app.dataapi.api.data.filter.ordering.ByIdRecordOrder;
import ru.yandex.chemodan.app.dataapi.api.data.filter.ordering.ByRevisionOrder;
import ru.yandex.chemodan.app.dataapi.api.data.filter.ordering.OrderedUUID;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataRecord;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataRecordId;
import ru.yandex.chemodan.app.dataapi.api.data.record.SimpleRecordId;
import ru.yandex.chemodan.app.dataapi.api.db.Database;
import ru.yandex.chemodan.app.dataapi.api.db.handle.DatabaseHandle;
import ru.yandex.chemodan.app.dataapi.api.db.ref.AppDatabaseRef;
import ru.yandex.chemodan.app.dataapi.api.db.ref.UserDatabaseSpec;
import ru.yandex.chemodan.app.dataapi.api.deltas.Delta;
import ru.yandex.chemodan.app.dataapi.api.deltas.FieldChange;
import ru.yandex.chemodan.app.dataapi.api.deltas.RecordChange;
import ru.yandex.chemodan.app.dataapi.api.deltas.RecordChangeType;
import ru.yandex.chemodan.app.dataapi.api.deltas.RevisionCheckMode;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.dataapi.core.manager.DataApiManager;
import ru.yandex.chemodan.app.lentaloader.DynamicVars;
import ru.yandex.chemodan.app.lentaloader.lenta.limit.FolderBlockData;
import ru.yandex.chemodan.app.lentaloader.lenta.update.DeleteHandler;
import ru.yandex.chemodan.app.lentaloader.lenta.update.LentaBlockBaseData;
import ru.yandex.chemodan.app.lentaloader.lenta.update.LentaBlockCreateData;
import ru.yandex.chemodan.app.lentaloader.lenta.update.LentaBlockModifyData;
import ru.yandex.chemodan.app.lentaloader.lenta.update.LentaBlockUpdateData;
import ru.yandex.chemodan.app.lentaloader.lenta.update.UpdateHandler;
import ru.yandex.chemodan.app.lentaloader.lenta.update.UpdateOrDeleteHandler;
import ru.yandex.chemodan.app.lentaloader.log.ActionInfo;
import ru.yandex.chemodan.app.lentaloader.log.ActionReason;
import ru.yandex.chemodan.app.lentaloader.log.DataOrRefusal;
import ru.yandex.chemodan.app.lentaloader.log.LentaBlockEvent;
import ru.yandex.chemodan.app.lentaloader.log.ReasonedAction;
import ru.yandex.chemodan.app.lentaloader.worker.tasks.DeleteEmptyLentaBlockTask;
import ru.yandex.chemodan.app.lentaloader.worker.tasks.DeleteOrUpdateLentaBlockTask;
import ru.yandex.chemodan.app.lentaloader.worker.tasks.FindAndDeleteOrUpdateLentaBlocksTask;
import ru.yandex.chemodan.app.lentaloader.worker.tasks.HolderForUpdateAndUpOrDeleteTask;
import ru.yandex.chemodan.app.lentaloader.worker.tasks.UnpinLentaBlockTask;
import ru.yandex.chemodan.app.lentaloader.worker.tasks.UpdateAndUpOrDeleteLentaBlockTask;
import ru.yandex.chemodan.app.uaas.experiments.ExperimentsManager;
import ru.yandex.chemodan.bazinga.PgOnetimeUtils;
import ru.yandex.chemodan.cache.MeteredCache;
import ru.yandex.commune.bazinga.impl.OnetimeJob;
import ru.yandex.commune.bazinga.impl.OnetimeUtils;
import ru.yandex.commune.bazinga.pg.storage.JobSaveResult;
import ru.yandex.commune.bazinga.pg.storage.PgBazingaStorage;
import ru.yandex.commune.bazinga.scheduler.ActiveUidDuplicateBehavior;
import ru.yandex.commune.bazinga.scheduler.OnetimeTaskSupport;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.db.q.SqlLimits;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.time.InstantInterval;

/**
 * @author dbrylev
 */
public class LentaManagerImpl implements LentaManager {
    private static final Logger logger = LoggerFactory.getLogger(LentaManagerImpl.class);

    public static final String DISABLE_BLOCK_TYPE_PREFIX = "disk_lenta_disable_block_";
    public static final ListF<String> HARDCODED_DISABLE_BLOCK_TYPES =
            Cf.list("public_resource_owned", "block_public_resource");


    public static final AppDatabaseRef DB_REF = new AppDatabaseRef("lenta", "lenta_blocks");

    static final Duration BLOCK_TTL = Duration.standardDays(1);
    static final Duration BLOCK_MFROM_BEHIND = Duration.standardMinutes(5);

    private static int BLOCKS_COUNT_LIMIT = 100;

    private final DataApiManager dataApiManager;
    private final PgBazingaStorage bazingaStorage;
    private final ExperimentsManager experimentsManager;

    private final MeteredCache<UserBlockGroupKeyAndType, LentaBlockRecord> blocksCache;
    private final MeteredCache<DataApiUserId, DatabaseHandle> dbHandleCache;

    public LentaManagerImpl(
            DataApiManager dataApiManager,
            PgBazingaStorage bazingaStorage,
            ExperimentsManager experimentsManager,
            MeteredCache<UserBlockGroupKeyAndType, LentaBlockRecord> blocksCache,
            MeteredCache<DataApiUserId, DatabaseHandle> dbHandleCache)
    {
        this.dataApiManager = dataApiManager;
        this.bazingaStorage = bazingaStorage;
        this.experimentsManager = experimentsManager;
        this.blocksCache = blocksCache;
        this.dbHandleCache = dbHandleCache;
    }

    @Override
    public Option<LentaBlockRecord> findBlock(DataApiUserId uid, String blockId, ActionInfo actionInfo) {
        Validate.notEquals(LentaNextIndexRecord.ID, blockId);

        return findBlock(dbSupplier(uid), blockId, LookupCollections.ANY);
    }

    @Override
    public ListF<LentaBlockRecord> findBlocks(
            DataApiUserId uid, LentaRecordType type, FieldPredicate fieldFilter, ActionInfo actionInfo)
    {
        return findAnyBlockRecords(dbSupplier(uid), type, fieldFilter).map(LentaBlockRecord::fromDataRecord);
    }

    @Override
    public Option<LentaBlockRecord> findMostRecentBlock(
            DataApiUserId uid, LentaRecordType type, FieldPredicate filter, ActionInfo actionInfo)
    {
        ListF<DataRecord> records = findLastBlockRecords(
                dbSupplier(uid), type, filter, Option.empty(), Option.empty(),
                LookupCollections.INDEX_AND_PINNED, Option.of(1));

        return records.map(LentaBlockRecord::fromDataRecord).singleO();
    }

    @Override
    public FindOrCreateResult findOrCreateBlock(DataApiUserId uid, LentaBlockCreateData data, ActionInfo actionInfo) {
        DatabaseSupplier dbs = dbSupplier(uid).withNowaitLock(DynamicVars.nowaitLockForBlockInsert.get());

        return findOrCreateBlock(dbs, data, false, actionInfo);
    }

    @Override
    public boolean findAndUpdateBlock(DataApiUserId uid, LentaBlockUpdateData data, ActionInfo actionInfo) {
        return findAndUpdateBlock(uid, data, false, actionInfo);
    }

    @Override
    public boolean findCachedAndUpdateBlock(DataApiUserId uid, LentaBlockUpdateData data, ActionInfo actionInfo) {
        return findAndUpdateBlock(uid, data, true, actionInfo);
    }

    private boolean findAndUpdateBlock(
            DataApiUserId uid, LentaBlockUpdateData data, boolean cached, ActionInfo actionInfo)
    {
        DatabaseSupplier dbs = dbSupplier(uid);

        Option<CachedOrLoadedBlock> foundBlock = findBlockToUpdate(dbs, data.base, cached, actionInfo);

        if (foundBlock.isPresent()) {
            dbs = dbs.withNowaitLock(DynamicVars.nowaitLockForBlockModify.get());

            return updateBlock(dbs, foundBlock.get(), data, actionInfo);

        } else {
            return false;
        }
    }

    @Override
    public FindOrCreateResult findAndUpdateOrCreateBlock(
            DataApiUserId uid, LentaBlockModifyData data, ActionInfo actionInfo)
    {
        return findAndUpdateOrCreateBlock(uid, data, false, actionInfo);
    }

    @Override
    public FindOrCreateResult findCachedAndUpdateOrCreateBlock(
            DataApiUserId uid, LentaBlockModifyData data, ActionInfo actionInfo)
    {
        return findAndUpdateOrCreateBlock(uid, data, true, actionInfo);
    }

    private FindOrCreateResult findAndUpdateOrCreateBlock(
            DataApiUserId uid, LentaBlockModifyData data, boolean cached, ActionInfo actionInfo)
    {
        DatabaseSupplier dbs = dbSupplier(uid).withNowaitLock(DynamicVars.nowaitLockForBlockInsert.get());

        if (isBlockTypeDenied(dbs.uid, data.base.type)) {
            return FindOrCreateResult.ignored(
                    new LentaBlockRecord(
                            dbs.uid + "_" + data.base.type,
                            0,
                            data.base.type,
                            data.base.groupKey,
                            Option.empty(),
                            Instant.now(),
                            Option.empty(),
                            Cf.map()
                    ),
                    ActionReason.IGNORED_BY_EXPERIMENT
            );
        }

        FindOrCreateResult result = findOrCreateBlock(dbs, data.getCreateData(), cached, actionInfo);

        if (result.isFound()) {
            dbs = dbs.withNowaitLock(DynamicVars.nowaitLockForBlockModify.get());

            return updateBlock(dbs, result.foundBlock.get(), data.getUpdateData(), actionInfo)
                    ? FindOrCreateResult.found(result.foundBlock.get())
                    : FindOrCreateResult.ignored(result.record, ActionReason.UNSPECIFIED);

        } else {
            return result;
        }
    }

    @Override
    public Option<String> findAndPinOrCreatePinnedBlock(
            DataApiUserId uid, LentaBlockCreateData data, ActionInfo actionInfo)
    {
        FindOrCreateResult result = findAndPinOrCreatePinnedBlock(dbSupplier(uid), data, actionInfo);

        return Option.when(!result.isIgnored(), result.record.id);
    }

    private FindOrCreateResult findOrCreateBlock(
            DatabaseSupplier dbs, LentaBlockCreateData data, boolean cached, ActionInfo actionInfo)
    {
        FindOrCreateResult result = findOrCreateBlockData(dbs, data, cached, actionInfo);

        if (result.isCreated()) {
            Database db = dbs.db();

            Tuple2<ListF<RecordChange>, ListF<DataRecord>> cleanupChanges = Tuple2.tuple(Cf.list(), Cf.list());
            DynamicVars.CleanupBlocksConfiguration conf = DynamicVars.cleanupBlocksInplaceConf.get();

            if (db.meta.recordsCount > conf.blocksSearchLimit) {
                cleanupChanges = deleteEldestBlocksChanges(dbs, (int) (db.meta.recordsCount - conf.blocksEnough));
            }

            ListF<RecordChange> changes = Cf.<RecordChange>list()
                    .plus1(RecordChange.insert("index", result.record.id, result.record.toData()))
                    .plus(siftIndexLastBlockIfExcessChanges(dbs))
                    .plus(cleanupChanges.get1());

            dataApiManager.applyDelta(db, RevisionCheckMode.WHOLE_DATABASE, new Delta(changes));

            blocksCache.put(new UserBlockGroupKeyAndType(db.uid, result.record), result.record);

            LentaBlockEvent.created(db.uid, result.record, actionInfo).log();

            cleanupChanges.get2().map(rec -> LentaBlockEvent.deleted(db.uid, rec,
                    actionInfo.withReason(ActionReason.LENTA_SIZE_LIMIT_EXCEEDED))).forEach(LentaBlockEvent::log);
        }
        if (result.isIgnored()) {
            LentaBlockEvent.createIgnored(dbs.uid, result.record, result.ignoreReason.get(), actionInfo).log();
        }

        return result;
    }

    private FindOrCreateResult findAndPinOrCreatePinnedBlock(
            DatabaseSupplier db, LentaBlockCreateData data, ActionInfo actionInfo)
    {
        FindOrCreateResult result = findOrCreateBlockData(db, data, false, actionInfo);

        ListF<RecordChange> changes = Cf.list();

        Option<LentaBlockEvent> event = Option.empty();

        if (result.isCreated()) {
            db = db.withNowaitLock(DynamicVars.nowaitLockForBlockInsert.get());
            changes = changes.plus1(RecordChange.insert("pinned", result.record.id, result.record.toData()));

            event = Option.of(LentaBlockEvent.createdPin(db.uid, result.record, actionInfo));

        } else if (result.isFound() && !result.foundBlock.get().asLoaded().getCollectionId().equals("pinned")) {
            db = db.withNowaitLock(DynamicVars.nowaitLockForBlockModify.get());

            DataRecord record = result.foundBlock.get().asLoaded();

            DataOrRefusal<MapF<String, DataField>> specific = data.handler.getAction(result.record).specific;

            LentaBlockRecord updated = new LentaBlockRecord(
                    record.getRecordId(), db.db().rev + 1, data.base.type, data.base.groupKey,
                    result.record.mFrom, actionInfo.now, result.record.mTill,
                    specific.getOrElse(result.record.specific));

            changes = changes
                    .plus1(RecordChange.insert("pinned", updated.id, updated.toData()))
                    .plus(deleteBlocksAndOrphanedCollectionsChanges(db, Cf.list(record.id)));

            event = Option.of(LentaBlockEvent.pin(
                    db.uid, result.record, result.foundBlock.get().asLoaded().getCollectionId(), actionInfo));
        }

        if (result.isIgnored()) {
            event = Option.of(LentaBlockEvent.createPinIgnored(
                    db.uid, result.record, result.ignoreReason.get(), actionInfo));
        }

        if (changes.isNotEmpty()) {
            unpinBlockDelayed(db.uid, result.record.getIdAndType(), actionInfo);

            dataApiManager.applyDelta(db.db(), RevisionCheckMode.WHOLE_DATABASE, new Delta(changes));

            blocksCache.put(new UserBlockGroupKeyAndType(db.uid, result.record), result.record);
        }
        event.forEach(LentaBlockEvent::log);

        return result;
    }

    private FindOrCreateResult findOrCreateBlockData(
            DatabaseSupplier dbs, LentaBlockCreateData data, boolean cached, ActionInfo actionInfo)
    {
        if (isBlockTypeDenied(dbs.uid, data.base.type)) {
            return FindOrCreateResult.ignored(
                    new LentaBlockRecord(
                            dbs.uid + "_" + data.base.type,
                            0,
                            data.base.type,
                            data.base.groupKey,
                            Option.empty(),
                            Instant.now(),
                            Option.empty(),
                            Cf.map()
                    ),
                    ActionReason.IGNORED_BY_EXPERIMENT
            );
        }
        Either<CachedOrLoadedBlock, Option<InstantInterval>> blockOrRange = findBlockOrFreeTimeRange(
                dbs, data.base, cached, true, actionInfo);

        if (blockOrRange.isLeft()) {
            return FindOrCreateResult.found(blockOrRange.getLeft());
        }
        Option<InstantInterval> range = blockOrRange.getRight();

        long order = dbs.db().rev + 1;

        LentaBlockRecord record = new LentaBlockRecord(
                OrderedUUID.generateOrderedUUID(actionInfo.now), order, data.base.type, data.base.groupKey,
                range.map(InstantInterval::getStart), actionInfo.now, range.map(InstantInterval::getEnd), Cf.map());

        DataOrRefusal<MapF<String, DataField>> specific = data.handler.getAction(record).specific;

        return !specific.isRefusal()
                ? FindOrCreateResult.created(record.withSpecific(specific.getData()))
                : FindOrCreateResult.ignored(record, specific.getRefusalReason());
    }

    private boolean isBlockTypeDenied(DataApiUserId uid, LentaRecordType type) {
        String blockTypeStr = type.value();

        if (HARDCODED_DISABLE_BLOCK_TYPES.containsTs(blockTypeStr)) {
            return true;
        }
        if (!DynamicVars.disableBlockTypes.get().containsTs(blockTypeStr)) {
            return false;
        }
        if(DynamicVars.disableBlockUids.get().containsTs(uid.toPassportUidOrZero().getUid())) {
            return DynamicVars.disableBlockTypes.get().containsTs(blockTypeStr);
        }

        return experimentsManager.getFlags(uid.toPassportUidOrZero().getUid())
                .containsTs(DISABLE_BLOCK_TYPE_PREFIX + blockTypeStr);
    }

    private Option<CachedOrLoadedBlock> findBlockToUpdate(
            DatabaseSupplier dbs, LentaBlockBaseData data, boolean cached, ActionInfo actionInfo)
    {
        return findBlockOrFreeTimeRange(dbs, data, cached, false, actionInfo).leftO();
    }

    private Either<CachedOrLoadedBlock, Option<InstantInterval>> findBlockOrFreeTimeRange(
            DatabaseSupplier dbs, LentaBlockBaseData data, boolean cached, boolean forCreate, ActionInfo actionInfo)
    {
        Option<CachedOrLoadedBlock> found;
        Option<InstantInterval> range;

        if (data.type.isTimeLimited()) {
            Either<CachedOrLoadedBlock, Option<InstantInterval>> blockOrRange =
                    findTimeLimitedBlockOrFreeTimeRange(dbs, data, cached, forCreate, actionInfo);

            found = blockOrRange.leftO();
            range = blockOrRange.rightO().flatMapO(r -> r);

        } else {
            found = Option.empty();
            range = Option.empty();

            if (cached) {
                found = blocksCache.getO(new UserBlockGroupKeyAndType(dbs.uid, data))
                        .map(CachedOrLoadedBlock::cached);
            }
            if (!found.isPresent()) {
                if (forCreate) {
                    dbs.obtain();
                }
                found = findLastBlockRecord(dbs, data.type, data.groupKey, LookupCollections.ANY)
                        .map(CachedOrLoadedBlock::loaded);
            }
        }

        if (found.isPresent()) {
            if (!found.get().isCached()) {
                blocksCache.put(new UserBlockGroupKeyAndType(dbs.uid, found.get()), found.get().toBlockRecord());
            } else  {
                blocksCache.incHits();
            }
            return Either.left(found.get());
        }
        return Either.right(range);
    }

    private Either<CachedOrLoadedBlock, Option<InstantInterval>> findTimeLimitedBlockOrFreeTimeRange(
            DatabaseSupplier dbs, LentaBlockBaseData data, boolean cached, boolean forCreate, ActionInfo actionInfo)
    {
        Instant timestamp = actionInfo.getActionTime().getOrElse(actionInfo.now);

        if (cached) {
            Option<LentaBlockRecord> found = blocksCache.getO(new UserBlockGroupKeyAndType(dbs.uid, data));

            if (found.isPresent()) {
                if (!found.get().mFrom.exists(timestamp::isBefore) && found.get().mTill.exists(timestamp::isBefore)) {
                    return Either.left(CachedOrLoadedBlock.cached(found.get()));
                }
            }
        }
        if (forCreate) {
            dbs.obtain();
        }

        Instant searchSince = timestamp.minus(BLOCK_TTL).minus(BLOCK_MFROM_BEHIND);
        Instant searchTill = timestamp.plus(BLOCK_TTL);

        ListF<DataRecord> records = findLastBlockRecords(dbs, data.type, GroupKeyPredicate.eq(data.groupKey),
                Option.of(searchSince), Option.of(searchTill), LookupCollections.ANY, Option.empty());

        Option<DataRecord> found = records.find(rec ->
                !LentaBlockRecord.Fields.MFROM.get(rec).isAfter(timestamp)
                        && timestamp.isBefore(LentaBlockRecord.Fields.MTILL.get(rec)));

        if (found.isPresent()) {
            return Either.left(CachedOrLoadedBlock.loaded(found.get()));
        }
        if (!forCreate) {
            return Either.right(Option.empty());
        }

        ListF<LentaBlockRecord> blocks = records.map(LentaBlockRecord::fromDataRecord);

        Option<LentaBlockRecord> preceding = blocks.takeWhile(r -> !r.mFrom.get().isAfter(timestamp)).lastO();
        Option<LentaBlockRecord> following = blocks.dropWhile(r -> !r.mTill.get().isAfter(timestamp)).firstO();

        Instant mFrom = preceding.map(r -> r.mTill.get()).plus(timestamp.minus(BLOCK_MFROM_BEHIND)).max();
        Instant mTill = following.map(r -> r.mFrom.get()).plus(mFrom.plus(BLOCK_TTL)).min();

        long shiftMs = new Duration(mFrom.plus(BLOCK_TTL), mTill).getMillis();

        if (!preceding.isPresent() && shiftMs < 0) {
            preceding = findLastBlock(dbs, data.type, data.groupKey,
                    Option.of(searchSince.plus(shiftMs)), Option.of(searchSince), LookupCollections.ANY);
        }
        mFrom = preceding.map(r -> r.mTill.get()).plus(mTill.minus(BLOCK_TTL)).max();

        return Either.right(Option.of(new InstantInterval(mFrom, mTill)));
    }

    @Override
    public void createAndDeleteBlocks(
            DataApiUserId uid, ListF<FolderBlockData> datas, ListF<String> deleteIds, ReasonedAction actionInfo)
    {
        if (datas.isEmpty() && deleteIds.isEmpty()) return;

        if (DynamicVars.exclusiveLockForBlockMerge.get()) {
            dataApiManager.runWithLockedDatabase(new UserDatabaseSpec(uid, DB_REF), session ->
                    createAndDeleteBlocks(session.getDb(), session::applyDeltas, uid, datas, deleteIds, actionInfo));

        } else {
            Database db = dbSupplier(uid).withNowaitLock(DynamicVars.nowaitLockForBlockMerge.get()).db();

            createAndDeleteBlocks(
                    db, (mode, deltas) -> dataApiManager.applyDeltas(UserDatabaseSpec.fromDatabase(db), db.rev, mode, deltas),
                    uid, datas, deleteIds, actionInfo);
        }
    }

    private void createAndDeleteBlocks(
            Database db, Function2V<RevisionCheckMode, ListF<Delta>> applyDeltas,
            DataApiUserId uid, ListF<FolderBlockData> datas, ListF<String> deleteIds, ReasonedAction actionInfo)
    {
        DatabaseSupplier dbs = new DatabaseSupplier(db);

        ListF<DataRecord> foundRecords = getRecords(dbs, RecordsFilter.DEFAULT
                .withRecordIdCond(RecordIdCondition.inSet(deleteIds.plus(datas.map(d -> d.blockId)))));

        ListF<DataRecordId> recordIdsToDelete = foundRecords.map(r -> r.id);

        ListF<LentaBlockRecord> recordsToCreate = datas.zipWithIndex()
                .map((data, idx) -> data.toRecord(db.rev + idx + 1));

        ListF<ListF<RecordChange>> changes = insertAndDeleteBlocksChanges(dbs, recordsToCreate, recordIdsToDelete);

        if (changes.isNotEmpty()) {
            applyDeltas.apply(RevisionCheckMode.WHOLE_DATABASE, changes.map(Delta::new));

            foundRecords.forEach(r -> blocksCache.invalidate(new UserBlockGroupKeyAndType(uid, r)));

            MapF<String, LentaBlockRecord> createdById = recordsToCreate.toMapMappingToKey(b -> b.id);

            IteratorF<LentaBlockEvent> created = recordsToCreate.iterator()
                    .filterNot(recordIdsToDelete.map(DataRecordId::recordId).containsF().compose(b -> b.id))
                    .map(r -> LentaBlockEvent.created(uid, r, actionInfo.actionInfo));

            IteratorF<LentaBlockEvent> updated = foundRecords
                    .zipWithFlatMapO(r -> createdById.getO(r.getRecordId())).iterator()
                    .map(r -> LentaBlockEvent.updatedAndUpped(uid, r.get1(), r.get2(), actionInfo.actionInfo));

            IteratorF<LentaBlockEvent> deleted = foundRecords.iterator()
                    .filterNot(r -> createdById.containsKeyTs(r.getRecordId()))
                    .map(r -> LentaBlockEvent.deleted(uid, r, actionInfo));

            created.plus(updated).plus(deleted).forEachRemaining(LentaBlockEvent::log);
        }
    }

    @Override
    public boolean updateBlock(DataApiUserId uid, String blockId, LentaBlockUpdateData data, ActionInfo actionInfo) {
        DatabaseSupplier dbs = dbSupplier(uid).withNowaitLock(DynamicVars.nowaitLockForBlockModify.get());

        return findBlockRecord(dbs, blockId, LookupCollections.ANY)
                .map(rec -> updateBlock(dbs, CachedOrLoadedBlock.loaded(rec), data, actionInfo))
                .getOrElse(false);
    }

    @Override
    public void updateOrDeleteBlock(
            DataApiUserId uid, String blockId, UpdateOrDeleteHandler handler, ActionInfo actionInfo)
    {
        Validate.notEquals(LentaNextIndexRecord.ID, blockId);

        DatabaseSupplier dbs = dbSupplier(uid).withNowaitLock(DynamicVars.nowaitLockForBlockModifyAsync.get());

        Option<DataRecord> rec = findBlockRecord(dbs, blockId, LookupCollections.ANY);

        if (!rec.isPresent()) return;

        UpdateOrDeleteHandler.Action action = handler.getAction(LentaBlockRecord.fromDataRecord(rec.get()));

        if (action instanceof UpdateOrDeleteHandler.Delete) {
            ListF<RecordChange> changes = deleteBlocksAndOrphanedCollectionsChanges(dbs, Option.of(rec.get().id));

            dataApiManager.applyDelta(dbs.db(), RevisionCheckMode.WHOLE_DATABASE, new Delta(changes));

            blocksCache.invalidate(new UserBlockGroupKeyAndType(uid, rec.get()));

            LentaBlockEvent.deleted(uid, rec.get(), ((UpdateOrDeleteHandler.Delete) action).reason, actionInfo).log();

        } else if (action instanceof UpdateOrDeleteHandler.Update) {
            updateBlock(dbs, CachedOrLoadedBlock.loaded(rec.get()),
                    ((UpdateOrDeleteHandler.Update) action).data, actionInfo);

        } else {
            LentaBlockEvent.deleteIgnored(uid, rec.get(), actionInfo).log();
        }
    }

    private boolean updateBlock(
            DatabaseSupplier dbs, CachedOrLoadedBlock block,
            LentaBlockUpdateData data, ActionInfo actionInfo)
    {
        Validate.notEquals(LentaNextIndexRecord.ID, block.getId());

        UpdateHandler.Action action = data.handler.getAction(block.toBlockRecord());

        if (action instanceof UpdateHandler.Ignore) {
            LentaBlockEvent.updateIgnored(dbs.uid, block, ((UpdateHandler.Ignore) action).reason, actionInfo).log();

            return false;
        }

        if (action instanceof UpdateHandler.Delayed) {
            updateAndUpOrDeleteBlockDelayed(dbs.uid, block.getIdAndType(), actionInfo);
            return true;
        }

        if (((UpdateHandler.Update) action).throttled) {
            return updateAndUpThrottled(dbs, block, data.base, ((UpdateHandler.Update) action).specific, actionInfo);

        } else if (((UpdateHandler.Update) action).updateAndUp) {
            return updateAndUpBlock(dbs, block, data.base, ((UpdateHandler.Update) action).specific, actionInfo);

        } else {
            return updateBlock(dbs, block, data.base, ((UpdateHandler.Update) action).specific, actionInfo);
        }
    }

    private boolean updateBlock(
            DatabaseSupplier db, CachedOrLoadedBlock block, LentaBlockBaseData data,
            Supplier<MapF<String, DataField>> specific, ActionInfo actionInfo)
    {
        Option<DataRecord> dataRecordO = findBlockRecordInvalidateAndReport(db, block);

        if (!dataRecordO.isPresent()) {
            return false;
        }
        DataRecord dataRecord = dataRecordO.get();

        LentaBlockRecord record = new LentaBlockRecord(
                dataRecord.getRecordId(),
                LentaBlockRecord.Fields.ORDER.get(dataRecord),
                data.type, data.groupKey,
                LentaBlockRecord.Fields.MFROM.getO(dataRecord),
                LentaBlockRecord.Fields.MTIME.get(dataRecord),
                LentaBlockRecord.Fields.MTILL.getO(dataRecord), specific.get());

        ListF<FieldChange> changes = record.diffFrom(dataRecord.getData());

        if (changes.isNotEmpty()) {
            dataApiManager.applyDelta(db.db(), RevisionCheckMode.PER_RECORD, new Delta(
                    RecordChange.update(dataRecord.getCollectionId(), record.id, changes)));
        }

        blocksCache.put(new UserBlockGroupKeyAndType(db.uid, record), record);

        LentaBlockEvent.updated(db.uid, dataRecord, record, actionInfo).log();

        return true;
    }

    private boolean updateAndUpBlock(
            DatabaseSupplier db, CachedOrLoadedBlock block, LentaBlockBaseData data,
            Supplier<MapF<String, DataField>> specific, ActionInfo actionInfo)
    {
        Validate.notEquals(LentaNextIndexRecord.ID, block.getId());

        Option<DataRecord> dataRecordO = findBlockRecordInvalidateAndReport(db, block);

        if (!dataRecordO.isPresent()) {
            return false;
        }
        DataRecord dataRecord = dataRecordO.get();

        long order = db.db().rev + 1;

        LentaBlockRecord record = new LentaBlockRecord(
                dataRecord.getRecordId(), order, data.type, data.groupKey,
                LentaBlockRecord.Fields.MFROM.getO(dataRecord), actionInfo.now,
                LentaBlockRecord.Fields.MTILL.getO(dataRecord), specific.get());

        ListF<RecordChange> changes = Cf.list();

        if (dataRecord.getCollectionId().startsWith("index_")) {
            Option<String> nextCollId = findNextIndexCollectionId(db, "index");
            ListF<RecordChange> siftChanges = siftIndexLastBlockIfExcessChanges(db);

            if (nextCollId.isSome(dataRecord.getCollectionId()) && siftChanges.isNotEmpty()) {
                changes = changes
                        .plus1(RecordChange.delete(dataRecord.getCollectionId(), record.id))
                        .plus1(RecordChange.insert("index", record.id, record.toData()))
                        .plus(siftChanges);
            } else {
                changes = changes
                        .plus(deleteBlocksAndOrphanedCollectionsChanges(db, Cf.list(dataRecord.id)))
                        .plus1(RecordChange.insert("index", record.id, record.toData()))
                        .plus(siftChanges);
            }
        } else {
            changes = changes.plus(RecordChange.update(
                    dataRecord.getCollectionId(), dataRecord.getRecordId(), record.diffFrom(dataRecord.getData())));
        }
        dataApiManager.applyDelta(db.db(), RevisionCheckMode.WHOLE_DATABASE, new Delta(changes));

        blocksCache.put(new UserBlockGroupKeyAndType(db.uid, record), record);

        LentaBlockEvent.updatedAndUpped(db.uid, dataRecord, record, actionInfo).log();

        return true;
    }

    private boolean updateAndUpThrottled(
            DatabaseSupplier dbs, CachedOrLoadedBlock block, LentaBlockBaseData data,
            Supplier<MapF<String, DataField>> specific, ActionInfo actionInfo)
    {
        DataApiUserId uid = dbs.uid;
        String blockId = block.getId();

        Instant mtime = block.getMTime();

        HolderForUpdateAndUpOrDeleteTask holder = new HolderForUpdateAndUpOrDeleteTask(uid, blockId);

        OnetimeJob holderJob = PgOnetimeUtils.makeJob(holder, Instant.now().plus(DynamicVars.blockUpdateDelay.get()));

        JobSaveResult scheduled;

        if (mtime.isBefore(actionInfo.now.minus(DynamicVars.blockUpdateDelay.get()))) {
            scheduled = bazingaStorage.addOnetimeJobX(holderJob, ActiveUidDuplicateBehavior.DO_NOTHING);
        } else {
            scheduled = new JobSaveResult.AutoCleaned(holderJob.getId());
        }

        if (scheduled.isCreated()) {
            actionInfo = actionInfo.withParameter("update_mode", "inplace");

            try {
                return updateAndUpBlock(dbs.withNowaitLock(DynamicVars.nowaitLockForBlockUpdateInplace.get()),
                        block, data, specific, actionInfo);

            } catch (RuntimeException e) {
                logger.warn("Inplace update failed for {} {}: {}", uid, blockId, ExceptionUtils.getAllMessages(e));
            }
        }
        Instant scheduleTime = Cf.list(
                scheduled.isMerged() ? scheduled.asMerged().job.getScheduleTime() : Instant.now(),
                scheduled.isAutoCleaned() ? mtime.plus(DynamicVars.blockUpdateDelay.get()) : Instant.now(),
                Instant.now().plus(DynamicVars.blockUpdateInplaceRetryDelay.get())).max();

        scheduled = schedule(new UpdateAndUpOrDeleteLentaBlockTask(uid, blockId, actionInfo),
                scheduleTime, uid, block.getIdAndType(), Option.empty(), actionInfo);

        if (scheduled.isCreated()) {
            bazingaStorage.addOnetimeJob(
                    PgOnetimeUtils.makeJob(holder, scheduleTime.plus(DynamicVars.blockUpdateDelay.get())),
                    ActiveUidDuplicateBehavior.MERGE);
        }
        return true;
    }

    @Override
    public Option<FindOrCreateResult> unpinBlockIfPinned(DataApiUserId uid, String blockId, ActionInfo actionInfo) {
        DatabaseSupplier dbs = dbSupplier(uid).withNowaitLock(DynamicVars.nowaitLockForBlockModifyAsync.get());

        Option<LentaBlockRecord> pinnedO = findBlock(dbs, blockId, LookupCollections.PINNED);

        return pinnedO.isPresent() ? Option.of(unpinPinnedBlock(dbs, pinnedO.get(), actionInfo)) : Option.empty();
    }

    @Override
    public Option<FindOrCreateResult> unpinBlockIfPinned(
            DataApiUserId uid, LentaRecordType type, String groupKey, ActionInfo actionInfo)
    {
        DatabaseSupplier dbs = dbSupplier(uid).withNowaitLock(DynamicVars.nowaitLockForBlockModify.get());

        Option<LentaBlockRecord> pinnedO = findLastPinnedBlock(dbs, type, groupKey);

        return pinnedO.isPresent() ? Option.of(unpinPinnedBlock(dbs, pinnedO.get(), actionInfo)) : Option.empty();
    }

    private FindOrCreateResult unpinPinnedBlock(DatabaseSupplier db, LentaBlockRecord pinned, ActionInfo actionInfo) {
        long order = db.db().rev + 1;

        LentaBlockRecord record = new LentaBlockRecord(
                pinned.id, order, pinned.type, pinned.groupKey,
                pinned.mFrom, Instant.now(), pinned.mTill, pinned.specific);

        ListF<RecordChange> changes = Cf.<RecordChange>list()
                .plus1(RecordChange.delete("pinned", record.id))
                .plus1(RecordChange.insert("index", record.id, record.toData()))
                .plus(siftIndexLastBlockIfExcessChanges(db));

        dataApiManager.applyDelta(db.db(), RevisionCheckMode.WHOLE_DATABASE, new Delta(changes));

        blocksCache.put(new UserBlockGroupKeyAndType(db.uid, record), record);

        LentaBlockEvent.unpin(db.uid, record, actionInfo).log();

        return FindOrCreateResult.created(record);
    }

    @Override
    public void deleteBlock(DataApiUserId uid, String blockId, ReasonedAction actionInfo) {
        deleteBlock(uid, blockId, rec -> DeleteHandler.delete(actionInfo.reason), actionInfo.actionInfo);
    }

    @Override
    public void deleteBlock(DataApiUserId uid, String blockId, DeleteHandler handler, ActionInfo actionInfo) {
        updateOrDeleteBlock(uid, blockId, rec -> {
            DeleteHandler.Action action = handler.getAction(rec);

            return action instanceof DeleteHandler.Delete
                    ? UpdateOrDeleteHandler.delete(((DeleteHandler.Delete) action).reason)
                    : UpdateOrDeleteHandler.ignore();
        }, actionInfo);
    }

    @Override
    public void deleteMostRecentBlock(
            DataApiUserId uid, LentaRecordType type, String groupKey, DeleteHandler handler, ActionInfo actionInfo)
    {
        DatabaseSupplier dbs = dbSupplier(uid).withNowaitLock(DynamicVars.nowaitLockForBlockModify.get());

        Option<DataRecord> recO = findLastBlockRecord(dbs, type, groupKey, LookupCollections.INDEX_AND_PINNED);

        if (recO.isPresent()) {
            DeleteHandler.Action action = handler.getAction(LentaBlockRecord.fromDataRecord(recO.get()));

            if (action instanceof DeleteHandler.Delete) {
                dataApiManager.applyDelta(dbs.db(), RevisionCheckMode.PER_RECORD,
                        new Delta(recO.map(rec -> RecordChange.delete(rec.id))));

                blocksCache.invalidate(new UserBlockGroupKeyAndType(uid, recO.get()));

                LentaBlockEvent.deleted(uid, recO.get(), ((DeleteHandler.Delete) action).reason, actionInfo).log();

            } else {
                LentaBlockEvent.deleteIgnored(uid, recO.get(), actionInfo).log();
            }
        }
    }

    @Override
    public void deleteBlocks(DataApiUserId uid, LentaRecordType type, String groupKey, ReasonedAction actionInfo) {
        DatabaseSupplier dbs = dbSupplier(uid).withNowaitLock(DynamicVars.nowaitLockForBlockModify.get());

        ListF<DataRecord> records = findAnyBlockRecords(dbs, type, GroupKeyPredicate.eq(groupKey));

        if (records.isEmpty()) return;

        dataApiManager.applyDelta(dbs.db(), RevisionCheckMode.WHOLE_DATABASE,
                new Delta(records.map(rec -> RecordChange.delete(rec.id))));

        records.forEach(r -> blocksCache.invalidate(new UserBlockGroupKeyAndType(uid, r)));

        records.map(r -> LentaBlockEvent.deleted(uid, r, actionInfo)).forEach(LentaBlockEvent::log);
    }

    @Override
    public void findAndUpThrottledLatestBlock(
            DataApiUserId uid, LentaRecordType type, String groupKey, ActionInfo actionInfo)
    {
        DatabaseSupplier dbs = dbSupplier(uid);

        Option<DataRecord> record = findLastBlockRecords(
                dbs, type, GroupKeyPredicate.eq(groupKey),
                Option.empty(), Option.empty(), LookupCollections.ANY, Option.of(1)).singleO();

        if (record.isPresent()) {
            LentaBlockRecord block = LentaBlockRecord.fromDataRecord(record.get());

            updateAndUpThrottled(dbs, CachedOrLoadedBlock.loaded(record.get()),
                    new LentaBlockBaseData(block.type, block.groupKey), () -> block.specific, actionInfo);
        }
    }

    @Override
    public void updateAndUpOrDeleteBlockDelayed(DataApiUserId uid, LentaBlockIdAndType block, ActionInfo actionInfo) {
        schedule(new UpdateAndUpOrDeleteLentaBlockTask(uid, block.id, actionInfo),
                Instant.now().plus(DynamicVars.blockUpdateDelay.get()),
                uid, block, Option.empty(), actionInfo);
    }

    @Override
    public void unpinBlockDelayed(DataApiUserId uid, LentaBlockIdAndType block, ActionInfo actionInfo) {
        schedule(new UnpinLentaBlockTask(uid, block.id),
                Instant.now().plus(DynamicVars.blockPinTtl.get()),
                uid, block, Option.empty(), actionInfo);
    }

    @Override
    public void deleteEmptyBlockDelayed(DataApiUserId uid, LentaBlockIdAndType block, ReasonedAction actionInfo) {
        schedule(new DeleteEmptyLentaBlockTask(uid, block.id, actionInfo),
                Instant.now().plus(DynamicVars.blockDeleteDelayRandom.get()),
                uid, block, Option.of(actionInfo.reason), actionInfo.actionInfo);
    }

    @Override
    public void deleteOrUpdateBlockDelayed(
            DataApiUserId uid, LentaBlockIdAndType block, Duration delay, ReasonedAction actionInfo)
    {
        schedule(new DeleteOrUpdateLentaBlockTask(uid, block.id, actionInfo),
                Instant.now().plus(delay), uid, block, Option.of(actionInfo.reason), actionInfo.actionInfo);
    }

    @Override
    public void deleteOrUpdateBlocksAsync(
            DataApiUserId uid, LentaRecordType type, FieldPredicate fieldFilter, ReasonedAction actionInfo)
    {
        schedule(new FindAndDeleteOrUpdateLentaBlocksTask(uid, type, fieldFilter, actionInfo),
                Instant.now().plus(DynamicVars.blocksFindAndModifyDelay.get()),
                uid, LentaBlockIdAndType.noId(type), Option.of(actionInfo.reason), actionInfo.actionInfo);
    }

    private JobSaveResult schedule(
            OnetimeTaskSupport<?> task, Instant scheduleTime,
            DataApiUserId uid, LentaBlockIdAndType block, Option<ActionReason> reason, ActionInfo actionInfo)
    {
        OnetimeJob job = PgOnetimeUtils.makeJob(
                task, scheduleTime, Option.of(OnetimeUtils.getActiveUniqueIdentifier(task)), task.priority());

        JobSaveResult result = bazingaStorage.addOnetimeJobX(job, task.activeUidBehavior().getDuplicateBehavior());

        if (result.isMerged()) {
            scheduleTime = result.asMerged().job.getScheduleTime();
            LentaBlockEvent.taskMerged(uid, result.getJobId(), scheduleTime, block, reason, actionInfo).log();

        } else {
            LentaBlockEvent.taskScheduled(uid, result.getJobId(), scheduleTime, block, reason, actionInfo).log();
        }
        return result;
    }

    @Override
    public int deleteEldestBlocks(DataApiUserId uid, int limit, ReasonedAction actionInfo) {
        DatabaseSupplier dbs = dbSupplier(uid);

        Tuple2<ListF<RecordChange>, ListF<DataRecord>> changesRecords = deleteEldestBlocksChanges(dbs, limit);

        if (changesRecords.get1().isNotEmpty()) {
            dataApiManager.applyDelta(dbs.db(), RevisionCheckMode.WHOLE_DATABASE, new Delta(changesRecords.get1()));

            changesRecords.get2().forEach(r -> blocksCache.invalidate(new UserBlockGroupKeyAndType(uid, r)));

            changesRecords.get2().map(r -> LentaBlockEvent.deleted(uid, r, actionInfo)).forEach(LentaBlockEvent::log);
        }
        return changesRecords.get2().size();
    }

    private Tuple2<ListF<RecordChange>, ListF<DataRecord>> deleteEldestBlocksChanges(DatabaseSupplier db, int limit) {
        if (limit <= 0) return Tuple2.tuple(Cf.list(), Cf.list());

        RecordsFilter filter = RecordsFilter.DEFAULT
                .withCollectionIdCond(LookupCollections.INDEXES.condition)
                .withRecordIdCond(RecordIdCondition.ne(LentaNextIndexRecord.ID))
                .withRecordOrder(ByRevisionOrder.ORDER)
                .withLimits(SqlLimits.first(limit));

        ListF<DataRecord> records = getRecords(db, filter);

        return Tuple2.tuple(deleteBlocksAndOrphanedCollectionsChanges(db, records.map(r -> r.id)), records);
    }

    private ListF<RecordChange> siftIndexLastBlockIfExcessChanges(DatabaseSupplier db) {
        if (findBlocksCount(db, "index") < BLOCKS_COUNT_LIMIT) {
            return Cf.list();
        }
        Option<LentaBlockRecord> lastBlockO = findLeastOrderedBlockRecord(db, "index");

        if (!lastBlockO.isPresent()) {
            return Cf.list();
        }
        LentaBlockRecord lastBlock = lastBlockO.get();
        ListF<RecordChange> result = Cf.arrayList();

        result.add(RecordChange.delete("index", lastBlock.id));

        Option<String> nextCollId = findNextIndexCollectionId(db, "index");

        if (!nextCollId.isPresent()) {
            nextCollId = Option.of(generateNextIndexCollectionId("index"));

            result.add(RecordChange.insert(nextCollId.get(), lastBlock.id, lastBlock.toData()));

            result.add(RecordChange.insert("index", LentaNextIndexRecord.ID,
                    new LentaNextIndexRecord(nextCollId.get()).toData()));

        } else if (findBlocksCount(db, nextCollId.get()) >= BLOCKS_COUNT_LIMIT) {
            String nextForNextCollId = generateNextIndexCollectionId(nextCollId.get());

            result.add(RecordChange.insert(nextForNextCollId, lastBlock.id, lastBlock.toData()));

            result.add(RecordChange.update("index", LentaNextIndexRecord.ID,
                    LentaNextIndexRecord.Fields.COLLECTION_ID.changedTo(nextForNextCollId)));

            result.add(RecordChange.insert(nextForNextCollId, LentaNextIndexRecord.ID,
                    new LentaNextIndexRecord(nextCollId.get()).toData()));
        } else {
            result.add(RecordChange.insert(nextCollId.get(), lastBlock.id, lastBlock.toData()));
        }
        return result;
    }

    private ListF<RecordChange> deleteBlocksAndOrphanedCollectionsChanges(
            DatabaseSupplier db, ListF<DataRecordId> recordIds)
    {
        if (recordIds.isEmpty()) return Cf.list();

        db.obtain();

        ListF<RecordChange> result = Cf.arrayList();

        result.addAll(recordIds.map(RecordChange::delete));

        MapF<String, Integer> counts = recordIds
                .filter(id -> id.collectionId().startsWith("index_"))
                .countBy(DataRecordId::collectionId);

        ListF<String> orphaned = counts.entrySet()
                .filterMap(e -> Option.when(findBlocksCount(db, e.getKey()) == e.getValue(), e.getKey()))
                .sortedByDesc(LentaManagerImpl::parseIndexCollectionNumber);

        for (int i = 0, j; i < orphaned.size(); i = j) {
            Option<String> prevCollId = findPrevIndexCollectionId(db, orphaned.get(i));
            Option<String> nextCollId = findNextIndexCollectionId(db, orphaned.get(i));

            for (j = i + 1; j < orphaned.size() && nextCollId.isSome(orphaned.get(j)); ++j) {
                nextCollId = findNextIndexCollectionId(db, orphaned.get(j));

                result.add(RecordChange.delete(orphaned.get(j - 1), LentaNextIndexRecord.ID));
            }

            if (nextCollId.isPresent() && prevCollId.isPresent()) {
                result.add(RecordChange.update(prevCollId.get(), LentaNextIndexRecord.ID,
                        LentaNextIndexRecord.Fields.COLLECTION_ID.changedTo(nextCollId.get())));

                result.add(RecordChange.delete(orphaned.get(j - 1), LentaNextIndexRecord.ID));

            } else if (prevCollId.isPresent()) {
                result.add(RecordChange.delete(prevCollId.get(), LentaNextIndexRecord.ID));
            }
        }
        return result;
    }

    private ListF<ListF<RecordChange>> insertAndDeleteBlocksChanges(
            DatabaseSupplier db, ListF<LentaBlockRecord> insertBlocks, ListF<DataRecordId> deleteIds)
    {
        ListF<RecordChange> deletions = deleteBlocksAndOrphanedCollectionsChanges(db, deleteIds);

        ListF<DataRecordId> indexDeleteIds = deleteIds.filter(r -> r.collectionId().equals("index"));

        int indexSize = findBlocksCount(db, "index") - indexDeleteIds.size();

        int indexCapacity = BLOCKS_COUNT_LIMIT - indexSize;

        if (insertBlocks.size() <= indexCapacity) {
            return Option.when(deletions.isNotEmpty(), deletions)
                    .plus(insertBlocks.map(r -> Cf.list(RecordChange.insert("index", r.id, r.toData()))));
        }

        ListF<ListF<RecordChange>> changes = Cf.arrayList();

        Option<String> nextId = findNextIndexCollectionIdFromChangesOrDb(db, deletions);

        String[] collId = new String[] { nextId.getOrElse(generateNextIndexCollectionId("index")) };

        int siftCount = insertBlocks.size() < BLOCKS_COUNT_LIMIT ? insertBlocks.size() - indexCapacity : indexSize;

        ListF<DataRecord> siftRecs = findLeastOrderedRecords(db, "index", siftCount + indexDeleteIds.size())
                .filterNot(indexDeleteIds.unique().containsF().compose(r -> r.id)).take(siftCount);

        ListF<RecordChange> siftChanges = Cf.arrayList();

        siftChanges.addAll(siftRecs.map(r -> RecordChange.delete("index", r.getRecordId())));

        int nextCapacity = nextId.map(coll -> BLOCKS_COUNT_LIMIT - findBlocksCount(db, coll)
                + deleteIds.count(r -> r.collectionId().equals(coll))).getOrElse(BLOCKS_COUNT_LIMIT);

        siftChanges.addAll(siftRecs.take(nextCapacity).map(r ->
                RecordChange.insert(collId[0], r.getRecordId(), LentaBlockRecord.fromDataRecord(r).toData())));

        changes.add(siftChanges.plus(Option.when(!nextId.isPresent(),
                RecordChange.insert("index", LentaNextIndexRecord.ID, new LentaNextIndexRecord(collId[0]).toData()))));

        changes.addAll(insertBlocks.rtake(BLOCKS_COUNT_LIMIT)
                .map(r -> Cf.list(RecordChange.insert("index", r.id, r.toData()))));

        insertBlocks = siftRecs.drop(nextCapacity).map(LentaBlockRecord::fromDataRecord)
                .plus(insertBlocks.rdrop(BLOCKS_COUNT_LIMIT));

        changes.addAll(insertBlocks.take(nextCapacity - siftRecs.size())
                .map(r -> Cf.list(RecordChange.insert(collId[0], r.id, r.toData()))));

        insertBlocks.drop(nextCapacity - siftRecs.size()).paginate(BLOCKS_COUNT_LIMIT).forEach(blocks -> {
            String prevId = collId[0];
            collId[0] = generateNextIndexCollectionId(collId[0]);

            ListF<RecordChange> nextChanges = Cf.list(
                    RecordChange.insert(collId[0], LentaNextIndexRecord.ID,
                            new LentaNextIndexRecord(prevId).toData()),
                    RecordChange.update("index", LentaNextIndexRecord.ID,
                            LentaNextIndexRecord.Fields.COLLECTION_ID.changedTo(collId[0])));

            changes.add(nextChanges);
            changes.addAll(blocks.map(b -> Cf.list(RecordChange.insert(collId[0], b.id, b.toData()))));
        });

        return Option.of(deletions).plus(changes).filterNot(Cf.List.isEmptyF());
    }

    private Option<LentaBlockRecord> findLastPinnedBlock(DatabaseSupplier db, LentaRecordType type, String group) {
        LookupCollections collections = LookupCollections.PINNED;

        ListF<DataRecord> records = findLastBlockRecords(
                db, type, GroupKeyPredicate.eq(group), Option.empty(), Option.empty(), collections, Option.of(1));

        return records.singleO().map(LentaBlockRecord::fromDataRecord);
    }

    private Option<LentaBlockRecord> findLastBlock(
            DatabaseSupplier db, LentaRecordType type, String group,
            Option<Instant> mFromSince, Option<Instant> mFromTill, LookupCollections collections)
    {
        return findLastBlockRecord(db, type, group, mFromSince, mFromTill, collections)
                .map(LentaBlockRecord::fromDataRecord);
    }

    private Option<DataRecord> findLastBlockRecord(
            DatabaseSupplier db, LentaRecordType type, String group, LookupCollections collections)
    {
        return findLastBlockRecord(db, type, group, Option.empty(), Option.empty(), collections);
    }

    private Option<DataRecord> findLastBlockRecord(
            DatabaseSupplier db, LentaRecordType type, String group,
            Option<Instant> mFromSince, Option<Instant> mFromTill, LookupCollections collections)
    {
        return findLastBlockRecords(
                db, type, GroupKeyPredicate.eq(group), mFromSince, mFromTill, collections, Option.of(1)).singleO();
    }

    private ListF<DataRecord> findLastBlockRecords(
            DatabaseSupplier db, LentaRecordType type, FieldPredicate fieldFilter,
            Option<Instant> mFromSince, Option<Instant> mFromTill,
            LookupCollections collections, Option<Integer> limit)
    {
        CollectionIdCondition collectionIdCond = collections.condition;

        DataCondition dataCond = DataCondition.all()
                .and(mFromSince.map(LentaBlockRecord.Fields.MFROM.column()::ge))
                .and(mFromTill.map(LentaBlockRecord.Fields.MFROM.column()::lt))
                .and(LentaBlockRecord.Fields.TYPE.column().eq(type))
                .and(fieldFilter.toDataCondition());

        RecordsFilter filter = RecordsFilter.DEFAULT
                .withCollectionIdCond(collectionIdCond)
                .withDataCond(dataCond)
                .withRecordOrder(ByIdRecordOrder.RECORD_ID_DESC)
                .withLimits(limit.map(SqlLimits::first).getOrElse(SqlLimits.all()));

        return getRecords(db, filter);
    }

    private ListF<DataRecord> findAnyBlockRecords(
            DatabaseSupplier db, LentaRecordType type, FieldPredicate fieldFilter)
    {
        CollectionIdCondition collectionIdCond = LookupCollections.ANY.condition;

        DataCondition dataCond = DataCondition.all()
                .and(LentaBlockRecord.Fields.TYPE.column().eq(type))
                .and(fieldFilter.toDataCondition());

        RecordsFilter filter = RecordsFilter.DEFAULT
                .withCollectionIdCond(collectionIdCond)
                .withDataCond(dataCond)
                .withRecordOrder(ByIdRecordOrder.RECORD_ID_ASC);

        return getRecords(db, filter);
    }

    private Option<LentaBlockRecord> findBlock(DatabaseSupplier db, String blockId, LookupCollections collections) {
        return findBlockRecord(db, blockId, collections).map(LentaBlockRecord::fromDataRecord);
    }

    private Option<DataRecord> findBlockRecordInvalidateAndReport(DatabaseSupplier db, CachedOrLoadedBlock block) {
        if (block.isCached()) {
            Option<DataRecord> recordO = findBlockRecord(db, block.asCached().id, LookupCollections.ANY);

            if (!recordO.exists(rec -> LentaBlockRecord.Fields.MTIME.get(rec).isEqual(block.getMTime()))) {
                blocksCache.incOutdated();
                blocksCache.invalidate(new UserBlockGroupKeyAndType(db.uid, block));
            }
            return recordO;

        } else {
            return Option.of(block.asLoaded());
        }
    }

    private Option<DataRecord> findBlockRecord(DatabaseSupplier db, String recordId, LookupCollections collections) {
        CollectionIdCondition collectionIdCond = collections.condition;

        RecordIdCondition recordIdCond = RecordIdCondition.eq(recordId);

        RecordsFilter filter = RecordsFilter.DEFAULT
                .withCollectionIdCond(collectionIdCond)
                .withRecordIdCond(recordIdCond)
                .withRecordOrder(ByIdRecordOrder.RECORD_ID_DESC)
                .withLimits(SqlLimits.first(1));

        return getRecords(db, filter).singleO();
    }

    private int findBlocksCount(DatabaseSupplier db, String collectionId) {
        return getRecordsCount(db, RecordsFilter.DEFAULT
                .withCollectionId(collectionId)
                .withRecordIdCond(RecordIdCondition.ne(LentaNextIndexRecord.ID)));
    }

    private Option<LentaBlockRecord> findLeastOrderedBlockRecord(DatabaseSupplier db, String collectionId) {
        return findLeastOrderedRecords(db, collectionId, 1).singleO().map(LentaBlockRecord::fromDataRecord);
    }

    private ListF<DataRecord> findLeastOrderedRecords(DatabaseSupplier db, String collectionId, int limit) {
        return getRecords(db, RecordsFilter.DEFAULT
                .withCollectionId(collectionId)
                .withRecordIdCond(RecordIdCondition.ne(LentaNextIndexRecord.ID))
                .withRecordOrder(LentaBlockRecord.Fields.ORDER.column().orderBy())
                .withLimits(SqlLimits.first(limit)));
    }

    private Option<String> findNextIndexCollectionIdFromChangesOrDb(DatabaseSupplier db, ListF<RecordChange> changes) {
        Option<RecordChange> nextCollChange = changes
                .find(c -> c.getRecordId().equals(new SimpleRecordId("index", LentaNextIndexRecord.ID)));

        if (nextCollChange.exists(c -> c.type != RecordChangeType.DELETE)) {
            return Option.of(nextCollChange.get().fieldChanges.filterMap(c -> Option.when(
                    c.key.equals(LentaNextIndexRecord.Fields.COLLECTION_ID.name),
                    c.getValue().stringValue())).single());

        } else if (nextCollChange.exists(c -> c.type == RecordChangeType.DELETE)) {
            return Option.empty();

        } else {
            return findNextIndexCollectionId(db, "index");
        }
    }

    private Option<String> findNextIndexCollectionId(DatabaseSupplier db, String collectionId) {
        ListF<DataRecord> records = getRecords(
                db, RecordsFilter.DEFAULT.withCollectionId(collectionId).withRecordId(LentaNextIndexRecord.ID));

        return records.singleO().map(LentaNextIndexRecord.Fields.COLLECTION_ID::get);
    }

    private Option<String> findPrevIndexCollectionId(DatabaseSupplier db, String collectionId) {
        Validate.isFalse(collectionId.equals("index"));

        Option<String> indexNextCollId = findNextIndexCollectionId(db, "index");

        if (!indexNextCollId.isPresent()) {
            return Option.empty();
        }
        if (indexNextCollId.isSome(collectionId)) {
            return Option.of("index");
        }
        int indexNextCollectionNum = parseIndexCollectionNumber(indexNextCollId.get());
        int collectionNum = parseIndexCollectionNumber(collectionId);

        while (collectionNum < indexNextCollectionNum) {
            collectionId = generateNextIndexCollectionId(collectionId);
            collectionNum = parseIndexCollectionNumber(collectionId);

            if (findNextIndexCollectionId(db, collectionId).isPresent()) {
                return Option.of(collectionId);
            }
        }
        return Option.empty();
    }

    private DatabaseSupplier dbSupplier(DataApiUserId uid) {
        return new DatabaseSupplier(uid, Option.empty());
    }

    private ListF<DataRecord> getRecords(DatabaseSupplier supplier, RecordsFilter filter) {
        return supplier.handle().flatMap(spec -> dataApiManager.getRecords(spec, filter));
    }

    private int getRecordsCount(DatabaseSupplier supplier, RecordsFilter filter) {
        return supplier.handle().map(spec -> dataApiManager.getRecordsCount(spec, filter)).getOrElse(0);
    }

    private static int parseIndexCollectionNumber(String collectionId) {
        return collectionId.equals("index") ? 0 : Integer.parseInt(collectionId.replaceAll("^index_0*", ""));
    }

    private static String generateNextIndexCollectionId(String collectionId) {
        return "index_" + (parseIndexCollectionNumber(collectionId) + 1);
    }

    enum LookupCollections {
        INDEXES(CollectionIdCondition.like("index%")),
        PINNED(CollectionIdCondition.eq("pinned")),
        INDEX_AND_PINNED(CollectionIdCondition.inSet(Cf.list("index", "pinned"))),
        ANY(CollectionIdCondition.all()),
        ;

        public final CollectionIdCondition condition;

        LookupCollections(CollectionIdCondition condition) {
            this.condition = condition;
        }
    }

    class DatabaseSupplier {
        private final DataApiUserId uid;
        private final Option<Boolean> nowaitLock;

        private volatile Option<Database> database = Option.empty();

        public DatabaseSupplier(DataApiUserId uid, Option<Boolean> nowaitLock) {
            this.uid = uid;
            this.nowaitLock = nowaitLock;
        }

        public DatabaseSupplier(Database db) {
            this(db.uid, Option.empty());
        }

        public Option<UserDatabaseSpec> handle() {
            Option<DatabaseHandle> handle = dbHandleCache.getO(uid);

            if (!handle.isPresent()) {
                if (!database.isPresent()) {
                    database = dataApiManager.getDatabaseO(new UserDatabaseSpec(uid, DB_REF));
                }
                handle = database.map(db -> db.dbHandle);
                handle.forEach(h -> dbHandleCache.put(uid, h));
            } else {
                dbHandleCache.incHits();
            }
            return handle.map(h -> UserDatabaseSpec.fromUserAndHandle(uid, h));
        }

        public Database db() {
            obtain();
            return nowaitLock.map(database.get()::withNowaitLock).getOrElse(database.get());
        }

        public void obtain() {
            if (!database.isPresent()) {
                database = Option.of(dataApiManager.getOrCreateDatabase(new UserDatabaseSpec(uid, DB_REF)));

                dbHandleCache.put(uid, database.get().dbHandle);
            }
        }

        public DatabaseSupplier withNowaitLock(boolean value) {
            DatabaseSupplier that = new DatabaseSupplier(uid, Option.of(value));

            that.database = database;
            return that;
        }
    }

    static void setBlocksCountLimitForTest(int limit) {
        BLOCKS_COUNT_LIMIT = limit;
    }
}
