package ru.yandex.chemodan.app.lentaloader.lenta;

import com.google.common.cache.Cache;
import org.apache.commons.lang3.mutable.MutableObject;
import org.jetbrains.annotations.NotNull;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.junit.Before;
import org.junit.Test;
import org.mockito.ArgumentCaptor;
import org.mockito.Mockito;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function0;
import ru.yandex.bolts.function.Function1V;
import ru.yandex.bolts.function.Function2V;
import ru.yandex.bolts.function.Function3B;
import ru.yandex.bolts.function.forhuman.Comparator;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataField;
import ru.yandex.chemodan.app.dataapi.api.data.filter.RecordsFilter;
import ru.yandex.chemodan.app.dataapi.api.data.filter.ordering.ByIdRecordOrder;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataRecord;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataRecordId;
import ru.yandex.chemodan.app.dataapi.api.db.Database;
import ru.yandex.chemodan.app.dataapi.api.db.handle.DatabaseHandle;
import ru.yandex.chemodan.app.dataapi.api.db.ref.UserDatabaseSpec;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.dataapi.core.manager.DataApiManager;
import ru.yandex.chemodan.app.dataapi.support.I18nValue;
import ru.yandex.chemodan.app.dataapi.test.DataApiTestSupport;
import ru.yandex.chemodan.app.lentaloader.DynamicVars;
import ru.yandex.chemodan.app.lentaloader.blocks.GenericBlockData;
import ru.yandex.chemodan.app.lentaloader.blocks.GenericBlockFields;
import ru.yandex.chemodan.app.lentaloader.lenta.LentaManagerImpl.LookupCollections;
import ru.yandex.chemodan.app.lentaloader.lenta.limit.FolderBlockData;
import ru.yandex.chemodan.app.lentaloader.lenta.update.CreateHandler;
import ru.yandex.chemodan.app.lentaloader.lenta.update.DeleteHandler;
import ru.yandex.chemodan.app.lentaloader.lenta.update.LentaBlockCreateData;
import ru.yandex.chemodan.app.lentaloader.lenta.update.LentaBlockModifyData;
import ru.yandex.chemodan.app.lentaloader.lenta.update.LentaBlockUpdateData;
import ru.yandex.chemodan.app.lentaloader.lenta.update.UpdateHandler;
import ru.yandex.chemodan.app.lentaloader.log.ActionInfo;
import ru.yandex.chemodan.app.lentaloader.log.ActionReason;
import ru.yandex.chemodan.app.lentaloader.log.ActionSource;
import ru.yandex.chemodan.app.lentaloader.log.ReasonedAction;
import ru.yandex.chemodan.app.uaas.experiments.ExperimentsManager;
import ru.yandex.chemodan.cache.MeteredCache;
import ru.yandex.chemodan.eventlog.events.MpfsPath;
import ru.yandex.chemodan.mpfs.MpfsResourceId;
import ru.yandex.chemodan.mpfs.MpfsUid;
import ru.yandex.commune.bazinga.impl.FullJobId;
import ru.yandex.commune.bazinga.impl.OnetimeJob;
import ru.yandex.commune.bazinga.pg.storage.JobSaveResult;
import ru.yandex.commune.bazinga.pg.storage.PgBazingaStorage;
import ru.yandex.commune.bazinga.pg.storage.dao.support.BazingaRandomValueGenerator;
import ru.yandex.inside.utils.Language;
import ru.yandex.misc.random.Random2;
import ru.yandex.misc.reflection.ClassX;
import ru.yandex.misc.regex.Pattern2;
import ru.yandex.misc.test.Assert;

/**
 * @author dbrylev
 */
public class LentaManagerTest extends DataApiTestSupport {

    private PgBazingaStorage bazingaStorage;

    private LentaManagerImpl lentaManager;
    @Autowired
    private DataApiManager dataApiManager;

    private Cache<UserBlockGroupKeyAndType, LentaBlockRecord> blocksCache;

    private Database db;
    private DataApiUserId uid;

    @Before
    public void setup() {
        uid = createRandomCleanUserInDefaultShard();
        db = dataApiManager.createDatabase(new UserDatabaseSpec(uid, LentaManagerImpl.DB_REF));

        bazingaStorage = Mockito.mock(PgBazingaStorage.class);

        Mockito.when(bazingaStorage.addOnetimeJobX(Mockito.any(), Mockito.any()))
                .thenReturn(new JobSaveResult.Created(FullJobId.parse("NONE/NONE")));

        blocksCache = Mockito.mock(ClassX.wrap(Cache.class)
                .<Cache<UserBlockGroupKeyAndType, LentaBlockRecord>>uncheckedCast().getClazz());

        lentaManager = new LentaManagerImpl(
                dataApiManager, bazingaStorage, getExperimentsManager(), new MeteredCache<>(blocksCache),
                new MeteredCache<>(1, 1, Duration.standardMinutes(5)));

        LentaManagerImpl.setBlocksCountLimitForTest(2);
    }

    @NotNull
    private ExperimentsManager getExperimentsManager() {
        ExperimentsManager experimentsManager = Mockito.mock(ExperimentsManager.class);
        Mockito.when(experimentsManager.getFlags(Mockito.anyLong())).thenReturn(Cf.list());
        return experimentsManager;
    }

    @Test
    public void siftLastBlock() {
        Cf.range(1, 7).map(group -> lentaManager.findOrCreateBlock(
                uid, contentBlockCreateData(group + ""), actionInfoNow()));

        Assert.equals(Cf.list("6", "5", "index_2", "4", "3", "index_1", "2", "1"),
                loadIndexesRecordsGroupsOrCollectionIds(db));
    }

    @Test
    public void createIgnore() {
        LentaBlockModifyData data = new LentaBlockModifyData(
                LentaRecordType.CONTENT_BLOCK, "group",
                rec -> CreateHandler.ignore(ActionReason.UNSPECIFIED),
                rec -> UpdateHandler.ignore(ActionReason.UNSPECIFIED));

        Assert.isTrue(lentaManager.findAndUpdateOrCreateBlock(uid, data, actionInfoNow()).isIgnored());
        Assert.isEmpty(loadIndexesRecordsGroupsOrCollectionIds(db));
    }

    @Test
    public void deleteOrphanedCollection() {
        Function0<ListF<String>> load = () -> loadIndexesRecordsGroupsOrCollectionIds(db);

        Function1V<ListF<String>> delete = ids -> ids.forEach(
                id -> lentaManager.deleteBlock(uid, id, actionReasonNow()));

        ListF<String> recordIds = Cf.range(0, 8).map(group ->
                lentaManager.findOrCreateBlock(uid, contentBlockCreateData(group + ""), actionInfoNow()).getRecordId());

        Assert.equals(Cf.list("7", "6", "index_3", "5", "4", "index_2", "3", "2", "index_1", "1", "0"), load.apply());

        delete.apply(recordIds.subList(0, 2));
        Assert.equals(Cf.list("7", "6", "index_3", "5", "4", "index_2", "3", "2"), load.apply());

        delete.apply(recordIds.subList(4, 6));
        Assert.equals(Cf.list("7", "6", "index_2", "3", "2"), load.apply());

        delete.apply(recordIds.subList(2, 4));
        Assert.equals(Cf.list("7", "6"), load.apply());

        delete.apply(recordIds.subList(6, 8));
        Assert.isEmpty(load.apply());
    }

    @Test
    public void updateBlock() {
        Instant since = Instant.now().minus(1);
        Instant mTime = since.plus(1);

        String blockId = lentaManager.findOrCreateBlock(uid, contentBlockCreateData(
                "group", Cf.map("media_type", DataField.string("audio"))), actionInfo(since)).getRecordId();

        LentaBlockRecord block = lentaManager.findBlock(uid, blockId, actionInfoNow()).get();

        lentaManager.updateBlock(uid, blockId, LentaBlockUpdateData.update(
                LentaRecordType.CONTENT_BLOCK, "group",
                Cf.map("media_type", DataField.string("video"))), actionInfo(mTime));

        LentaBlockRecord updated = lentaManager.findBlock(uid, blockId, actionInfoNow()).get();

        Assert.equals(DataField.string("audio"), block.specific.getTs("media_type"));
        Assert.equals(DataField.string("video"), updated.specific.getTs("media_type"));

        Assert.equals(block.mFrom, updated.mFrom);
        Assert.equals(block.mTime, updated.mTime);
        Assert.equals(block.mTill, updated.mTill);
    }

    @Test
    public void findAndUpdateBlock() {
        String blockId = lentaManager.findOrCreateBlock(uid, contentBlockCreateData(
                "key", Cf.map("media", DataField.string("audio"))), actionInfoNow()).getRecordId();

        Function3B<String, UpdateHandler, Instant> update = (group, handler, now) -> lentaManager.findAndUpdateBlock(
                uid, new LentaBlockUpdateData(LentaRecordType.CONTENT_BLOCK, group, handler), actionInfo(now));

        Assert.isFalse(update.apply("key", rec -> UpdateHandler.ignore(ActionReason.UNSPECIFIED), now));

        Assert.isFalse(update.apply("lock", rec -> UpdateHandler.update(Cf.map()), now));

        Assert.isFalse(update.apply("key", rec -> UpdateHandler.update(Cf.map()), now.minus(Duration.standardDays(2))));

        Assert.isTrue(update.apply("key", rec -> UpdateHandler.update(Cf.map("media", DataField.string("text"))), now));

        LentaBlockRecord updated = lentaManager.findBlock(uid, blockId, actionInfoNow()).get();

        Assert.equals(DataField.string("text"), updated.specific.getTs("media"));
    }

    @Test
    public void findCached() {
        DataApiManager dataApi = Mockito.mock(DataApiManager.class);
        LentaManagerImpl lentaManager = new LentaManagerImpl(
                dataApi, bazingaStorage, getExperimentsManager(), new MeteredCache<>(blocksCache), MeteredCache.noCache());

        Mockito.when(dataApi.getOrCreateDatabase(Mockito.any())).thenReturn(db);
        Mockito.when(dataApi.getDatabaseO(Mockito.any())).thenReturn(Option.of(db));

        Instant now = Instant.now();

        MutableObject<String> blockIdCaptor = new MutableObject<>();

        Function<LentaRecordType, LentaBlockUpdateData> mock = type -> {
            LentaBlockRecord cached = new LentaBlockRecord("cached", 1, type, "group",
                    Option.of(now), now, Option.of(now.plus(Duration.standardHours(1))), Cf.map());

            DataRecord loaded = new DataRecord(
                    uid, new DataRecordId(new DatabaseHandle("", "", ""), "", "loaded"), 1,
                    cached.toData().plus(Cf.toMap(Cf.list(
                            LentaBlockRecord.Fields.MFROM.toData(now.minus(Duration.standardDays(10))),
                            LentaBlockRecord.Fields.MTILL.toData(now.plus(Duration.standardDays(10)))))));

            Mockito.when(blocksCache.getIfPresent(Mockito.any())).thenReturn(cached);
            Mockito.when(dataApi.getRecords(Mockito.any(), Mockito.any())).thenReturn(Cf.list(loaded));

            return new LentaBlockUpdateData(cached.type, cached.groupKey, rec -> {
                blockIdCaptor.setValue(rec.id);
                return UpdateHandler.ignore(ActionReason.UNSPECIFIED);
            });
        };
        LentaBlockUpdateData mocked = mock.apply(LentaRecordType.CONTENT_BLOCK);

        lentaManager.findCachedAndUpdateBlock(uid, mocked, actionInfo(now.minus(Duration.standardDays(1))));
        Assert.equals("loaded", blockIdCaptor.getValue());

        lentaManager.findCachedAndUpdateBlock(uid, mocked, actionInfo(now));
        Assert.equals("cached", blockIdCaptor.getValue());

        lentaManager.findCachedAndUpdateBlock(uid, mocked, actionInfo(now.plus(Duration.standardDays(3))));
        Assert.equals("loaded", blockIdCaptor.getValue());

        mocked = mock.apply(LentaRecordType.PUBLIC_RESOURCE);

        lentaManager.findCachedAndUpdateBlock(uid, mocked, actionInfo(now.plus(Duration.standardDays(1))));
        Assert.equals("cached", blockIdCaptor.getValue());

        blockIdCaptor.setValue(null);

        lentaManager.findCachedAndUpdateBlock(uid, mocked, actionInfo(now.plus(Duration.standardDays(3))));
        Assert.equals("cached", blockIdCaptor.getValue());
    }

    @Test
    public void updateCached() {
        String blockId = lentaManager.findOrCreateBlock(uid, contentBlockCreateData(
                "group", Cf.map("source", DataField.string("database"))), actionInfoNow()).getRecordId();

        LentaBlockRecord cached = new LentaBlockRecord(
                blockId, 1, LentaRecordType.CONTENT_BLOCK, "group",
                Option.of(now), now, Option.of(now.plus(Duration.standardHours(1))),
                Cf.map("source", DataField.string("cached")));

        LentaBlockUpdateData update = new LentaBlockUpdateData(
                cached.type, cached.groupKey, rec -> UpdateHandler.update(rec.specific));

        Mockito.when(blocksCache.getIfPresent(Mockito.any())).thenReturn(cached);

        Assert.isTrue(lentaManager.findCachedAndUpdateBlock(uid, update, actionInfoNow()));

        LentaBlockRecord block = lentaManager.findBlock(uid, blockId, actionInfoNow()).get();

        Assert.equals("cached", block.specific.getTs("source").stringValue());
    }

    @Test
    public void updateAndUpBlockThrottled() {
        MutableObject<Instant> now = new MutableObject<>(Instant.now());

        String blockId = lentaManager.findOrCreateBlock(
                uid, contentBlockCreateData("group"), actionInfo(now.getValue())).getRecordId();

        Function1V<String> update = mediaType -> lentaManager.updateBlock(uid, blockId,
                LentaBlockUpdateData.updateAndUpThrottled(
                        LentaRecordType.CONTENT_BLOCK, "group", Cf.map("media_type", DataField.string(mediaType))),
                actionInfo(now.getValue()));

        Function0<String> find = () -> lentaManager.findBlock(uid, blockId, actionInfo(now.getValue())).get()
                .specific.getTs("media_type").stringValue();

        ArgumentCaptor<OnetimeJob> jobCaptor = ArgumentCaptor.forClass(OnetimeJob.class);

        Function1V<JobSaveResult> mockScheduler = result -> {
            Mockito.reset(bazingaStorage);
            Mockito.when(bazingaStorage.addOnetimeJobX(jobCaptor.capture(), Mockito.any())).thenReturn(result);
        };

        now.setValue(now.getValue().plus(Duration.standardHours(1)));

        update.apply("audio");
        Assert.equals("audio", find.apply());

        mockScheduler.apply(new JobSaveResult.Created(FullJobId.parse("NONE/NONE")));

        now.setValue(now.getValue().plus(Duration.standardHours(1)));

        update.apply("video");
        Assert.equals("video", find.apply());

        Mockito.verify(bazingaStorage, Mockito.only()).addOnetimeJobX(Mockito.any(), Mockito.any());
        Assert.isTrue(jobCaptor.getValue().getTaskId().getId().endsWith(".holder"));

        mockScheduler.apply(new JobSaveResult.Merged(new BazingaRandomValueGenerator().randomValue(OnetimeJob.class)));

        now.setValue(now.getValue().plus(Duration.standardHours(1)));

        update.apply("photos");
        Assert.equals("video", find.apply());

        Mockito.verify(bazingaStorage, Mockito.times(2)).addOnetimeJobX(Mockito.any(), Mockito.any());
        Assert.isFalse(jobCaptor.getValue().getTaskId().getId().endsWith(".holder"));

        now.setValue(now.getValue().minus(Duration.standardHours(1)));

        update.apply("photos");
        Assert.equals("video", find.apply());

        Assert.isFalse(jobCaptor.getValue().getTaskId().getId().endsWith(".holder"));
        Assert.equals(now.getValue().plus(DynamicVars.blockUpdateDelay.get()), jobCaptor.getValue().getScheduleTime());
    }

    @Test
    public void updateAndUpBlock() {
        ListF<String> recordIds = Cf.arrayList();

        Function0<Boolean> create = () -> recordIds.add(lentaManager.findOrCreateBlock(
                uid, contentBlockCreateData(recordIds.size() + ""), actionInfoNow()).getRecordId());

        Function1V<Integer> updateAndUp = index ->
                lentaManager.updateBlock(uid, recordIds.get(index),
                        contentBlockUpdateAndUpData(index + "").getUpdateData(), actionInfoNow());

        Function1V<Integer> delete = index -> lentaManager.deleteBlock(uid, recordIds.get(index), actionReasonNow());

        Function0<ListF<String>> load = () -> loadIndexesRecordsGroupsOrCollectionIds(db);

        Cf.repeat(create, 2);
        Assert.equals(Cf.list("1", "0"), load.apply());

        updateAndUp.apply(0);
        Assert.equals(Cf.list("0", "1"), load.apply());

        Cf.repeat(create, 3);
        Assert.equals(Cf.list("4", "3", "index_2", "2", "index_1", "0", "1"), load.apply());

        delete.apply(3);
        Assert.equals(Cf.list("4", "index_2", "2", "index_1", "0", "1"), load.apply());

        updateAndUp.apply(1);
        Assert.equals(Cf.list("1", "4", "index_2", "2", "index_1", "0"), load.apply());

        delete.apply(4);
        updateAndUp.apply(2);
        Assert.equals(Cf.list("2", "1", "index_1", "0"), load.apply());

        updateAndUp.apply(0);
        Assert.equals(Cf.list("0", "2", "index_1", "1"), load.apply());
    }

    @Test
    public void createBehind() {
        Instant now = Instant.now();

        Duration ttl = LentaManagerImpl.BLOCK_TTL;
        Duration thirdTtl = new Duration(ttl.getMillis() / 3);

        Function<Instant, String> findOrCreate =
                ts -> lentaManager.findOrCreateBlock(uid, contentBlockCreateData("group"), actionInfo(ts)).getRecordId();

        Function<Instant, LentaBlockRecord> findOrCreateAndGet =
                ts -> lentaManager.findBlock(uid, findOrCreate.apply(ts), actionInfoNow()).get();

        LentaBlockRecord right = findOrCreateAndGet.apply(now.plus(thirdTtl));
        LentaBlockRecord left = findOrCreateAndGet.apply(now.minus(ttl).minus(thirdTtl));

        Assert.notEquals(right.id, left.id);

        Assert.equals(right.id, findOrCreate.apply(right.mFrom.get().plus(thirdTtl)));
        Assert.equals(left.id, findOrCreate.apply(left.mTill.get().minus(thirdTtl)));

        LentaBlockRecord middle = findOrCreateAndGet.apply(right.mFrom.get().minus(thirdTtl));

        Assert.equals(left.mTill, middle.mFrom);
        Assert.equals(right.mFrom, middle.mTill);
    }

    @Test
    public void timeUnlimitedBlocks() {
        String recordId = lentaManager.findOrCreateBlock(uid, new LentaBlockCreateData(
                LentaRecordType.PUBLIC_RESOURCE, "group",
                rec -> CreateHandler.create(Cf.map())),
                actionInfo(Instant.now().minus(Duration.standardDays(2)))).getRecordId();

        Assert.equals(recordId, lentaManager.findAndUpdateOrCreateBlock(uid, new LentaBlockModifyData(
                LentaRecordType.PUBLIC_RESOURCE, "group",
                rec -> CreateHandler.create(Cf.map()), rec -> UpdateHandler.ignore(ActionReason.UNSPECIFIED)),
                actionInfo(Instant.now().plus(Duration.standardDays(2)))).getRecordId());
    }

    @Test
    public void pinnedBlocks() {
        Function0<Tuple2<ListF<String>, ListF<String>>> load = () -> Tuple2.tuple(
                loadRecordsGroupsOrCollectionIds(db, LookupCollections.PINNED),
                loadRecordsGroupsOrCollectionIds(db, LookupCollections.INDEXES));

        Function<LentaBlockCreateData, Option<String>> unpinIfPinned = data -> lentaManager.unpinBlockIfPinned(
                uid, data.base.type, data.base.groupKey, actionInfoNow()).map(r -> r.record.id);

        ListF<LentaBlockCreateData> datas = Cf.range(1, 4).map(gr -> contentBlockCreateData(gr + ""));

        ListF<String> blockIds = datas.map(data ->
                lentaManager.findAndPinOrCreatePinnedBlock(uid, data, actionInfoNow()).get());

        Assert.some(blockIds.first(), lentaManager.findAndPinOrCreatePinnedBlock(uid, datas.first(), actionInfoNow()));

        Assert.some(blockIds.first(), unpinIfPinned.apply(datas.first()));
        Assert.equals(Tuple2.tuple(Cf.list("3", "2"), Cf.list("1")), load.apply());

        Instant later = Instant.now().plus(Duration.standardHours(1));

        Assert.some(blockIds.first(), lentaManager.findAndPinOrCreatePinnedBlock(uid, datas.first(), actionInfo(later)));
        Assert.equals(later, lentaManager.findBlock(uid, blockIds.first(), actionInfoNow()).get().mTime);

        Assert.equals(Tuple2.tuple(Cf.list("1", "3", "2"), Cf.list()), load.apply());

        lentaManager.deleteBlocks(uid, datas.get(1).base.type, datas.get(1).base.groupKey, actionReasonNow());
        lentaManager.deleteBlock(uid, blockIds.get(2), rec -> DeleteHandler.delete(ActionReason.UNSPECIFIED), actionInfoNow());

        Assert.equals(Tuple2.tuple(Cf.list("1"), Cf.list()), load.apply());
    }

    @Test
    public void deleteEldestBlocks() {
        Function0<ListF<String>> load = () -> loadIndexesRecordsGroupsOrCollectionIds(db);

        ListF<String> recordIds = Cf.arrayList();

        Function0<Boolean> create = () -> recordIds.add(lentaManager.findOrCreateBlock(
                uid, contentBlockCreateData(recordIds.size() + ""), actionInfoNow()).getRecordId());

        Function1V<Integer> update = index ->
                lentaManager.updateBlock(uid, recordIds.get(index), contentBlockUpdateData(
                        index + "", Cf.map("rnd", DataField.string(Random2.R.nextAlnum(5)))), actionInfoNow());

        Function1V<Integer> deleteEldest = limit -> lentaManager.deleteEldestBlocks(uid, limit, actionReasonNow());

        Cf.repeat(create, 8);
        Assert.equals(Cf.list("7", "6", "index_3", "5", "4", "index_2", "3", "2", "index_1", "1", "0"), load.apply());

        deleteEldest.apply(3);
        Assert.equals(Cf.list("7", "6", "index_3", "5", "4", "index_2", "3"), load.apply());

        Cf.list(3, 5, 7).forEach(update);

        deleteEldest.apply(2);
        Assert.equals(Cf.list("7", "index_3", "5", "index_2", "3"), load.apply());

        deleteEldest.apply(3);
        Assert.equals(Cf.list(), load.apply());

        recordIds.clear();
        LentaManagerImpl.setBlocksCountLimitForTest(1);

        Cf.repeat(create, 8);
        Assert.equals(Cf.list("7", "index_7", "6", "index_6", "5", "index_5",
                "4", "index_4", "3", "index_3", "2", "index_2", "1", "index_1", "0"), load.apply());

        Cf.list(1, 3, 7).forEach(update);

        deleteEldest.apply(5);
        Assert.equals(Cf.list("7", "index_4", "3", "index_2", "1"), load.apply());
    }

    @Test
    public void createAndDeleteBlocks() {
        Function2V<ListF<String>, ListF<String>> createAndDelete =
                (createIds, deleteIds) -> lentaManager.createAndDeleteBlocks(uid,
                        createIds.map(c -> new FolderBlockData(c,
                                MpfsPath.parseDir("/" + c), MpfsResourceId.parse(uid + ":" + c), new MpfsUid(1L), 1)),
                        deleteIds, actionReasonNow());

        Function0<ListF<String>> load = () -> loadIndexesRecordsGroupsOrCollectionIds(db)
                .map(id -> Pattern2.compile("([^:]+):1$").findNthGroup(id, 1).getOrElse(id));

        LentaManagerImpl.setBlocksCountLimitForTest(3);

        createAndDelete.apply(Cf.range(0, 9).map(Object::toString), Cf.list());

        Assert.equals(Cf.list("8", "7", "6", "index_2", "5", "4", "3", "index_1", "2", "1", "0"), load.apply());

        createAndDelete.apply(Cf.list("9", "10"), Cf.list("5", "4", "3", "2", "1"));

        Assert.equals(Cf.list("10", "9", "8", "index_1", "7", "6", "0"), load.apply());

        createAndDelete.apply(Cf.list("11", "12", "13"), Cf.list("6", "7", "8"));

        Assert.equals(Cf.list("13", "12", "11", "index_1", "10", "9", "0"), load.apply());

        createAndDelete.apply(Cf.list("14"), Cf.list("0"));

        Assert.equals(Cf.list("14", "13", "12", "index_1", "11", "10", "9"), load.apply());

        createAndDelete.apply(Cf.list(), Cf.range(0, 15).map(Object::toString));


        LentaManagerImpl.setBlocksCountLimitForTest(2);

        createAndDelete.apply(Cf.list("0", "1", "2", "3", "4", "5", "6", "7"), Cf.list());

        Assert.equals(Cf.list("7", "6", "index_3", "5", "4", "index_2", "3", "2", "index_1", "1", "0"), load.apply());

        createAndDelete.apply(Cf.list("8", "9", "10", "11", "12"), Cf.list("1", "2", "3", "4", "5", "6"));

        Assert.equals(Cf.list("12", "11", "index_3", "10", "index_2", "9", "8", "index_1", "7", "0"), load.apply());
    }

    @Test
    public void createAndDeleteBlocksRecreateExisting() {
        Function2V<Tuple2List<String, String>, ListF<String>> createAndDelete =
                (createPathsFileIds, deleteIds) -> lentaManager.createAndDeleteBlocks(uid,
                        createPathsFileIds.map(pathFileId -> new FolderBlockData(pathFileId.get1(),
                                MpfsPath.parseDir("/" + pathFileId.get1()),
                                MpfsResourceId.parse(uid + ":" + pathFileId.get2()),
                                new MpfsUid(1L), 1)),
                        deleteIds, actionReasonNow());

        Function0<ListF<String>> load = () -> loadIndexesRecordsGroupsOrCollectionIds(db)
                .map(id -> Pattern2.compile("([^:]+):1$").findNthGroup(id, 1).getOrElse(id));

        createAndDelete.apply(Tuple2List.fromPairs("1", "a"), Cf.list());
        Assert.equals(Cf.list("a"), load.apply());

        createAndDelete.apply(Tuple2List.fromPairs("1", "b"), Cf.list());
        Assert.equals(Cf.list("b"), load.apply());

        createAndDelete.apply(Tuple2List.fromPairs("1", "c", "2", "d", "3", "e"), Cf.list());
        Assert.equals(Cf.list("e", "d", "index_1", "c"), load.apply());
    }

    @Test
    public void createGenericBlock() {
        GenericBlockData data = new GenericBlockData("new-year-2016",
                new I18nValue<>("ru", "en", "uk", "tr"),
                new I18nValue<>("ru-text", "en-text", "uk-text", "tr-text"),
                new I18nValue<>("icon"),
                new I18nValue<>("picture"),
                Cf.map("holiday", DataField.string("new-year")));

        lentaManager.findOrCreateBlock(uid, data.asCreateData(), actionInfoNow());

        ListF<LentaBlockRecord> records = lentaManager.findBlocks(
                uid, LentaRecordType.GENERIC_BLOCK, FieldPredicate.groupKey().eq(data.groupKey), actionInfoNow());

        Assert.hasSize(1, records);

        Assert.some("ru", GenericBlockFields.TITLE.get(records.single()).translations.getO(Language.RUSSIAN));
        Assert.some("icon", GenericBlockFields.ICON.get(records.single()).base);
        Assert.some("new-year", records.single().specific.getO("holiday").map(DataField::stringValue));
    }

    private ListF<String> loadIndexesRecordsGroupsOrCollectionIds(Database db) {
        return loadRecordsGroupsOrCollectionIds(db, LookupCollections.INDEXES);
    }

    private ListF<String> loadRecordsGroupsOrCollectionIds(Database db, LookupCollections collections) {
        return loadRecordsOrdered(db, collections)
                .map(rec -> LentaBlockRecord.Fields.GROUP_KEY.getO(rec)
                        .getOrElse(() -> LentaNextIndexRecord.Fields.COLLECTION_ID.get(rec)));
    }

    private ListF<DataRecord> loadRecordsOrdered(Database db, LookupCollections collections) {
        ListF<DataRecord> records = dataApiManager.getRecords(db.spec(), RecordsFilter.DEFAULT
                .withCollectionIdCond(collections.condition)
                .withRecordOrder(ByIdRecordOrder.COLLECTION_ID_DESC_RECORD_ID_DESC));

        return records.sorted(Comparator.<DataRecord>constEqualComparator()
                .thenComparing(rec -> rec.getCollectionId().equals("index"))
                .thenComparing(DataRecord::getCollectionId)
                .thenComparing(rec -> LentaBlockRecord.Fields.ORDER.getO(rec).getOrElse(0L))
                .reversed());
    }

    private static LentaBlockCreateData contentBlockCreateData(String groupKey) {
        return contentBlockUpdateAndUpData(groupKey).getCreateData();
    }

    private static LentaBlockCreateData contentBlockCreateData(
            String groupKey, MapF<String, DataField> specific)
    {
        return contentBlockUpdateAndUpData(groupKey, specific).getCreateData();
    }

    private static LentaBlockModifyData contentBlockUpdateAndUpData(String groupKey) {
        return contentBlockUpdateAndUpData(groupKey, Cf.map());
    }

    private static LentaBlockUpdateData contentBlockUpdateData(
            String groupKey, MapF<String, DataField> specific)
    {
        return LentaBlockUpdateData.update(LentaRecordType.CONTENT_BLOCK, groupKey, specific);
    }

    private static LentaBlockModifyData contentBlockUpdateAndUpData(
            String groupKey, MapF<String, DataField> specific)
    {
        return LentaBlockModifyData.createOrUpdateAndUp(LentaRecordType.CONTENT_BLOCK, groupKey, specific);
    }

    private static ReasonedAction actionReasonNow() {
        return actionInfoNow().withReason(ActionReason.UNSPECIFIED);
    }

    private static ActionInfo actionInfoNow() {
        return actionInfo(Instant.now());
    }

    private static ActionInfo actionInfo(Instant timestamp) {
        return ActionInfo.internal(timestamp, ActionSource.test());
    }
}
