package ru.yandex.chemodan.app.lentaloader.lenta;

import org.joda.time.DateTime;
import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.uaas.experiments.ExperimentsManager;
import ru.yandex.chemodan.bazinga.PgOnetimeUtils;
import ru.yandex.chemodan.notifier.AddLentaNotificationBlockTask;
import ru.yandex.chemodan.notifier.AddLentaNotificationBlockerTask;
import ru.yandex.chemodan.util.TimeUtils;
import ru.yandex.chemodan.util.blackbox.UserTimezoneHelper;
import ru.yandex.commune.bazinga.impl.FullJobId;
import ru.yandex.commune.bazinga.impl.OnetimeJob;
import ru.yandex.commune.bazinga.pg.storage.PgBazingaStorage;
import ru.yandex.commune.bazinga.scheduler.ActiveUidDuplicateBehavior;
import ru.yandex.commune.bazinga.scheduler.OnetimeTask;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author akirakozov
 */
public class LentaNotificationManager {
    private static final Logger logger = LoggerFactory.getLogger(LentaNotificationManager.class);

    public static final int AUTOUPLOAD_NOTIFICATION_HOUR_FROM = 16;
    public static final int AUTOUPLOAD_NOTIFICATION_HOUR_TO = 18;
    private static final String NOTIFICATION_HOURS_EXPERIMENT = "disk_lenta_notification_hours";

    private DynamicProperty<ListF<String>> notificationHours =
            DynamicProperty.cons("lenta-autosave-notification-hours",
                    Cf.list("16-18", "16-18", "16-18", "16-18", "16-18", "20-22", "20-22"));
    private DynamicProperty<ListF<String>> experimentalNotificationHours =
            DynamicProperty.cons("lenta-autosave-notification-hours-experimental",
                    Cf.list("16-18", "16-18", "16-18", "16-18", "16-18", "20-22", "20-22"));
    private DynamicProperty<Integer> sharedFolderNotificationDelayInMinutes =
            DynamicProperty.cons("lenta-shared-folder-notification-delay-in-minutes", 5);
    private DynamicProperty<Boolean> sharedFolderNotificationEnabled =
            DynamicProperty.cons("lenta-shared-folder-notification-enabled", true);

    private DynamicProperty<Integer> addLentaBlockTaskTtlMinutes =
            DynamicProperty.cons("lenta-add-lenta-block-task-ttl[minutes]", 60);

    private final PgBazingaStorage bazingaStorage;
    private final UserTimezoneHelper userTimezoneHelper;
    private final ExperimentsManager experimentsManager;
    private final boolean createWithoutDelay;
    private final boolean notificationsDisabled;

    public LentaNotificationManager(
            PgBazingaStorage bazingaStorage, UserTimezoneHelper userTimezoneHelper,
            ExperimentsManager experimentsManager,
            boolean createWithoutDelay, boolean notificationsEnabled)
    {
        this.bazingaStorage = bazingaStorage;
        this.createWithoutDelay = createWithoutDelay;
        this.userTimezoneHelper = userTimezoneHelper;
        this.experimentsManager = experimentsManager;
        this.notificationsDisabled = !notificationsEnabled;
    }

    public void scheduleAutouploadNotification(DataApiUserId uid, String blockId) {
        if (disabled()) {
            return;
        }
        logger.debug("Schedule autoupload notificaiton for {} and block {}", uid, blockId);

        DateTime dateTime = DateTime.now(userTimezoneHelper.getUserTimezone(uid.toPassportUid()));
        int dayOfWeek = dateTime.plusDays(1).dayOfWeek().get();
        int[] hours = getAutoSaveNotificationHoursRange(dayOfWeek, uid);
        Instant scheduleTime = createWithoutDelay ?
                Instant.now() :
                TimeUtils.randomTimeOnNextDay(dateTime, hours[0], hours[1]);
        Instant deadline = getDeadline(scheduleTime);
        logger.debug("ScheduleTime: {} for {} and block {}", scheduleTime, uid, blockId);
        scheduleTask(new AddLentaNotificationBlockTask(uid, blockId, deadline), scheduleTime);
    }

    int[] getAutoSaveNotificationHoursRange(int dayOfWeek, DataApiUserId uid) {
        try {
            if (experimentsManager.getFlags(uid.toPassportUid().toUidOrZero().getUid()).containsTs(NOTIFICATION_HOURS_EXPERIMENT)) {
                logger.debug("User is in autoupload notification experiment uid={}", uid);
                return Cf.list(StringUtils.split(experimentalNotificationHours.get().get(dayOfWeek - 1), "-"))
                        .mapToIntArray(Integer::parseInt);
            }
            logger.debug("User is not in autoupload notification experiment uid={}", uid);
            return Cf.list(StringUtils.split(notificationHours.get().get(dayOfWeek - 1), "-"))
                    .mapToIntArray(Integer::parseInt);
        } catch (Exception e) {
            logger.warn("Incorrect format: " + e.getMessage(), e);
            return new int[]{AUTOUPLOAD_NOTIFICATION_HOUR_FROM, AUTOUPLOAD_NOTIFICATION_HOUR_TO};
        }
    }

    public void scheduleReminderBlockNotification(DataApiUserId uid, String blockId, boolean sendPushNow) {
        if (disabled()) {
            return;
        }
        logger.debug("Schedule reminder notificaiton for {} and block {}", uid, blockId);

        Instant scheduleTime = Instant.now();
        Instant deadline = getDeadline(scheduleTime);
        scheduleTask(new AddLentaNotificationBlockTask(uid, blockId, sendPushNow, deadline), scheduleTime);
    }

    public void scheduleSharedFolderBlockNotification(DataApiUserId uid, String blockId) {
        if (disabled() || !sharedFolderNotificationEnabled.get()) {
            return;
        }
        logger.debug("Schedule shared folder notificaiton for {} and block {}", uid, blockId);

        AddLentaNotificationBlockerTask blockerTask = new AddLentaNotificationBlockerTask(uid, blockId);

        String blockerActiveUid = PgOnetimeUtils.getActiveUniqueIdentifier(blockerTask);
        Option<OnetimeJob> activeBlocker = bazingaStorage.findOnetimeJobByActiveUid(blockerActiveUid);

        Duration delay = Duration.standardMinutes(sharedFolderNotificationDelayInMinutes.get());
        Instant nowPlusDelay = Instant.now().plus(delay);

        if (activeBlocker.isPresent()) {
            Instant blockerScheduleTime = activeBlocker.get().getScheduleTime();

            if (blockerScheduleTime.isAfter(nowPlusDelay)) {
                scheduleTask(uid, blockId, blockerScheduleTime.minus(delay));
            } else {
                scheduleTask(uid, blockId, blockerScheduleTime);
                bazingaStorage.saveOnetimeJob(
                        activeBlocker.get().withScheduleTime(blockerScheduleTime.plus(delay)),
                        ActiveUidDuplicateBehavior.MERGE
                );
            }
        } else {
            scheduleTask(uid, blockId, Instant.now());
            scheduleTask(blockerTask, nowPlusDelay);
        }
    }

    private FullJobId scheduleTask(OnetimeTask task, Instant scheduleTime) {
        OnetimeJob job = PgOnetimeUtils.makeJob(task, scheduleTime);
        return bazingaStorage.addOnetimeJob(job, ActiveUidDuplicateBehavior.DO_NOTHING);
    }

    private FullJobId scheduleTask(DataApiUserId uid, String blockId, Instant scheduleTime) {
        Instant deadline = getDeadline(scheduleTime);
        AddLentaNotificationBlockTask addNewNotificationTask =
                new AddLentaNotificationBlockTask(uid, blockId, deadline);
        return scheduleTask(addNewNotificationTask, scheduleTime);
    }

    private boolean disabled() {
        if (notificationsDisabled) {
            logger.debug("Creation notifications from lenta is disabled");
        }
        return notificationsDisabled;
    }

    private Instant getDeadline(Instant scheduleTime) {
        return scheduleTime.plus(Duration.standardMinutes(addLentaBlockTaskTtlMinutes.get()));
    }
}
