package ru.yandex.chemodan.app.lentaloader.lenta;

import org.joda.time.Duration;
import org.joda.time.Instant;
import org.junit.Before;
import org.junit.Test;
import org.mockito.ArgumentCaptor;
import org.mockito.Mock;
import org.mockito.Mockito;
import org.mockito.MockitoAnnotations;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiPassportUserId;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.uaas.experiments.ExperimentsManager;
import ru.yandex.chemodan.bazinga.PgOnetimeUtils;
import ru.yandex.chemodan.http.YandexCloudRequestIdHolder;
import ru.yandex.chemodan.notifier.AddLentaNotificationBlockerTask;
import ru.yandex.chemodan.util.blackbox.UserTimezoneHelper;
import ru.yandex.commune.bazinga.impl.OnetimeJob;
import ru.yandex.commune.bazinga.pg.storage.PgBazingaStorage;
import ru.yandex.inside.passport.blackbox2.Blackbox2;
import ru.yandex.misc.test.Assert;
import ru.yandex.misc.test.TestBase;

import static org.mockito.Matchers.any;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

/**
 * @author akirakozov
 */
public class LentaNotificationManagerTest extends TestBase {
    private LentaNotificationManager lentaNotificationManager;

    @Mock
    private PgBazingaStorage bazingaStorage;
    @Mock
    private Blackbox2 blackbox2;
    @Mock
    private ExperimentsManager experimentsManager;

    private DataApiUserId uid = new DataApiPassportUserId(123L);

    @Before
    public void init() {
        MockitoAnnotations.initMocks(this);
        Mockito.when(experimentsManager.getFlags(uid.toPassportUid().getUid())).thenReturn(Cf.list());

        lentaNotificationManager = new LentaNotificationManager(
                bazingaStorage, new UserTimezoneHelper(blackbox2), experimentsManager, false, true);
    }

    @Test
    public void getAutoSaveNotificationHoursRange() {
        LentaNotificationManager lnm = new LentaNotificationManager(null, null, experimentsManager, true, true);
        Assert.arraysEquals(new int[]{16, 18}, lnm.getAutoSaveNotificationHoursRange(3, uid));
        Assert.arraysEquals(new int[]{20, 22}, lnm.getAutoSaveNotificationHoursRange(6, uid));
    }

    @Test
    public void scheduleFirstSharedFolderBlockNotification() {
        DataApiUserId uid = new DataApiPassportUserId(123L);
        String blockId = "block-id_345";
        YandexCloudRequestIdHolder.set("ycrid-some-id123");

        when(bazingaStorage.findOnetimeJobByActiveUid(any())).thenReturn(Option.empty());
        lentaNotificationManager.scheduleSharedFolderBlockNotification(uid, blockId);

        ArgumentCaptor<OnetimeJob> args = ArgumentCaptor.forClass(OnetimeJob.class);
        verify(bazingaStorage, times(2)).addOnetimeJob(args.capture(), any());

        Assert.hasSize(2, args.getAllValues());
        Assert.equals("addLentaNotificationBlock", args.getAllValues().get(0).getId().getTaskId().getId());
        Assert.equals("{\"ycrid\":\"ycrid-some-id123\",\"uid\":\"123\",\"blockId\":\"block-id_345\",\"deadline\":0}",
                suppressDeadline(args.getAllValues().get(0).getParameters()));
        Assert.some("addLentaNotificationBlock_{\"uid\":\"123\",\"blockId\":\"block-id_345\"}",
                args.getAllValues().get(0).getActiveUniqueIdentifier());
        Assert.equals("addLentaNotificationBlocker", args.getAllValues().get(1).getId().getTaskId().getId());
        Assert.equals("{\"ycrid\":\"ycrid-some-id123\",\"uid\":\"123\",\"blockId\":\"block-id_345\"}",
                args.getAllValues().get(1).getParameters());
        Assert.some("addLentaNotificationBlocker_{\"uid\":\"123\",\"blockId\":\"block-id_345\"}",
                args.getAllValues().get(1).getActiveUniqueIdentifier());
    }

    @Test
    public void scheduleFirstSharedFolderBlockNotificationWhenExists() {
        DataApiUserId uid = new DataApiPassportUserId(123L);
        String blockId = "block-id_345";
        YandexCloudRequestIdHolder.set("ycrid-some-id");

        AddLentaNotificationBlockerTask blockerTask = new AddLentaNotificationBlockerTask(uid, blockId);

        String blockerActiveUid ="addLentaNotificationBlocker_{\"uid\":\"123\",\"blockId\":\"block-id_345\"}";
        OnetimeJob existedJob = PgOnetimeUtils.makeJob(blockerTask, Instant.now());

        when(bazingaStorage.findOnetimeJobByActiveUid(blockerActiveUid)).thenReturn(Option.of(existedJob));
        lentaNotificationManager.scheduleSharedFolderBlockNotification(uid, blockId);

        ArgumentCaptor<OnetimeJob> args = ArgumentCaptor.forClass(OnetimeJob.class);
        verify(bazingaStorage, times(1)).addOnetimeJob(args.capture(), any());
        verify(bazingaStorage, times(1)).saveOnetimeJob(args.capture(), any());

        Assert.hasSize(2, args.getAllValues());
        Assert.equals("addLentaNotificationBlock", args.getAllValues().get(0).getId().getTaskId().getId());
        Assert.equals("{\"ycrid\":\"ycrid-some-id\",\"uid\":\"123\",\"blockId\":\"block-id_345\",\"deadline\":0}",
                suppressDeadline(args.getAllValues().get(0).getParameters()));
        Assert.equals("addLentaNotificationBlocker", args.getAllValues().get(1).getId().getTaskId().getId());
        Assert.equals("{\"ycrid\":\"ycrid-some-id\",\"uid\":\"123\",\"blockId\":\"block-id_345\"}",
                args.getAllValues().get(1).getParameters());
        Assert.equals(existedJob.getScheduleTime().plus(Duration.standardMinutes(5)),
                args.getAllValues().get(1).getScheduleTime());
    }

    private String suppressDeadline(String parameters) {
        return parameters.replaceFirst("\"deadline\":[0-9]*", "\"deadline\":0");

    }

}
