package ru.yandex.chemodan.app.lentaloader.lenta.limit;

import java.util.Comparator;
import java.util.TreeMap;
import java.util.TreeSet;

import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.CollectionF;
import ru.yandex.bolts.collection.Either;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function0;
import ru.yandex.bolts.function.Function2B;
import ru.yandex.chemodan.app.dataapi.api.data.record.SimpleRecordId;
import ru.yandex.chemodan.eventlog.events.MpfsPath;
import ru.yandex.chemodan.mpfs.MpfsResourceId;
import ru.yandex.chemodan.mpfs.MpfsUid;
import ru.yandex.misc.lang.DefaultObject;

/**
 * @author dbrylev
 */
public class ContentMerger {
    public final Function<MpfsPath, Option<MpfsResourceId>> resourceIdProvider;
    public final Function2B<MpfsPath, Integer> limitExceededChecker;

    public ContentMerger(
            Function<MpfsPath, Option<MpfsResourceId>> resourceIdProvider,
            Function2B<MpfsPath, Integer> limitExceededChecker)
    {
        this.resourceIdProvider = resourceIdProvider;
        this.limitExceededChecker = limitExceededChecker;
    }

    public Result merge(ListF<ContentCreature> creatures) {
        TreeMap<UidPath, ListF<ContentCreatureOrData>> creaturesByUidPath =
                new TreeMap<>(Comparator.comparing(UidPath::getValue));

        TreeSet<UidPath> uidPathsDesc = new TreeSet<>(Comparator.comparing(UidPath::getValue).reversed());

        creatures.forEach(c -> {
            uidPathsDesc.add(new UidPath(c.modifierUid, c.path));
            uidPathsDesc.addAll(c.path.getParents().map(p -> new UidPath(c.modifierUid, p)));

            creaturesByUidPath.computeIfAbsent(new UidPath(c.modifierUid, c.path), p -> Cf.arrayList())
                    .add(ContentCreatureOrData.creature(c));
        });

        SetF<String> blockIdsToDelete = Cf.hashSet();
        SetF<SimpleRecordId> creatureIdsToDelete = Cf.hashSet();

        Tuple2List<SimpleRecordId, Integer> blocksCountToUpdate = Tuple2List.arrayList();

        uidPathsDesc.forEach(uidPath -> {
            UidPath nextPath = uidPath.withPath(MpfsPath.parseDir(uidPath.path.value.replaceAll("/$", "0")));

            CollectionF<ListF<ContentCreatureOrData>> pathsCreatures =
                    Cf.x(creaturesByUidPath.subMap(uidPath, true, nextPath, false).values());

            ListF<ContentCreatureOrData> descendants = pathsCreatures.flatMap(pathCreatures -> {
                Tuple2<ListF<ContentCreatureOrData>, ListF<ContentCreatureOrData>> contentAndNot =
                        pathCreatures.partition(ContentCreatureOrData::isContentCreature);

                ListF<ContentCreatureOrData> content = contentAndNot.get1()
                        .groupBy(c -> c.getCreature().mediaType).values().map(contents -> {
                            contents = contents.sortedByDesc(ContentCreatureOrData::cTime);

                            contents.drop(1).forEach(c -> {
                                blockIdsToDelete.add(c.blockId());
                                creatureIdsToDelete.addAll(c.recordId());
                            });

                            return contents.first();
                        });
                return content.plus(contentAndNot.get2());
            });

            Option<ContentCreatureOrData> currentFolder = creaturesByUidPath.getOrDefault(uidPath, Cf.list())
                    .filter(ContentCreatureOrData::isFolderCreature)
                    .minByO(ContentCreatureOrData::cTime);

            int blocksCount = descendants.map(ContentCreatureOrData::blocksCount).sum(Cf.Integer);

            Function0<Option<MpfsResourceId>> resourceId = resourceIdProvider.bind(uidPath.path).memoize();

            boolean limitExceeded = !(descendants.size() == 1 && !descendants.single().isContentCreature())
                    && limitExceededChecker.apply(uidPath.path, blocksCount);

            if (currentFolder.isPresent() || limitExceeded && resourceId.apply().isPresent()) {
                descendants.filterNot(currentFolder::isSome).forEach(c -> {
                    blockIdsToDelete.add(c.blockId());
                    creatureIdsToDelete.addAll(c.recordId());

                    creaturesByUidPath.remove(uidPath.withPath(c.path()));
                });

                creaturesByUidPath.put(uidPath, currentFolder.map(
                        c -> ContentCreatureOrData.creature(c.getCreature().withBlocksCount(blocksCount))));
            }
            if (!currentFolder.isPresent() && limitExceeded && resourceId.apply().isPresent()) {
                String minId = descendants.minBy(ContentCreatureOrData::cTime).blockId();

                FolderBlockData data = new FolderBlockData(
                        minId + "X", uidPath.path, resourceId.apply().get(), uidPath.uid, blocksCount);

                creaturesByUidPath.put(uidPath, Cf.list(ContentCreatureOrData.data(data)));
            }

            if (currentFolder.isPresent() && !currentFolder.get().getCreature().blocksCount.isSome(blocksCount)) {
                blocksCountToUpdate.add(currentFolder.get().getCreature().getRecordId(), blocksCount);
            }
        });

        ListF<FolderBlockData> dataForCreate = Cf.x(creaturesByUidPath.values())
                .flatMap(cs -> cs.filterMap(ContentCreatureOrData::getDataO));

        return new Result(
                blockIdsToDelete.toList(), creatureIdsToDelete.toList(), dataForCreate,
                blocksCountToUpdate.filterBy1Not(creatureIdsToDelete::containsTs));
    }

    public static class ContentCreatureOrData {
        public final Either<ContentCreature, FolderBlockData> either;

        public ContentCreatureOrData(
                Either<ContentCreature, FolderBlockData> either)
        {
            this.either = either;
        }

        public static ContentCreatureOrData creature(ContentCreature c) {
            return new ContentCreatureOrData(Either.left(c));
        }

        public static ContentCreatureOrData data(FolderBlockData c) {
            return new ContentCreatureOrData(Either.right(c));
        }

        public final Instant cTime() {
            return either.fold(c -> c.cTime, c -> Instant.now());
        }

        public final MpfsPath path() {
            return either.fold(c -> c.path, c -> c.path);
        }

        public final String blockId() {
            return either.fold(c -> c.blockId, c -> c.blockId);
        }

        public final boolean isFolderCreature() {
            return either.leftO().exists(ContentCreature::isFolder);
        }

        public final boolean isContentCreature() {
            return either.leftO().exists(c -> !c.isFolder());
        }

        public final ContentCreature getCreature() {
            return either.getLeft();
        }

        public final Option<FolderBlockData> getDataO() {
            return either.rightO();
        }

        public final Option<SimpleRecordId> recordId() {
            return either.leftO().map(ContentCreature::getRecordId);
        }

        public final int blocksCount() {
            return either.fold(c -> c.blocksCount.getOrElse(1), c -> c.blocksCount);
        }
    }

    public static class UidPath extends DefaultObject {
        public final MpfsUid uid;
        public final MpfsPath path;

        public transient final String value;

        public UidPath(MpfsUid uid, MpfsPath path) {
            this.uid = uid;
            this.path = path;
            this.value = uid.getRawValue() + ":" + path.getValue();
        }

        public UidPath withPath(MpfsPath path) {
            return new UidPath(uid, path);
        }

        public String getValue() {
            return value;
        }
    }

    public static class Result {
        public final ListF<String> blockIdsToDelete;
        public final ListF<SimpleRecordId> creatureIdsToDelete;
        public final ListF<FolderBlockData> dataToCreate;
        public final Tuple2List<SimpleRecordId, Integer> blocksCountsToUpdate;

        public Result(
                ListF<String> blockIdsToDelete,
                ListF<SimpleRecordId> creatureIdsToDelete,
                ListF<FolderBlockData> dataToCreate,
                Tuple2List<SimpleRecordId, Integer> blocksCountsToUpdate)
        {
            this.blockIdsToDelete = blockIdsToDelete;
            this.creatureIdsToDelete = creatureIdsToDelete;
            this.dataToCreate = dataToCreate;
            this.blocksCountsToUpdate = blocksCountsToUpdate;
        }
    }
}
