package ru.yandex.chemodan.app.lentaloader.lenta.limit;

import java.util.concurrent.atomic.AtomicLong;

import org.joda.time.Instant;
import org.junit.Test;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.function.Function2;
import ru.yandex.chemodan.app.dataapi.api.data.record.SimpleRecordId;
import ru.yandex.chemodan.eventlog.events.MpfsPath;
import ru.yandex.chemodan.mpfs.MpfsResourceId;
import ru.yandex.chemodan.mpfs.MpfsUid;
import ru.yandex.misc.test.Assert;

/**
 * @author dbrylev
 */
public class ContentMergerTest {

    @Test
    public void merge() {
        Instant start = Instant.now();
        AtomicLong counter = new AtomicLong();

        Function2<String, String, ContentCreature> consCreature = (pathMediaType, collectionId) -> {
            ListF<String> split = Cf.x(pathMediaType.split(":"));

            return new ContentCreature(
                    pathMediaType, collectionId, MpfsPath.parseDir(split.first()),
                    Option.when(split.size() > 1, split::last),
                    new MpfsUid(1L), Option.empty(), start.plus(counter.getAndAdd(1)));
        };

        ContentMerger merger = new ContentMerger(
                (path) -> Option.of(MpfsResourceId.parse("1:0")),
                (path, size) -> size > 1);

        Function2<ListF<String>, ListF<String>, ContentMerger.Result> merge = (contentPaths, folderPaths) ->
                merger.merge(Cf.<ContentCreature>list()
                        .plus(contentPaths.map(consCreature.bind2(LentaLimitManagerImpl.CONTENT_BLOCKS)))
                        .plus(folderPaths.map(consCreature.bind2(LentaLimitManagerImpl.FOLDER_BLOCKS))));

        ContentMerger.Result result = merge.apply(Cf.list("/d/1:image", "/d/1:video", "/d/1:document"), Cf.list());

        Assert.equals("/d/1/", result.dataToCreate.single().path.value);
        Assert.equals("/d/1:imageX", result.dataToCreate.single().blockId);

        Assert.equals(Cf.repeat("/d/1", 3), result.blockIdsToDelete.map(s -> s.split(":")[0]));
        Assert.equals(Cf.repeat("/d/1", 3), result.creatureIdsToDelete.map(c -> c.recordId).map(s -> s.split(":")[0]));

        result = merge.apply(Cf.list("/d/1", "/d/1/2", "/d/1/2/3"), Cf.list());

        Assert.equals("/d/1/", result.dataToCreate.single().path.value);

        Assert.equals(Cf.list("/d/1", "/d/1/2", "/d/1/2/3"), result.creatureIdsToDelete.map(c -> c.recordId).sorted());
        Assert.equals(Cf.list("/d/1", "/d/1/2", "/d/1/2/3", "/d/1/2X"), result.blockIdsToDelete.sorted());

        result = merge.apply(Cf.list("/d/1/2/3"), Cf.list("/d/1", "/d/1/2", "/d/1/2/3"));

        Assert.isEmpty(result.dataToCreate);
        Assert.equals(Cf.list("/d/1/2", "/d/1/2/3"), result.blockIdsToDelete.sorted());

        ListF<String> paths = Cf.range(1, 3).flatMap(i -> Cf.range(3, 5).flatMap(j -> Cf.range(5, 7)
                .map(k -> Cf.list(i, j, k).mkString("/d/", "/", ""))));

        result = merge.apply(paths, Cf.list());
        Assert.equals("/d/", result.dataToCreate.single().path.value);

        Assert.equals(paths, result.creatureIdsToDelete.map(c -> c.recordId).sorted());

        Assert.equals(paths.plus(Cf.list(
                "/d/1/3/5X", "/d/1/4/5X", "/d/1/4/5XX",
                "/d/2/3/5X", "/d/2/4/5X", "/d/2/4/5XX")).sorted(), result.blockIdsToDelete.sorted());
    }

    @Test
    public void modifierUid() {
        Function2<String, Integer, ContentCreature> consCreature = (path, uid) -> new ContentCreature(
                path, LentaLimitManagerImpl.CONTENT_BLOCKS, MpfsPath.parseDir(path),
                Option.empty(), new MpfsUid((long) uid), Option.empty(), Instant.now());

        ContentMerger merger = new ContentMerger(
                (path) -> Option.of(MpfsResourceId.parse("1:0")),
                (path, size) -> size > 1);

        ContentMerger.Result result = merger.merge(Tuple2List.<String, Integer>fromPairs(
                "/disk/shared/arcadia/src", 1,
                "/disk/shared/arcadia/out", 1,
                "/disk/shared/arcadia/target", 1,
                "/disk/shared/arcadia/target", 2,
                "/disk/shared/iceberg/src", 1,
                "/disk/shared/iceberg/out", 4).map(consCreature));

        Assert.equals(new MpfsUid(1L), result.dataToCreate.single().modifierUid);
        Assert.equals("/disk/shared/", result.dataToCreate.single().path.value);
    }

    @Test
    public void updateCounts() {
        Function2<String, Option<Integer>, ContentCreature> consCreature = (path, count) -> new ContentCreature(
                path, count.isPresent() ? LentaLimitManagerImpl.FOLDER_BLOCKS : LentaLimitManagerImpl.CONTENT_BLOCKS,
                MpfsPath.parseDir(path), Option.empty(), new MpfsUid(1L), count, Instant.now());

        ContentMerger merger = new ContentMerger(
                (path) -> Option.of(MpfsResourceId.parse("1:0")),
                (path, size) -> size > 2);

        ContentMerger.Result result = merger.merge(Cf.list(
                consCreature.apply("/blocked", Option.of(3)),
                consCreature.apply("/blocked/addition", Option.empty()),
                consCreature.apply("/free", Option.of(1)),
                consCreature.apply("/free/addition", Option.empty())));

        Assert.equals(Tuple2List.fromPairs("/free", 2, "/blocked", 4),
                result.blocksCountsToUpdate.map1(SimpleRecordId::recordId));

        Assert.equals(Cf.list("/free/addition", "/blocked/addition"), result.blockIdsToDelete);
    }
}
