package ru.yandex.chemodan.app.lentaloader.lenta.limit;

import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function0;
import ru.yandex.bolts.function.Function2;
import ru.yandex.bolts.function.Function2B;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataField;
import ru.yandex.chemodan.app.dataapi.api.data.filter.RecordsFilter;
import ru.yandex.chemodan.app.dataapi.api.data.filter.condition.DataCondition;
import ru.yandex.chemodan.app.dataapi.api.db.Database;
import ru.yandex.chemodan.app.dataapi.api.db.ref.AppDatabaseRef;
import ru.yandex.chemodan.app.dataapi.api.db.ref.UserDatabaseSpec;
import ru.yandex.chemodan.app.dataapi.api.deltas.Delta;
import ru.yandex.chemodan.app.dataapi.api.deltas.FieldChange;
import ru.yandex.chemodan.app.dataapi.api.deltas.RecordChange;
import ru.yandex.chemodan.app.dataapi.api.deltas.RevisionCheckMode;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.dataapi.core.manager.DataApiManager;
import ru.yandex.chemodan.app.lentaloader.DynamicVars;
import ru.yandex.chemodan.app.lentaloader.blocks.ModifiedResource;
import ru.yandex.chemodan.app.lentaloader.lenta.LentaManager;
import ru.yandex.chemodan.app.lentaloader.lenta.LentaRecordType;
import ru.yandex.chemodan.app.lentaloader.log.ActionInfo;
import ru.yandex.chemodan.app.lentaloader.log.ActionReason;
import ru.yandex.chemodan.app.lentaloader.worker.tasks.CleanupLentaBlockCreaturesTask;
import ru.yandex.chemodan.app.lentaloader.worker.tasks.MergeContentLentaBlocksTask;
import ru.yandex.chemodan.cache.MeteredCache;
import ru.yandex.chemodan.eventlog.events.MpfsPath;
import ru.yandex.chemodan.mpfs.MpfsClient;
import ru.yandex.chemodan.mpfs.MpfsFileInfo;
import ru.yandex.chemodan.mpfs.MpfsResourceId;
import ru.yandex.chemodan.mpfs.MpfsUid;
import ru.yandex.chemodan.util.postgres.PgSqlQueryUtils;
import ru.yandex.commune.bazinga.pg.PgBazingaTaskManager;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.db.q.SqlLimits;

/**
 * @author dbrylev
 */
public class LentaLimitManagerImpl implements LentaLimitManager {
    public static final AppDatabaseRef CREATURES_DB = new AppDatabaseRef("lenta", "block_creatures");

    public static final String CONTENT_BLOCKS = "content_blocks";
    public static final String FOLDER_BLOCKS = "folder_blocks";

    private final LentaManager lentaManager;
    private final DataApiManager dataApiManager;

    private final PgBazingaTaskManager bazingaTaskManager;

    private final MeteredCache<UserContentPath, String> foldersCache;

    private final Function0<Function2B<MpfsPath, Integer>> limitCheckerProvider;
    private final Function2<DataApiUserId, MpfsPath, Option<MpfsResourceId>> resourceIdProvider;

    public LentaLimitManagerImpl(
            LentaManager lentaManager, DataApiManager dataApiManager,
            PgBazingaTaskManager bazingaTaskManager, MpfsClient mpfsClient,
            MeteredCache<UserContentPath, String> foldersCache)
    {
        this(lentaManager, dataApiManager, bazingaTaskManager,
                () -> pathLevelLimitExceededF(DynamicVars.contentPathLevelLimits.get()),
                (uid, path) -> getResourceId(mpfsClient, uid, path),
                foldersCache);
    }

    public LentaLimitManagerImpl(
            LentaManager lentaManager, DataApiManager dataApiManager, PgBazingaTaskManager bazingaTaskManager,
            Function0<Function2B<MpfsPath, Integer>> limitCheckerProvider,
            Function2<DataApiUserId, MpfsPath, Option<MpfsResourceId>> resourceIdProvider,
            MeteredCache<UserContentPath, String> foldersCache)
    {
        this.lentaManager = lentaManager;
        this.dataApiManager = dataApiManager;
        this.bazingaTaskManager = bazingaTaskManager;
        this.limitCheckerProvider = limitCheckerProvider;
        this.resourceIdProvider = resourceIdProvider;
        this.foldersCache = foldersCache;
    }

    @Override
    public boolean handleContentPathBlocking(DataApiUserId uid, ModifiedResource resource, ActionInfo actionInfo) {
        MpfsPath path = escapedDirectory(resource.address.path);

        ListF<UserContentPath> paths = Option.of(path).plus(path.getParents())
                .map(p -> new UserContentPath(uid, p, resource.modifier));

        Option<String> blockId = paths.filterMap(foldersCache::getO).lastO();

        if (blockId.isPresent()) {
            foldersCache.incHits();

            lentaManager.updateAndUpOrDeleteBlockDelayed(uid, blockId.get(), LentaRecordType.FOLDER_BLOCK, actionInfo);

            return true;
        }

        ListF<ContentCreature> creatures = getCreaturesDatabaseO(uid)
                .flatMap(db -> findActualContentCreatures(db, Option.of(resource.modifier), FOLDER_BLOCKS));

        creatures.forEach(c -> foldersCache.put(c.toUserContentPath(uid), c.blockId));

        blockId = paths.filterMap(creatures.toMapMappingToKey(c -> c.toUserContentPath(uid))::getO).lastO()
                .map(c -> c.blockId);

        if (blockId.isPresent()) {
            lentaManager.updateAndUpOrDeleteBlockDelayed(uid, blockId.get(), LentaRecordType.FOLDER_BLOCK, actionInfo);

            return true;
        }

        return false;
    }

    @Override
    public void insertContentBlockCreature(
            DataApiUserId uid, String blockId, ModifiedResource resource, ActionInfo actionInfo)
    {
        MpfsPath path = escapedDirectory(resource.address.path);
        Instant now = Instant.now();

        scheduleMaintenanceTasks(uid);

        Database db = dataApiManager.getOrCreateDatabase(consCreaturesDatabaseSpec(uid));

        ContentCreature creature = new ContentCreature(
                blockId, CONTENT_BLOCKS, path,
                Option.of(resource.mediaType()), resource.modifier, Option.empty(), now);

        db = db.withNowaitLock(DynamicVars.nowaitLockForCreatureInsert.get());

        dataApiManager.applyDelta(db, RevisionCheckMode.PER_RECORD,
                new Delta(RecordChange.insert(CONTENT_BLOCKS, blockId, creature.toData())));
    }

    @Override
    public void mergeContentBlocks(DataApiUserId uid, ActionInfo actionInfo) {
        Option<Database> dbO = getCreaturesDatabaseO(uid);

        if (!dbO.isPresent()) return;

        ContentMerger merger = new ContentMerger(
                path -> resourceIdProvider.apply(uid, path), limitCheckerProvider.apply());

        ContentMerger.Result merge = merger.merge(
                findActualContentCreatures(dbO.get(), Option.empty(), CONTENT_BLOCKS, FOLDER_BLOCKS));

        lentaManager.createAndDeleteBlocks(uid, merge.dataToCreate, merge.blockIdsToDelete,
                actionInfo.withReason(ActionReason.CONTENT_PATH_MERGING));

        Instant now = Instant.now();

        ListF<ContentCreature> creatures = merge.dataToCreate.map(data -> new ContentCreature(
                data.blockId, FOLDER_BLOCKS, data.path,
                Option.empty(), data.modifierUid, Option.of(data.blocksCount), now));

        ListF<RecordChange> changes = creatures.map(c -> RecordChange.insert(c.getRecordId(), c.toData()));

        changes = changes.plus(merge.creatureIdsToDelete.map(RecordChange::delete));

        changes = changes.plus(merge.blocksCountsToUpdate.map(data -> RecordChange.update(data.get1(),
                Cf.list(FieldChange.put(ContentCreature.Fields.BLOCKS_COUNT.name, DataField.integer(data.get2()))))));

        if (changes.isNotEmpty()) {
            dbO = dbO.map(db -> db.withNowaitLock(DynamicVars.nowaitLockForCreatureMerge.get()));

            dataApiManager.applyDelta(dbO.get(), RevisionCheckMode.PER_RECORD, new Delta(changes));
        }

        if (merge.dataToCreate.isNotEmpty()) {
            scheduleMaintenanceTasks(uid);
        }
    }

    @Override
    public void cleanupCreatures(DataApiUserId uid, ActionInfo actionInfo) {
        Option<Database> dbO = getCreaturesDatabaseO(uid);

        if (!dbO.isPresent()) return;

        ListF<RecordChange> changes = findExpiredContentCreatures(dbO.get())
                .map(c -> RecordChange.delete(c.getRecordId()));

        if (changes.isNotEmpty()) {
            dataApiManager.applyDelta(dbO.get(), RevisionCheckMode.PER_RECORD, new Delta(changes));
        }

        findLatestContentCreature(dbO.get()).forEach(last -> bazingaTaskManager.schedule(
                new CleanupLentaBlockCreaturesTask(uid), last.cTime.plus(DynamicVars.contentCreatureTtl.get())));
    }

    public ListF<ContentCreature> findActualContentCreatures(Database db) {
        return findContentCreatures(db, true, Option.empty(), CONTENT_BLOCKS, FOLDER_BLOCKS);
    }

    private void scheduleMaintenanceTasks(DataApiUserId uid) {
        Instant now = Instant.now();

        bazingaTaskManager.schedule(new MergeContentLentaBlocksTask(uid),
                now.plus(DynamicVars.contentMergeDelay.get()));

        bazingaTaskManager.schedule(new CleanupLentaBlockCreaturesTask(uid),
                now.plus(DynamicVars.contentCreatureTtl.get()));
    }

    private Option<Database> getCreaturesDatabaseO(DataApiUserId uid) {
        return dataApiManager.getDatabaseO(consCreaturesDatabaseSpec(uid));
    }

    private UserDatabaseSpec consCreaturesDatabaseSpec(DataApiUserId uid) {
        return new UserDatabaseSpec(uid, CREATURES_DB);
    }

    public static Function2B<MpfsPath, Integer> pathLevelLimitExceededF(ListF<Integer> limits) {
        return (path, size) -> limits.getO(path.getLevel()).orElse(limits.lastO()).exists(lim -> lim > 0 && size > lim);
    }

    private static Option<MpfsResourceId> getResourceId(MpfsClient mpfsClient, DataApiUserId uid, MpfsPath path) {
        PassportUid passUid = uid.toPassportUid();

        Option<MpfsFileInfo> file = mpfsClient.getFileInfoOByUidAndPath(uid.forMpfs(), path.value, Cf.list("resource_id"));

        return file.flatMapO(f -> f.getMeta().getResourceId())
                .orElse(Option.when(path.isRoot(), new MpfsResourceId(new MpfsUid(passUid), path.value)));
    }

    private ListF<ContentCreature> findActualContentCreatures(
            Database db, Option<MpfsUid> modifier, String... collectionIds)
    {
        return findContentCreatures(db, true, modifier, collectionIds);
    }

    private ListF<ContentCreature> findExpiredContentCreatures(Database db) {
        return findContentCreatures(db, false, Option.empty(), CONTENT_BLOCKS, FOLDER_BLOCKS);
    }

    private ListF<ContentCreature> findContentCreatures(
            Database db, boolean unexpired, Option<MpfsUid> modifier, String... collectionIds)
    {
        Instant bornline = Instant.now().minus(DynamicVars.contentCreatureTtl.get());

        DataCondition dataCond = unexpired
                ? ContentCreature.Fields.CTIME.column().ge(bornline)
                : ContentCreature.Fields.CTIME.column().lt(bornline);

        dataCond = dataCond.and(
                modifier.map(MpfsUid::getRawValue).map(ContentCreature.Fields.MODIFIER_UID.column()::eq));

        RecordsFilter filter = RecordsFilter.DEFAULT.withCollectionIds(Cf.x(collectionIds)).withDataCond(dataCond);

        return dataApiManager.getRecords(db.spec(), filter).map(ContentCreature::fromDataRecord);
    }

    private Option<ContentCreature> findLatestContentCreature(Database db) {
        RecordsFilter filter = RecordsFilter.DEFAULT
                .withCollectionIds(Cf.list(CONTENT_BLOCKS, FOLDER_BLOCKS))
                .withRecordOrder(ContentCreature.Fields.CTIME.column().orderByDesc())
                .withLimits(SqlLimits.first(1));

        return dataApiManager.getRecords(db.spec(), filter).map(ContentCreature::fromDataRecord).singleO();
    }

    private static MpfsPath escapedDirectory(MpfsPath path) {
        return MpfsPath.parseDir(PgSqlQueryUtils.escapeJson(path.getDirectory().value));
    }
}
