package ru.yandex.chemodan.app.lentaloader.lenta.limit;

import com.google.common.cache.Cache;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.junit.Before;
import org.junit.Test;
import org.mockito.ArgumentCaptor;
import org.mockito.Mockito;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function0;
import ru.yandex.bolts.function.Function1B;
import ru.yandex.bolts.function.Function1V;
import ru.yandex.bolts.function.Function2B;
import ru.yandex.chemodan.app.dataapi.api.db.Database;
import ru.yandex.chemodan.app.dataapi.api.db.ref.UserDatabaseSpec;
import ru.yandex.chemodan.app.dataapi.api.deltas.Delta;
import ru.yandex.chemodan.app.dataapi.api.deltas.RecordChange;
import ru.yandex.chemodan.app.dataapi.api.deltas.RevisionCheckMode;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.dataapi.core.manager.DataApiManager;
import ru.yandex.chemodan.app.dataapi.test.DataApiTestSupport;
import ru.yandex.chemodan.app.lentaloader.blocks.ModifiedResource;
import ru.yandex.chemodan.app.lentaloader.lenta.LentaManager;
import ru.yandex.chemodan.app.lentaloader.lenta.LentaRecordType;
import ru.yandex.chemodan.app.lentaloader.log.ActionInfo;
import ru.yandex.chemodan.app.lentaloader.log.ActionSource;
import ru.yandex.chemodan.app.lentaloader.test.TestUtils;
import ru.yandex.chemodan.app.lentaloader.worker.tasks.CleanupLentaBlockCreaturesTask;
import ru.yandex.chemodan.app.lentaloader.worker.tasks.MergeContentLentaBlocksTask;
import ru.yandex.chemodan.cache.MeteredCache;
import ru.yandex.chemodan.eventlog.events.MpfsAddress;
import ru.yandex.chemodan.eventlog.events.MpfsPath;
import ru.yandex.chemodan.eventlog.events.Resource;
import ru.yandex.chemodan.mpfs.MpfsResourceId;
import ru.yandex.chemodan.mpfs.MpfsUid;
import ru.yandex.commune.bazinga.pg.PgBazingaTaskManager;
import ru.yandex.misc.reflection.ClassX;
import ru.yandex.misc.test.Assert;

import static org.mockito.Matchers.any;

/**
 * @author dbrylev
 */
public class LentaLimitManagerTest extends DataApiTestSupport {

    @Autowired
    private DataApiManager dataApiManager;

    private LentaManager lentaManager;
    private PgBazingaTaskManager pgBazingaTaskManager;

    private Cache<UserContentPath, String> foldersCache;

    private LentaLimitManagerImpl lentaLimitManager;

    private Database db;
    private DataApiUserId uid;

    private final MpfsUid mpfsUid = new MpfsUid(1234567L);

    @Before
    public void setup() {
        uid = createRandomCleanUserInDefaultShard();
        db = dataApiManager.createDatabase(new UserDatabaseSpec(uid, LentaLimitManagerImpl.CREATURES_DB));

        lentaManager = Mockito.mock(LentaManager.class);
        pgBazingaTaskManager = Mockito.mock(PgBazingaTaskManager.class);

        foldersCache = Mockito.mock(ClassX.wrap(Cache.class).<Cache<UserContentPath, String>>uncheckedCast().getClazz());

        lentaLimitManager = new LentaLimitManagerImpl(
                lentaManager, dataApiManager, pgBazingaTaskManager,
                () -> (path, size) -> true,
                (uid, path) -> Option.of(MpfsResourceId.parse(uid + ":" + path)),
                new MeteredCache<>(foldersCache));
    }

    @Test
    public void handleContentPathBlocking() {
        Function1B<String> handleBlocking = path -> lentaLimitManager.handleContentPathBlocking(
                uid, consModifiedResource(path), actionInfo());

        insertCreature("blocker1", LentaLimitManagerImpl.FOLDER_BLOCKS, consModifiedResource("blocked"));
        insertCreature("blocker2", LentaLimitManagerImpl.FOLDER_BLOCKS, consModifiedResource("blocked/1"));
        insertCreature("blocker3", LentaLimitManagerImpl.FOLDER_BLOCKS, consModifiedResource("blocked/1/2"));

        Assert.isTrue(handleBlocking.apply("blocked/1/2"));

        ArgumentCaptor<String> blockIdCaptor = ArgumentCaptor.forClass(String.class);
        Mockito.verify(lentaManager, Mockito.only())
                .updateAndUpOrDeleteBlockDelayed(any(), blockIdCaptor.capture(), any(), any());

        Assert.equals("blocker1", blockIdCaptor.getValue());

        Assert.isFalse(handleBlocking.apply("unmatched"));
    }

    @Test
    public void handleContentPathBlockingFromCache() {
        Function0<Boolean> handleBlocking = () -> lentaLimitManager.handleContentPathBlocking(
                uid, consModifiedResource("blocked/sub"), actionInfo());

        Assert.isFalse(handleBlocking.apply());

        mockFoldersCache(Cf.list("unmatched"));

        Assert.isFalse(handleBlocking.apply());

        mockFoldersCache(Cf.list("unmatched", "blocked"));

        Assert.isTrue(handleBlocking.apply());

        ArgumentCaptor<LentaRecordType> typeCaptor = ArgumentCaptor.forClass(LentaRecordType.class);
        ArgumentCaptor<String> blockIdCaptor = ArgumentCaptor.forClass(String.class);

        Mockito.verify(lentaManager, Mockito.only())
                .updateAndUpOrDeleteBlockDelayed(any(), blockIdCaptor.capture(), typeCaptor.capture(), any());

        Assert.equals("blocked", blockIdCaptor.getValue());
        Assert.equals(LentaRecordType.FOLDER_BLOCK, typeCaptor.getValue());

        mockFoldersCache(Cf.list("blocked/sub"));

        Assert.isTrue(handleBlocking.apply());
    }

    @Test
    public void insertContentBlockCreature() {
        Function1V<String> insertCreature = id -> lentaLimitManager.insertContentBlockCreature(
                uid, "" + id, consModifiedResource(id), actionInfo());

        insertCreature.apply("first");

        Mockito.verify(pgBazingaTaskManager).schedule(Mockito.isA(MergeContentLentaBlocksTask.class), any());
        Mockito.verify(pgBazingaTaskManager).schedule(Mockito.isA(CleanupLentaBlockCreaturesTask.class), any());

        Mockito.verifyZeroInteractions(lentaManager);

        insertCreature.apply("second");

        Assert.hasSize(2, lentaLimitManager.findActualContentCreatures(db));
    }

    @Test
    public void mergeCreatures() {
        insertCreature("first", LentaLimitManagerImpl.FOLDER_BLOCKS, consModifiedResource("one"));
        insertCreature("second", LentaLimitManagerImpl.CONTENT_BLOCKS, consModifiedResource("two"));

        lentaLimitManager.mergeContentBlocks(uid, actionInfo());

        ContentCreature creature = lentaLimitManager.findActualContentCreatures(db).single();

        Assert.isTrue(creature.isFolder());
        Assert.equals("/disk/", creature.path.getValue());

        Mockito.verify(pgBazingaTaskManager).schedule(Mockito.isA(MergeContentLentaBlocksTask.class), any());
        Mockito.verify(pgBazingaTaskManager).schedule(Mockito.isA(CleanupLentaBlockCreaturesTask.class), any());


        ArgumentCaptor<ListF<FolderBlockData>> dataCaptor = ArgumentCaptor.forClass(
                ClassX.wrap(ListF.class).<ListF<FolderBlockData>>uncheckedCast().getClazz());

        ArgumentCaptor<ListF<String>> idsCaptor = ArgumentCaptor.forClass(
                ClassX.wrap(ListF.class).<ListF<String>>uncheckedCast().getClazz());

        Mockito.verify(lentaManager, Mockito.only()).createAndDeleteBlocks(
                any(), dataCaptor.capture(), idsCaptor.capture(), any());

        Assert.forAll(Cf.list("first", "second"), idsCaptor.getValue()::containsTs);
        Assert.equals(creature.blockId, dataCaptor.getValue().single().blockId);
    }

    @Test
    public void cleanupCreatures() {
        insertCreature("one", LentaLimitManagerImpl.FOLDER_BLOCKS, consModifiedResource("one"));
        insertCreature("two", LentaLimitManagerImpl.CONTENT_BLOCKS, consModifiedResource("two"));
        insertCreature("three", LentaLimitManagerImpl.FOLDER_BLOCKS, consModifiedResource("three"));

        Assert.hasSize(3, lentaLimitManager.findActualContentCreatures(db));

        TestUtils.withFixedNow(Instant.now().plus(Duration.standardDays(2)),
                () -> lentaLimitManager.cleanupCreatures(uid, actionInfo()));

        Assert.isEmpty(lentaLimitManager.findActualContentCreatures(db));

        Mockito.verifyZeroInteractions(pgBazingaTaskManager);
    }

    @Test
    public void pathLevelLimitExceeded() {
        Function<ListF<Integer>, Function2B<Integer, Integer>> exceededF = limits ->
                LentaLimitManagerImpl.pathLevelLimitExceededF(limits)
                        .compose1(i -> MpfsPath.parseDir(Cf.repeat("a", i + 1).mkString("/", "/", "/")));

        Function2B<Integer, Integer> exceeded = exceededF.apply(Cf.list(-1, 10));

        Assert.isFalse(exceeded.apply(0, 100500));

        Assert.isFalse(exceeded.apply(1, 10));
        Assert.isTrue(exceeded.apply(1, 11));

        Assert.isTrue(exceeded.apply(2, 11));
        Assert.isTrue(exceeded.apply(100, 11));

        exceeded = exceededF.apply(Cf.list(100, 50, 50, -1));

        Assert.isTrue(exceeded.apply(0, 101));
        Assert.isTrue(exceeded.apply(1, 101));

        Assert.isTrue(exceeded.apply(2, 51));

        Assert.isFalse(exceeded.apply(4, 100500));
    }

    private void insertCreature(String id, String collectionId, ModifiedResource resource) {
        ContentCreature creature = new ContentCreature(id, collectionId,
                resource.address.path, Option.of(resource.mediaType()),
                resource.modifier, Option.empty(), Instant.now());

        db = dataApiManager.getDatabase(new UserDatabaseSpec(uid, LentaLimitManagerImpl.CREATURES_DB));

        dataApiManager.applyDelta(db, RevisionCheckMode.PER_RECORD,
                new Delta(RecordChange.insert(creature.collectionId, creature.blockId, creature.toData())));
    }

    private void mockFoldersCache(ListF<String> paths) {
        MapF<UserContentPath, String> cached = paths.toMap(
                path -> new UserContentPath(uid, MpfsPath.parseDir("/disk/" + path), mpfsUid), id -> id);

        ArgumentCaptor<UserContentPath> pathCaptor = ArgumentCaptor.forClass(UserContentPath.class);

        Mockito.when(foldersCache.getIfPresent(pathCaptor.capture()))
                .thenAnswer(x -> cached.getOrElse(pathCaptor.getValue(), null));
    }

    private ModifiedResource consModifiedResource(String path) {
        return new ModifiedResource(
                Resource.file("document", "fileid", mpfsUid),
                new MpfsAddress(mpfsUid, MpfsPath.parseDir("/disk/" + path)),
                MpfsResourceId.parse(mpfsUid + ":fileid"), mpfsUid, mpfsUid);
    }

    private static ActionInfo actionInfo() {
        return ActionInfo.internal(ActionSource.test());
    }
}
