package ru.yandex.chemodan.app.lentaloader.log;

import org.joda.time.Instant;

import ru.yandex.bolts.collection.Either;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataField;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataRecord;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.lentaloader.lenta.CachedOrLoadedBlock;
import ru.yandex.chemodan.app.lentaloader.lenta.LentaBlockIdAndType;
import ru.yandex.chemodan.app.lentaloader.lenta.LentaBlockRecord;
import ru.yandex.chemodan.app.lentaloader.lenta.LentaRecordType;
import ru.yandex.commune.bazinga.impl.FullJobId;
import ru.yandex.misc.enums.EnumUtils;
import ru.yandex.misc.lang.DefaultObject;
import ru.yandex.misc.lang.StringUtils;

/**
 * @author dbrylev
 */
public class LentaBlockEvent extends DefaultObject implements LogDataProvider {
    public final DataApiUserId uid;
    public final LentaEventType type;
    public final Option<ActionReason> reason;
    public final ActionInfo actionInfo;

    public final Option<String> blockId;
    public final LentaRecordType blockType;
    public final Option<String> blockCollection;
    public final Option<MapF<String, DataField>> blockData;
    public final Option<MapF<String, DataField>> changedFields;

    public final Option<FullJobId> scheduledTask;
    public final Option<Instant> scheduledTime;

    private LentaBlockEvent(Builder builder) {
        this.uid = builder.uid;
        this.type = builder.type;
        this.reason = builder.reason;
        this.actionInfo = builder.actionInfo;
        this.blockId = StringUtils.notEmptyO(builder.block.fold(b -> b.id, b -> b.id));
        this.blockType = builder.block.fold(b -> b.type, b -> b.type);
        this.blockCollection = builder.collection;
        this.blockData = builder.block.leftO().map(LentaBlockRecord::toData);
        this.changedFields = builder.changedFields;
        this.scheduledTask = builder.scheduledTask;
        this.scheduledTime = builder.scheduledTime;
    }

    public void log() {
        LentaEventLogger.log(this);
    }

    @Override
    public Tuple2List<String, String> logData() {
        Tuple2List<String, String> result = Tuple2List.arrayList();

        result.add("uid", uid.serialize());
        result.add("event_type", EnumUtils.toXmlName(type));
        reason.forEach(r -> result.add("reason", EnumUtils.toXmlName(r)));

        result.addAll(actionInfo.logData());

        scheduledTask.forEach(task -> result.add("scheduled_task", task.toSerializedString()));
        scheduledTime.forEach(time -> result.add("scheduled_time", LentaEventLogger.dateTimeFormat.print(time)));

        result.add("block_type", blockType.value());
        blockId.forEach(id -> result.add("block_id", id));
        blockCollection.forEach(c -> result.add("block_collection", c));

        blockData.forEach(d -> result.add("block_data", LentaEventLogger.serializeToJson(d)));
        changedFields.forEach(c -> result.add("block_changed", LentaEventLogger.serializeToJson(c)));

        return result;
    }

    public static LentaBlockEvent created(DataApiUserId uid, LentaBlockRecord block, ActionInfo actionInfo) {
        return builder(uid, LentaEventType.BLOCK_CREATE, block, actionInfo).build();
    }

    public static LentaBlockEvent createdPin(DataApiUserId uid, LentaBlockRecord block, ActionInfo actionInfo) {
        return builder(uid, LentaEventType.BLOCK_CREATE_PINNED, block, actionInfo).build();
    }

    public static LentaBlockEvent createIgnored(
            DataApiUserId uid, LentaBlockRecord block, ActionReason reason, ActionInfo actionInfo)
    {
        return builder(uid, LentaEventType.BLOCK_CREATE_IGNORED, block, actionInfo).setReason(reason).build();
    }

    public static LentaBlockEvent createPinIgnored(
            DataApiUserId uid, LentaBlockRecord block, ActionReason reason, ActionInfo actionInfo)
    {
        return builder(uid, LentaEventType.BLOCK_CREATE_IGNORED_PINNED, block, actionInfo).setReason(reason).build();
    }

    public static LentaBlockEvent updated(
            DataApiUserId uid, DataRecord record, LentaBlockRecord updated, ActionInfo actionInfo)
    {
        return updatedOrUpped(uid, LentaEventType.BLOCK_UPDATE, record, updated, actionInfo);
    }

    public static LentaBlockEvent updatedAndUpped(
            DataApiUserId uid, DataRecord record, LentaBlockRecord updated, ActionInfo actionInfo)
    {
        return updatedOrUpped(uid, LentaEventType.BLOCK_UPDATE_AND_UP, record, updated, actionInfo);
    }

    private static LentaBlockEvent updatedOrUpped(
            DataApiUserId uid, LentaEventType type, DataRecord record, LentaBlockRecord updated, ActionInfo actionInfo)
    {
        MapF<String, DataField> changedFields = record.getData()
                .filterKeys(updated.diffFrom(record.getData()).map(c -> c.key)::containsTs);

        return builder(uid, type, updated, actionInfo)
                .setCollection(record.getCollectionId())
                .setChangedFields(changedFields).build();
    }

    public static LentaBlockEvent updateIgnored(
            DataApiUserId uid, CachedOrLoadedBlock block, ActionReason reason, ActionInfo actionInfo)
    {
        return updateIgnored(uid, block.toBlockRecord(), block.getCollectionIdOrCached(), reason, actionInfo);
    }

    public static LentaBlockEvent updateIgnored(
            DataApiUserId uid, LentaBlockRecord block, String collection, ActionReason reason, ActionInfo actionInfo)
    {
        return builder(uid, LentaEventType.BLOCK_UPDATE_IGNORED, block, actionInfo)
                .setCollection(collection)
                .setReason(reason).build();
    }

    public static LentaBlockEvent pin(
            DataApiUserId uid, LentaBlockRecord block, String collection, ActionInfo actionInfo)
    {
        return builder(uid, LentaEventType.BLOCK_PIN, block, actionInfo).setCollection(collection).build();
    }

    public static LentaBlockEvent unpin(DataApiUserId uid, LentaBlockRecord block, ActionInfo actionInfo) {
        return builder(uid, LentaEventType.BLOCK_UNPIN, block, actionInfo).build();
    }

    public static LentaBlockEvent deleted(DataApiUserId uid, DataRecord record, ReasonedAction actionInfo) {
        return deleted(uid, record, actionInfo.reason, actionInfo.actionInfo);
    }

    public static LentaBlockEvent deleted(
            DataApiUserId uid, DataRecord record, ActionReason reason, ActionInfo actionInfo)
    {
        return builder(uid, LentaEventType.BLOCK_DELETE, LentaBlockRecord.fromDataRecord(record), actionInfo)
                .setCollection(record.getCollectionId())
                .setReason(reason).build();
    }

    public static LentaBlockEvent deleteIgnored(DataApiUserId uid, DataRecord record, ActionInfo actionInfo) {
        return builder(uid, LentaEventType.BLOCK_DELETE_IGNORED, LentaBlockRecord.fromDataRecord(record), actionInfo)
                .setCollection(record.getCollectionId()).build();
    }

    public static LentaBlockEvent taskScheduled(
            DataApiUserId uid, FullJobId jobId, Instant scheduleTime,
            LentaBlockIdAndType block, Option<ActionReason> reason, ActionInfo actionInfo)
    {
        return taskScheduledOrMerged(LentaEventType.BLOCK_TASK_SCHEDULED,
                uid, jobId, scheduleTime, block, reason, actionInfo);
    }

    public static LentaBlockEvent taskMerged(
            DataApiUserId uid, FullJobId jobId, Instant scheduleTime,
            LentaBlockIdAndType block, Option<ActionReason> reason, ActionInfo actionInfo)
    {
        return taskScheduledOrMerged(LentaEventType.BLOCK_TASK_MERGED,
                uid, jobId, scheduleTime, block, reason, actionInfo);
    }

    private static LentaBlockEvent taskScheduledOrMerged(
            LentaEventType type, DataApiUserId uid,
            FullJobId jobId, Instant scheduleTime,
            LentaBlockIdAndType block, Option<ActionReason> reason, ActionInfo actionInfo)
    {
        Builder b = new Builder(uid, type, block, actionInfo);

        if (reason.isPresent()) {
            b = b.setReason(reason.get());
        }
        return b.setScheduled(jobId, scheduleTime).build();
    }

    private static Builder builder(
            DataApiUserId uid, LentaEventType type, LentaBlockRecord block, ActionInfo actionInfo)
    {
        return new Builder(uid, type, block, actionInfo);
    }

    private static class Builder {
        private final DataApiUserId uid;
        private final LentaEventType type;
        private final Either<LentaBlockRecord, LentaBlockIdAndType> block;
        private final ActionInfo actionInfo;

        private Option<ActionReason> reason = Option.empty();
        private Option<String> collection = Option.empty();
        private Option<MapF<String, DataField>> changedFields = Option.empty();

        private Option<FullJobId> scheduledTask = Option.empty();
        private Option<Instant> scheduledTime = Option.empty();

        public Builder(DataApiUserId uid, LentaEventType type, LentaBlockRecord block, ActionInfo actionInfo) {
            this.uid = uid;
            this.type = type;
            this.block = Either.left(block);
            this.actionInfo = actionInfo;
        }

        public Builder(DataApiUserId uid, LentaEventType type, LentaBlockIdAndType block, ActionInfo actionInfo) {
            this.uid = uid;
            this.type = type;
            this.block = Either.right(block);
            this.actionInfo = actionInfo;
        }

        public LentaBlockEvent build() {
            return new LentaBlockEvent(this);
        }

        public Builder setCollection(String collection) {
            this.collection = Option.of(collection);
            return this;
        }

        public Builder setReason(ActionReason r) {
            reason = Option.of(r);
            return this;
        }

        public Builder setChangedFields(MapF<String, DataField> changes) {
            changedFields = Option.of(changes);
            return this;
        }

        public Builder setScheduled(FullJobId job, Instant time) {
            scheduledTask = Option.of(job);
            scheduledTime = Option.of(time);

            return this;
        }
    }
}
