package ru.yandex.chemodan.app.lentaloader.memories;

import org.joda.time.LocalDate;

import ru.yandex.chemodan.app.lentaloader.YtPathsUtils;
import ru.yandex.chemodan.app.lentaloader.worker.LentaTaskQueueName;
import ru.yandex.chemodan.util.yt.YqlHelper;
import ru.yandex.chemodan.util.yt.YtHelper;
import ru.yandex.commune.bazinga.scheduler.CronTask;
import ru.yandex.commune.bazinga.scheduler.ExecutionContext;
import ru.yandex.commune.bazinga.scheduler.TaskQueueName;
import ru.yandex.commune.bazinga.scheduler.schedule.Schedule;
import ru.yandex.commune.bazinga.scheduler.schedule.ScheduleCron;
import ru.yandex.commune.bazinga.scheduler.schedule.ScheduleWithRetry;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.time.MoscowTime;

/**
 * @author messiahlap
 */
public class CollectUsersWithBlocksDataCronTask extends CronTask {

    private static final Logger logger = LoggerFactory.getLogger(CollectUsersWithBlocksDataCronTask.class);

    private final int executionHour;

    private final int retryCount;

    private final YtHelper yt;

    private final int waitForSourceDataDurationMinutes;

    private final YqlHelper yqlHelper;

    public CollectUsersWithBlocksDataCronTask(YtHelper yt, YqlHelper yqlHelper, int executionHour, int retryCount,
                                              int waitForSourceDataDurationMinutes)
    {
        this.yt = yt;
        this.yqlHelper = yqlHelper;
        this.executionHour = executionHour;
        this.retryCount = retryCount;
        this.waitForSourceDataDurationMinutes = waitForSourceDataDurationMinutes;
    }

    @Override
    public Schedule cronExpression() {
        return new ScheduleWithRetry(new ScheduleCron("0 " + executionHour + " * * *", MoscowTime.TZ), retryCount);
    }

    @Override
    public void execute(ExecutionContext executionContext) throws Exception {
        collectData(LocalDate.now().minusDays(1));
    }

    public void collectData(LocalDate dateToCollectData) {
        YPath sourceDataPath = YPath.simple("//logs/ydisk-lenta-cool-events-log/1d/" + dateToCollectData);
        YPath outputFolderDataPath = YtPathsUtils.getCoolLentaUserWithBlocksYPath();
        YPath lastAvailableOutputDataPath = outputFolderDataPath.child(dateToCollectData.minusDays(1).toString());
        YPath currentOutputDataPath = outputFolderDataPath.child(dateToCollectData.toString());
        yt.waitForTableForMinutes(sourceDataPath, waitForSourceDataDurationMinutes);
        String yql =
                "PRAGMA yt.DefaultOperationWeight=\"50.0\";\n" +
                        "USE hahn;\n" +
                        "\n" +
                        "INSERT INTO {{output_path}}\n" +
                        "SELECT DISTINCT uid FROM (\n" +
                        "   SELECT uid FROM {{logInputPath}} WHERE event_type IN ('all-blocks-create', 'all-blocks-update')\n" +
                        "   UNION ALL\n" +
                        "   SELECT uid FROM {{prevTableInputPath}}\n" +
                        ")";

        yql = yql
                .replace("{{logInputPath}}", "`" + sourceDataPath + "`")
                .replace("{{prevTableInputPath}}", "`" + lastAvailableOutputDataPath + "`")
                .replace("{{output_path}}", "`" + currentOutputDataPath + "`");

        // Если таблица уже есть - незачем заново ее генерировать
        if (yt.existsWithRetries(currentOutputDataPath)) {
            logger.debug("The table {} exists already", currentOutputDataPath);
            return;
        }
        logger.debug("The table {} does not exist. Going to create it", currentOutputDataPath);
        yqlHelper.execute(yql);
    }

    @Override
    public TaskQueueName queueName() {
        return LentaTaskQueueName.LENTA_CRON;
    }
}
