package ru.yandex.chemodan.app.lentaloader.memories;

import lombok.Data;
import org.apache.commons.lang3.mutable.MutableInt;
import org.joda.time.DateTimeZone;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.joda.time.LocalDate;
import org.joda.time.LocalTime;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Either;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.lentaloader.YtPathsUtils;
import ru.yandex.chemodan.app.lentaloader.reminder.RemindPhotosTask;
import ru.yandex.chemodan.app.uaas.experiments.ExperimentsManager;
import ru.yandex.chemodan.util.blackbox.UserTimezoneHelper;
import ru.yandex.chemodan.util.yt.YtHelper;
import ru.yandex.commune.bazinga.BazingaTaskManager;
import ru.yandex.commune.bazinga.scheduler.OnetimeTask;
import ru.yandex.commune.util.RetryUtils;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.tables.YTableEntryTypes;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.random.Random2;
import ru.yandex.misc.thread.ThreadUtils;
import ru.yandex.misc.time.MoscowTime;

/**
 * @author yashunsky
 * @author dk666
 */
public class MemoriesRoutines {
    private static final Logger logger = LoggerFactory.getLogger(MemoriesRoutines.class);

    private static final String REMIND_PHOTOS_EXPERIMENT_PREFIX = "disk_lenta_remind_photo_task_count_per_day_";
    private static final long HOUR_IN_MILLIS = Duration.standardHours(1).getMillis();
    public static final String UIDS_WITH_BLOCKS_PROCESSED_ATTRIBUTE = "processed_by_schedule_memories";

    static final int PHOTO_REMIND_VALID_TIME_HOUR_FROM = 11;
    static final int PHOTO_REMIND_VALID_TIME_HOUR_TO = 20;

    private YtHelper yt;
    private final int scheduleMemoriesBatchSize;
    private final BazingaTaskManager bazingaTaskManager;
    private final UserTimezoneHelper userTimezoneHelper;
    private final ExperimentsManager experimentsManager;
    private final int limitToSearchTableInThePast;
    private final int errorRateForRemindPhotosPercents;
    private final int scheduleRemindPhotoRetryCount;
    private final int errorRateForCollectUsersWithBlocksPercents;
    private final int collectUidsOffsetDays;

    public MemoriesRoutines(YtHelper ytHelper, int scheduleMemoriesBatchSize, BazingaTaskManager bazingaTaskManager,
            UserTimezoneHelper userTimezoneHelper, ExperimentsManager experimentsManager,
            int limitToSearchTableInThePast, int errorRateForRemindPhotosPercents, int scheduleRemindPhotoRetryCount,
            int errorRateForCollectUsersWithBlocksPercents, int collectUidsOffsetDays)
    {
        this.yt = ytHelper;
        this.scheduleMemoriesBatchSize = scheduleMemoriesBatchSize;
        this.bazingaTaskManager = bazingaTaskManager;
        this.userTimezoneHelper = userTimezoneHelper;
        this.experimentsManager = experimentsManager;
        this.limitToSearchTableInThePast = limitToSearchTableInThePast;
        this.errorRateForRemindPhotosPercents = errorRateForRemindPhotosPercents;
        this.scheduleRemindPhotoRetryCount = scheduleRemindPhotoRetryCount;
        this.errorRateForCollectUsersWithBlocksPercents = errorRateForCollectUsersWithBlocksPercents;
        this.collectUidsOffsetDays = collectUidsOffsetDays;
    }

    public void scheduleMemoriesSetFromUidWithBlocks(LocalDate today, int futureOffsetDays) {
        MapF<String, Integer> failedSchedulesForPaths = Cf.hashMap();
        boolean shouldFail = false;
        for (int i = limitToSearchTableInThePast; i >= 1; i--) {
            LocalDate day = today.minusDays(i);
            YPath path = YtPathsUtils.getCoolLentaUserWithBlocksYPath().child(day.toString());
            YPath attributePath = path.attribute(UIDS_WITH_BLOCKS_PROCESSED_ATTRIBUTE);
            if (!yt.existsWithRetries(path)) {
                logger.debug("Table {} does not exist", path);
                continue;
            }
            if (!yt.existsWithRetries(attributePath) ||
                    !yt.getWithRetries(() -> yt.cypress().get(attributePath).boolValue())) {
                logger.debug("Starting process table {}", path);
                long rowCount = yt.getRowCount(path);
                if (rowCount == 0) {
                    logger.debug("no data to schedule tasks {}", path);
                    continue;
                }
                MutableInt failedCounter = new MutableInt(0);
                int taskCount = (int) (rowCount / scheduleMemoriesBatchSize) + 1;
                Cf.range(0, taskCount)
                        .forEach(batchIndex -> scheduleCollectUidsWithBlocksTask(
                                batchIndex, rowCount, path, futureOffsetDays, day, failedCounter, taskCount
                        ));
                failedSchedulesForPaths.put(path.toString(), failedCounter.getValue());
                if (isErrorLimitForSchedulingCollectUidsWithBlocksTaskReached(failedCounter, taskCount)) {
                    shouldFail = true;
                    continue;
                }
                setSuccessfulStatusOfScheduleMemoriesBlocksTasksForSourceTable(path);
            }
        }
        if (shouldFail) {
            throw new IllegalStateException(String.format("Too many not scheduled tasks failedSchedulesStatistics=%s errorRatePercent=%s",
                    failedSchedulesForPaths, errorRateForCollectUsersWithBlocksPercents));
        }
    }

    private boolean isErrorLimitForSchedulingCollectUidsWithBlocksTaskReached(MutableInt failedCounter, int taskCount) {
        return failedCounter.intValue() * 100d / taskCount >= errorRateForCollectUsersWithBlocksPercents;
    }

    private void setSuccessfulStatusOfScheduleMemoriesBlocksTasksForSourceTable(YPath pathToTable) {
        yt.runWithRetries(() -> yt.cypress().set(pathToTable.attribute(UIDS_WITH_BLOCKS_PROCESSED_ATTRIBUTE), true));
    }

    private void scheduleCollectUidsWithBlocksTask(int batchIndex, long rowCount, YPath path,
            int futureOffsetDays, LocalDate today, MutableInt failedCounter, int taskCount) {
        if (isErrorLimitForSchedulingCollectUidsWithBlocksTaskReached(failedCounter, taskCount)) {
            return;
        }
        long startIndex = (long) batchIndex * scheduleMemoriesBatchSize;
        if (startIndex >= rowCount) {
            return;
        }
        LocalDate dayToSchedule = today.plusDays(futureOffsetDays);
        Instant executionTime = Random2.R.nextInstant(dayToSchedule.toDateTimeAtStartOfDay().toInstant(),
                dayToSchedule.plusDays(1).toDateTimeAtStartOfDay().toInstant());
        if (!scheduleWithRetries(
                new CollectUidsWithBlocksTask(path, startIndex,
                        Math.min(startIndex + scheduleMemoriesBatchSize, rowCount)), executionTime)) {
            failedCounter.increment();
        }
    }

    private boolean scheduleWithRetries(OnetimeTask task, Instant executionTime) {
        ListF<Integer> sleeps = Cf.list(0, 1000, 1000, 3000, 5000);

        Exception lastException = null;
        for (int i = 0; i < 10; i++) {
            try {
                bazingaTaskManager.schedule(task, executionTime);
                return true;
            } catch (Exception e) {
                int sleep = sleeps.getO(i).getOrElse(sleeps.last());
                logger.warn("Retry submitting task to bazinga: {}. sleep {}", e, sleep);
                lastException = e;
                ThreadUtils.sleep(sleep);
            }
        }
        logger.error("Failed to schedule task: {}", lastException);
        return false;
    }

    public void scheduleMemoriesBlocksCreationWithStats(YPath pathWithRange, LocalDate tableDate) {
        MutableInt failedTasksCounter = new MutableInt(0);
        yt.runWithRetries(
                () -> yt.tables().read(pathWithRange, YTableEntryTypes.bender(UidWithBlocks.class),
                        uidWithBlocks -> {
                    scheduleRemindPhotoTaskForUid(uidWithBlocks, failedTasksCounter, tableDate);
                }));
        double failedTasksRate = failedTasksCounter.intValue() * 100d / scheduleMemoriesBatchSize;
        if (failedTasksRate > errorRateForRemindPhotosPercents) {
            throw new IllegalStateException(String
                    .format("Too much not scheduled remindPhoto tasks failedTasksCount=%s scheduleMemoriesBathSize=%s",
                            failedTasksCounter.intValue(), scheduleMemoriesBatchSize));
        }
    }

    private void scheduleRemindPhotoTaskForUid(UidWithBlocks uidWithBlocks, MutableInt failedTasksCounter, LocalDate tableDate) {
        DateTimeZone userTimeZone = userTimezoneHelper.getUserTimezone(uidWithBlocks.getUid());
        LocalDate scheduleDate = tableDate.plusDays(collectUidsOffsetDays).plusDays(1);
        if (scheduleDate.isBefore(LocalDate.now(MoscowTime.TZ))) {
            return;
        }
        Instant start = scheduleDate.toDateTime(new LocalTime(PHOTO_REMIND_VALID_TIME_HOUR_FROM, 0), userTimeZone).toInstant();
        Instant end = scheduleDate.toDateTime(new LocalTime(PHOTO_REMIND_VALID_TIME_HOUR_TO, 0), userTimeZone).toInstant();
        long fullInterval = end.getMillis() - start.getMillis();
        int intervalsCount = getRemindPhotoTaskCountForUser(uidWithBlocks.getUid());
        long intervalsLag = fullInterval / intervalsCount;
        if (new Duration(intervalsLag).getStandardMinutes() <= 60) {
            throw new IllegalStateException(String.format("The interval lag is less or equal 1h %s ms", intervalsLag));
        }
        Option<Long> previousExecutionTimeO = Option.empty();
        for (int i = 0; i < intervalsCount; i++) {
            long initialStartInterval = start.getMillis() + i * intervalsLag;
            long startInterval = previousExecutionTimeO
                    .filter(previousExecutionTime -> HOUR_IN_MILLIS > initialStartInterval - previousExecutionTime)
                    .map(previousExecutionTime -> previousExecutionTime + HOUR_IN_MILLIS).getOrElse(initialStartInterval);
            long endInterval = Math.min(end.getMillis(), initialStartInterval + intervalsLag);
            Instant executionTime = Random2.R.nextInstant(new Instant(startInterval), new Instant(endInterval));
            previousExecutionTimeO = Option.of(executionTime.getMillis());
            int index = i;
            Either<Boolean, Throwable> result = RetryUtils.retryE(logger, scheduleRemindPhotoRetryCount, () ->
                    scheduleWithRetries(new RemindPhotosTask(uidWithBlocks.getUid(), scheduleDate,
                            PHOTO_REMIND_VALID_TIME_HOUR_TO, index, intervalsCount), executionTime)
            );
            if (result.isRight()) {
                logger.warn("Can't schedule RemindPhotosTask for {} at {} with index {} with exception {}",
                        uidWithBlocks.getUid(), scheduleDate, index, result.getRight());
                failedTasksCounter.increment();
                continue;
            }
            if (!result.getLeft()) {
                logger.warn("Can't schedule RemindPhotosTask for {} at {} with index {}",
                        uidWithBlocks.getUid(), scheduleDate, index);
                failedTasksCounter.increment();
                continue;
            }
        }
    }

    private int getRemindPhotoTaskCountForUser(long uid) {
        Option<String> flagO = experimentsManager.getFlags(uid).find(flag -> flag.startsWith(REMIND_PHOTOS_EXPERIMENT_PREFIX));
        if (!flagO.isPresent()) {
            return 1;
        }
        String flag = flagO.get();
        return Integer.parseInt(flag.substring(REMIND_PHOTOS_EXPERIMENT_PREFIX.length()));
    }

    @Data
    @BenderBindAllFields
    public static class UidWithBlocks {
        private final long uid;
    }
}
