package ru.yandex.chemodan.app.lentaloader.reminder;

import lombok.AllArgsConstructor;
import lombok.Data;
import lombok.SneakyThrows;
import org.apache.http.client.utils.URIBuilder;
import org.jetbrains.annotations.NotNull;
import org.joda.time.DateTime;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.joda.time.LocalDate;
import org.joda.time.Years;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function1B;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataField;
import ru.yandex.chemodan.app.dataapi.api.data.filter.RecordsFilter;
import ru.yandex.chemodan.app.dataapi.api.data.filter.ordering.ByIdRecordOrder;
import ru.yandex.chemodan.app.dataapi.api.data.filter.ordering.OrderedUUID;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataRecord;
import ru.yandex.chemodan.app.dataapi.api.db.Database;
import ru.yandex.chemodan.app.dataapi.api.deltas.Delta;
import ru.yandex.chemodan.app.dataapi.api.deltas.RecordChange;
import ru.yandex.chemodan.app.dataapi.api.deltas.RevisionCheckMode;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiPassportUserId;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.dataapi.core.manager.DataApiManager;
import ru.yandex.chemodan.app.dataapi.support.I18nValue;
import ru.yandex.chemodan.app.lentaloader.cool.CoolLentaConfigurationManager;
import ru.yandex.chemodan.app.lentaloader.cool.CoolLentaFileItem;
import ru.yandex.chemodan.app.lentaloader.cool.CoolLentaManager;
import ru.yandex.chemodan.app.lentaloader.cool.InconsistentCoolLentaBlockException;
import ru.yandex.chemodan.app.lentaloader.cool.MordaPushManager;
import ru.yandex.chemodan.app.lentaloader.cool.generator.BlockGeneratorUtils;
import ru.yandex.chemodan.app.lentaloader.cool.generator.CoolLentaBlockGenerator;
import ru.yandex.chemodan.app.lentaloader.cool.generator.ThematicBlocksGenerator;
import ru.yandex.chemodan.app.lentaloader.cool.generator.ThemeDefinition;
import ru.yandex.chemodan.app.lentaloader.cool.generator.ThemeDefinitionRegistry;
import ru.yandex.chemodan.app.lentaloader.cool.log.CoolLentaBlockEvent;
import ru.yandex.chemodan.app.lentaloader.cool.log.CoolLentaEventLogger;
import ru.yandex.chemodan.app.lentaloader.cool.log.CoolLentaEventType;
import ru.yandex.chemodan.app.lentaloader.cool.model.CoolLentaBlock;
import ru.yandex.chemodan.app.lentaloader.cool.model.CoolLentaBlockFields;
import ru.yandex.chemodan.app.lentaloader.cool.model.CoolLentaModelUtils;
import ru.yandex.chemodan.app.lentaloader.cool.model.MinimalCoolLentaBlock;
import ru.yandex.chemodan.app.lentaloader.cool.utils.GenerationInterval;
import ru.yandex.chemodan.app.lentaloader.cool.utils.IntervalType;
import ru.yandex.chemodan.app.lentaloader.cool.utils.TermDefinition;
import ru.yandex.chemodan.app.lentaloader.cool.utils.TextProcessor;
import ru.yandex.chemodan.app.lentaloader.cool.utils.TitleGenerationContext;
import ru.yandex.chemodan.app.lentaloader.lenta.FindOrCreateResult;
import ru.yandex.chemodan.app.lentaloader.lenta.LentaBlockRecord;
import ru.yandex.chemodan.app.lentaloader.lenta.LentaManager;
import ru.yandex.chemodan.app.lentaloader.lenta.LentaNotificationManager;
import ru.yandex.chemodan.app.lentaloader.lenta.LentaRecordType;
import ru.yandex.chemodan.app.lentaloader.lenta.update.LentaBlockCreateData;
import ru.yandex.chemodan.app.lentaloader.log.ActionInfo;
import ru.yandex.chemodan.app.lentaloader.log.ActionSource;
import ru.yandex.chemodan.app.lentaloader.reminder.sendpush.CoolLentaBlockSendPushManager;
import ru.yandex.chemodan.app.lentaloader.reminder.sendpush.GncClient;
import ru.yandex.chemodan.app.lentaloader.reminder.titles.BlockTexts;
import ru.yandex.chemodan.app.lentaloader.reminder.titles.BlockTitlesType;
import ru.yandex.chemodan.app.lentaloader.reminder.titles.CoolLentaBlockTitlesManager;
import ru.yandex.chemodan.app.lentaloader.reminder.titles.GeoCoolLentaBlockTitlesGenerator;
import ru.yandex.chemodan.app.uaas.experiments.ExperimentsManager;
import ru.yandex.chemodan.mpfs.MpfsResourceId;
import ru.yandex.chemodan.mpfs.UserBlockedException;
import ru.yandex.chemodan.util.blackbox.UserTimezoneHelper;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.inside.passport.blackbox2.Blackbox2;
import ru.yandex.inside.utils.Language;
import ru.yandex.misc.geo.Coordinates;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.random.Random2;

/**
 * @author tolmalev
 */
public class CoolLentaReminder {
    private static final Logger logger = LoggerFactory.getLogger(CoolLentaReminder.class);

    public static final String YEARS_ATTRIBUTE_NAME = TextProcessor.NUMBER_ATTRIBUTES_PREFIX + "years";
    public static final String THEME_ID_ATTRIBUTE_NAME = "theme_id";
    public static final String N_YEARS_AGO_SUBTYPE_PREFIX = "cool_lenta_years_ago_";
    public static final String THEMATIC_SUBTYPE_PREFIX = "cool_lenta_thematic_";

    private static final String EXPERIMENTAL_THEMES_EXPERIMENT_NAME = "disk_cool_lenta_experimental_themes_reminding";

    private final CoolLentaManager coolLentaManager;
    private final LentaManager lentaManager;
    private final LentaNotificationManager lentaNotificationManager;
    private final Cvi2tProcessor cvi2tProcessor;
    private final ExperimentsManager experimentsManager;
    private final ThemeDefinitionRegistry themeDefinitionRegistry;
    private final ListF<CoolLentaBlockGenerator> blockGenerators;
    private final CoolLentaBlockTitlesManager coolLentaBlockTitlesManager;
    private final CoolLentaBlockSendPushManager coolLentaBlockSendPushManager;
    private final UserTimezoneHelper userTimezoneHelper;
    private final CoolLentaConfigurationManager coolLentaConfigurationManager;
    private final DataApiManager dataApiManager;
    private final MordaPushManager mordaPushManager;
    private final Blackbox2 blackbox2;
    private final GncClient lentaGncClient;

    private final Duration minRepeatDuration;
    private final Duration minBlockLifetime;
    private final int lastBlocksCountForThematicCheck;

    public final DynamicProperty<Boolean> geoRemoveOther = new DynamicProperty<>("cool-lenta-geo-remove-no-geo", false);
    public final DynamicProperty<Double> geoMinPercent = new DynamicProperty<>("cool-lenta-geo-min-percent", 0.6);
    private final DynamicProperty<Boolean> enableSimilarityCheck =
            new DynamicProperty<>("cool-lenta-reminder-similarity-check-enable", false);
    private final DynamicProperty<Boolean> enableStopWordsCheck =
            new DynamicProperty<>("cool-lenta-reminder-stop-words-check-enable", false);
    private final DynamicProperty<Integer> usualBlocksWeight;
    private final DynamicProperty<Integer> nYearsBlocksWeight;
    private final DynamicProperty<Integer> thematicBlocksWeight;

    public CoolLentaReminder(CoolLentaManager coolLentaManager, LentaManager lentaManager,
            LentaNotificationManager lentaNotificationManager, Cvi2tProcessor cvi2tProcessor,
            ExperimentsManager experimentsManager, ThemeDefinitionRegistry themeDefinitionRegistry,
            ListF<CoolLentaBlockGenerator> blockGenerators, Duration minRepeatDuration, Duration minBlockLifetime,
            int lastBlocksCountForThematicCheck, CoolLentaBlockTitlesManager coolLentaBlockTitlesManager,
            CoolLentaBlockSendPushManager coolLentaBlockSendPushManager, UserTimezoneHelper userTimezoneHelper,
            CoolLentaConfigurationManager coolLentaConfigurationManager, DataApiManager dataApiManager,
            MordaPushManager mordaPushManager, Blackbox2 blackbox2, GncClient lentaGncClient)
    {
        this(coolLentaManager, lentaManager, lentaNotificationManager, cvi2tProcessor,
                experimentsManager, themeDefinitionRegistry, blockGenerators, minRepeatDuration,
                minBlockLifetime, 40, 20, 40, lastBlocksCountForThematicCheck, coolLentaBlockTitlesManager,
                coolLentaBlockSendPushManager, userTimezoneHelper, coolLentaConfigurationManager, dataApiManager,
                mordaPushManager, blackbox2, lentaGncClient);
    }

    public CoolLentaReminder(CoolLentaManager coolLentaManager, LentaManager lentaManager,
            LentaNotificationManager lentaNotificationManager, Cvi2tProcessor cvi2tProcessor,
            ExperimentsManager experimentsManager, ThemeDefinitionRegistry themeDefinitionRegistry,
            ListF<CoolLentaBlockGenerator> blockGenerators, Duration minRepeatDuration, Duration minBlockLifetime,
            int usualBlocksWeight, int nYearsBlocksWeight, int thematicBlocksWeight,
            int lastBlocksCountForThematicCheck, CoolLentaBlockTitlesManager coolLentaBlockTitlesManager,
            CoolLentaBlockSendPushManager coolLentaBlockSendPushManager, UserTimezoneHelper userTimezoneHelper,
            CoolLentaConfigurationManager coolLentaConfigurationManager, DataApiManager dataApiManager,
            MordaPushManager mordaPushManager, Blackbox2 blackbox2, GncClient lentaGncClient)
    {
        this.coolLentaManager = coolLentaManager;
        this.lentaManager = lentaManager;
        this.lentaNotificationManager = lentaNotificationManager;
        this.cvi2tProcessor = cvi2tProcessor;
        this.experimentsManager = experimentsManager;
        this.themeDefinitionRegistry = themeDefinitionRegistry;
        this.blockGenerators = blockGenerators;
        this.minRepeatDuration = minRepeatDuration;
        this.minBlockLifetime = minBlockLifetime;
        this.coolLentaBlockTitlesManager = coolLentaBlockTitlesManager;
        this.usualBlocksWeight =
                new DynamicProperty<>("cool-lenta-block-generation-type-weight-usual", usualBlocksWeight);
        this.nYearsBlocksWeight =
                new DynamicProperty<>("cool-lenta-block-generation-type-weight-nYears", nYearsBlocksWeight);
        this.thematicBlocksWeight =
                new DynamicProperty<>("cool-lenta-block-generation-type-weight-thematic", thematicBlocksWeight);
        this.lastBlocksCountForThematicCheck = lastBlocksCountForThematicCheck;
        this.coolLentaBlockSendPushManager = coolLentaBlockSendPushManager;
        this.userTimezoneHelper = userTimezoneHelper;
        this.coolLentaConfigurationManager = coolLentaConfigurationManager;
        this.dataApiManager = dataApiManager;
        this.mordaPushManager = mordaPushManager;
        this.blackbox2 = blackbox2;
        this.lentaGncClient = lentaGncClient;
    }

    public Option<LentaBlockRecord> generateBlock(PassportUid uid, DateTime userTimeNow) {
        return generateBlock(uid, userTimeNow, new BlockGenerationConfig(coolLentaConfigurationManager.platformsToPush.get(), true, true, 100500,
                100500, 100500, 100500, Option.empty(), Option.empty(), true, false));
    }

    public Option<LentaBlockRecord> generateBlockWithGeo(PassportUid uid, DateTime userTimeNow) {
        return generateBlock(uid, userTimeNow, new BlockGenerationConfig(coolLentaConfigurationManager.platformsToPush.get(), true, true, 100500,
                100500, 100500, 100500, Option.of(defaultGeoConfig()), Option.of(BlockGenerationType.USUAL), true, false));
    }

    public Option<LentaBlockRecord> generateNYearsAgoBlock(PassportUid uid, DateTime userTimeNow) {
        return generateBlock(uid, userTimeNow, new BlockGenerationConfig(coolLentaConfigurationManager.platformsToPush.get(), true, true, 100500,
                100500, 100500, 100500, Option.of(defaultGeoConfig()), Option.of(BlockGenerationType.N_YEARS), true, false));
    }

    public Option<LentaBlockRecord> generateThematicBlock(PassportUid uid, DateTime userTimeNow) {
        return generateBlock(uid, userTimeNow, new BlockGenerationConfig(coolLentaConfigurationManager.platformsToPush.get(), true, true, 100500,
                100500, 100500, 100500, Option.of(defaultGeoConfig()), Option.of(BlockGenerationType.THEMATIC), true, false));
    }

    public Option<LentaBlockRecord> generateExperimentalThematicBlock(PassportUid uid, DateTime userTimeNow) {
        return generateBlock(uid, userTimeNow, new BlockGenerationConfig(coolLentaConfigurationManager.platformsToPush.get(), true, true, 100500,
                100500, 100500, 100500, Option.of(defaultGeoConfig()), Option.of(BlockGenerationType.THEMATIC), true, true));
    }

    public Option<LentaBlockRecord> generateUsualBlock(PassportUid uid, DateTime userTimeNow) {
        return generateBlock(uid, userTimeNow, new BlockGenerationConfig(coolLentaConfigurationManager.platformsToPush.get(), true, true, 100500,
                100500, 100500, 100500, Option.of(defaultGeoConfig()), Option.of(BlockGenerationType.USUAL), true, false));
    }

    public Option<LentaBlockRecord> generateBlock(PassportUid uid, DateTime userTimeNow, BlockGenerationConfig config) {
        try {
            return selectBlockToSave(uid, userTimeNow, config)
                    .flatMapO(selectedBlock -> processSelectedBlock(uid, selectedBlock, config));
        } catch (UserBlockedException ignored) {
            return Option.empty();
        }
    }

    public LentaBlockRecord sendBlockNow(PassportUid uid, DateTime userTimeNow, int count) {
        return sendBlockNow(coolLentaManager.createRandomBlockForUser(uid, count, userTimeNow), uid, userTimeNow);
    }

    public LentaBlockRecord sendBlockNow(CoolLentaBlock block, PassportUid uid, DateTime userTimeNow) {
        MinimalCoolLentaBlock minimalCoolLentaBlock = block.toMinimalBlock();
        BlockTexts texts = coolLentaBlockTitlesManager.generateTextsForBlock(new CoolLentaBlockTitlesManager.TitleParameters(
                titleGenerationContextForBlock(minimalCoolLentaBlock), userTimeNow.toLocalDate(), Option.empty(), uid),
                 BlockTitlesType.DEFAULT).get();
        MapF<String, DataField> specific = getSpecificLentaBlockFields(BlockGenerationType.USUAL, minimalCoolLentaBlock,
                texts, coolLentaConfigurationManager.platformsToPush.get(), userTimeNow, Option.empty());

        String groupKey = "cool_lenta_" + Random2.R.nextAlnum(10);
        LentaRecordType lentaRecordType = LentaRecordType.PHOTO_SELECTION_BLOCK;
        DataApiUserId userId = new DataApiPassportUserId(uid);

        FindOrCreateResult result = lentaManager.findOrCreateBlock(userId,
                new LentaBlockCreateData(lentaRecordType, groupKey, specific),
                ActionInfo.internal(ActionSource.photoReminders()));
        lentaNotificationManager.scheduleReminderBlockNotification(new DataApiPassportUserId(uid), result.getRecordId(),
                true);
        return result.record;
    }

    Option<SaveBlockContext> selectBlockToSave(PassportUid uid, DateTime userTimeNow, BlockGenerationConfig config) {
        ListF<String> platforms = config.platforms;

        logger.debug("uid={}, Cool generation with config platforms={}, max_per_week={}, config={}", uid,
                platforms.mkString(","), config.maxPushesPerWeek, config);
        int usualBlocksWeight = this.usualBlocksWeight.get();
        int nYearsAgoBlocksWeight = this.nYearsBlocksWeight.get();
        int thematicBlocksWeight = this.thematicBlocksWeight.get();
        boolean userIsConnectedToThemesExperiment = experimentsManager.getFlags(uid.toUidOrZero().getUid())
                .containsTs(EXPERIMENTAL_THEMES_EXPERIMENT_NAME);
        if (config.isExperimental() &&
                !userIsConnectedToThemesExperiment) {
            throw new IllegalArgumentException(String.format("User %s is not a member of lenta experiment", uid));
        }
        if (config.blockType.isPresent()) {
            usualBlocksWeight = 0;
            nYearsAgoBlocksWeight = 0;
            thematicBlocksWeight = 0;
            BlockGenerationType type = config.blockType.get();
            if (type == BlockGenerationType.USUAL) {
                usualBlocksWeight = 1;
            } else if (type == BlockGenerationType.N_YEARS) {
                nYearsAgoBlocksWeight = 1;
            } else if (type == BlockGenerationType.THEMATIC) {
                thematicBlocksWeight = 1;
            }
        }

        // Прибавляем 1 минуту для того чтобы все сравнения говорили о том, что целое число дней уже прошло
        Instant now = Instant.now().plus(Duration.standardMinutes(1));
        ListF<MinimalCoolLentaBlock> allBlocks = coolLentaManager.getAllBlocks(uid)
                .filter(block -> new Duration(block.maxDate, now).isLongerThan(minBlockLifetime));
        if (!config.forceSendAndPush &&
                !coolLentaBlockSendPushManager.shouldDoSend(
                        new CoolLentaBlockSendPushManager.SendBlockConfiguration(allBlocks, config.maxDaysWithSendsPerWeek,
                                config.maxSendsPerDay, userTimezoneHelper.getUserTimezone(uid))
                )) {
            logger.debug("Don't send block today for uid {}", uid);
            return Option.empty();
        }

        ListF<MinimalCoolLentaBlock> blocksPart = allBlocks
                .filter(block -> !block.legacyLentaLastShowDate.isPresent()
                        || new Duration(block.legacyLentaLastShowDate.get(), now).isLongerThan(minRepeatDuration));
        if (blocksPart.isEmpty()) {
            logger.debug("No blocks found found for uid {}", uid);
            return Option.empty();
        }
        ListF<MinimalCoolLentaBlock> nYearsBlocks = blocksPart.filter(block -> isBlockNYearsAgo(block, userTimeNow));
        BlocksBucketByTypes usualAndThematicBlocks = mapBlocksByTypes(blocksPart, config.isExperimental(),
                userIsConnectedToThemesExperiment);
        ListF<MinimalCoolLentaBlock> thematicBlocks = usualAndThematicBlocks.getThematicBlocks();
        blocksPart = usualAndThematicBlocks.getUsualBlocks();
        ListF<String> recentlySentThemes = allBlocks.filter(b -> b.getLegacyLentaLastShowDate().isPresent())
                .sortedByDesc(block -> block.getLegacyLentaLastShowDate().get()).take(lastBlocksCountForThematicCheck)
                .map(MinimalCoolLentaBlock::getId).filter(this::isThematic)
                .map(this::getThemeIdFromThematicBlockId).unique().toList();
        for (int i = 0; i < 5 && blocksPart.isNotEmpty(); i++) {
            ListF<String> allAvailableThemes = thematicBlocks.map(MinimalCoolLentaBlock::getId)
                    .map(this::getThemeIdFromThematicBlockId).unique().toList();
            BlockGenerationTypeSelector selector = new BlockGenerationTypeSelector();
            selector.addBlockToSelect(blocksPart, BlockGenerationType.USUAL, usualBlocksWeight,
                    Option.of(this::selectRandomBlock));
            selector.addBlockToSelect(nYearsBlocks, BlockGenerationType.N_YEARS, nYearsAgoBlocksWeight,
                    Option.empty());
            selector.addBlockToSelect(thematicBlocks, BlockGenerationType.THEMATIC, thematicBlocksWeight,
                    Option.of(this::selectRandomThematicBlock));
            Option<BlockWithGenerationType> blockWithGenerationTypeO = selector.selectBlock();
            if (!blockWithGenerationTypeO.isPresent()) {
                return Option.empty();
            }
            BlockWithGenerationType blockWithGenerationType = blockWithGenerationTypeO.get();
            MinimalCoolLentaBlock block = blockWithGenerationType.getBlock();
            BlockGenerationType blockType = blockWithGenerationType.getType();
            Function1B<MinimalCoolLentaBlock> removingBlockPredicate = b -> b != block;
            if (!config.getBlockType().filter(type -> BlockGenerationType.THEMATIC == type).isPresent() &&
                    blockType == BlockGenerationType.THEMATIC &&
                    !isThematicBlockValid(block, allAvailableThemes, recentlySentThemes)) {
                thematicBlocks = thematicBlocks.filter(removingBlockPredicate);
                continue;
            }
            CoolLentaBlock extendedBlock = null;
            try  {
                extendedBlock = coolLentaManager.getExtendedBlock(uid, block, Option.empty());
            } catch (InconsistentCoolLentaBlockException e) {
                logger.info("The block is not found in MPFS. block_id: {}", block.getId());
            }
            if (extendedBlock == null || !blockIsValid(allBlocks, extendedBlock)) {
                if (blockType == BlockGenerationType.THEMATIC) {
                    thematicBlocks = thematicBlocks.filter(removingBlockPredicate);
                } else {
                    blocksPart = blocksPart.filter(removingBlockPredicate);
                    nYearsBlocks = nYearsBlocks.filter(removingBlockPredicate);
                }
                continue;
            }
            return Option.of(new SaveBlockContext(extendedBlock, blockWithGenerationType.getType(), userTimeNow, allBlocks));
        }
        return Option.empty();
    }

    private BlocksBucketByTypes mapBlocksByTypes(ListF<MinimalCoolLentaBlock> allBlocks, boolean experimentalOnly,
            boolean userIsConnectedToExperiment)
    {
        ListF<MinimalCoolLentaBlock> usualBlocks = Cf.arrayList();
        ListF<MinimalCoolLentaBlock> thematicBlocks = Cf.arrayList();
        ListF<String> experimentalThemes = themeDefinitionRegistry.getAll()
                .filter(ThemeDefinition::isExperimentalForReminding).map(ThemeDefinition::getName);
        for (MinimalCoolLentaBlock block : allBlocks) {
            if (block.getGenerationType().startsWith(ThematicBlocksGenerator.BLOCK_ID_PREFIX)) {
                boolean isExperimentalBlock = isExperimentalBlock(block, experimentalThemes);
                if (isExperimentalBlock && !userIsConnectedToExperiment) {
                    continue;
                }
                if (!experimentalOnly || isExperimentalBlock) {
                    thematicBlocks.add(block);
                }
                continue;
            }
            usualBlocks.add(block);
        }
        return new BlocksBucketByTypes(usualBlocks.unmodifiable(), thematicBlocks.unmodifiable());
    }

    private boolean isExperimentalBlock(MinimalCoolLentaBlock block, ListF<String> experimentalThemes) {
        return experimentalThemes.exists(theme -> block.getId().endsWith("_" + theme));
    }

    private MinimalCoolLentaBlock selectRandomBlock(ListF<MinimalCoolLentaBlock> blocks) {
        MapF<IntervalType, ListF<MinimalCoolLentaBlock>> blockByIntervalType = blocks
                .groupBy(MinimalCoolLentaBlock::getIntervalType);

        // При добавлении блока сперва случайным образом выбирается тип блока (дневной, недельный и др.), а затем уже случайный блок из них
        IntervalType intervalType = Random2.R.randomElement(blockByIntervalType.keys());
        return Random2.R.randomElement(blockByIntervalType.getTs(intervalType));
    }

    private MinimalCoolLentaBlock selectRandomThematicBlock(ListF<MinimalCoolLentaBlock> blocks) {
        MapF<String, ListF<MinimalCoolLentaBlock>> blocksByTheme = blocks.groupBy(this::getThemeIdFromThematicBlock);

        // При добавлении тематического блока сперва случайным образом выбирается тема блока, а затем уже случайный блок для выбранной темы
        String theme = Random2.R.randomElement(blocksByTheme.keys());
        return Random2.R.randomElement(blocksByTheme.getTs(theme));
    }

    public void logGeoInfo(PassportUid uid, MinimalCoolLentaBlock block, GeoSelectionConfig config) {
        try {
            CoolLentaBlock extendedBlock = coolLentaManager.getExtendedBlock(uid, block, Option.empty());

            Tuple2<ListF<CoolLentaFileItem>, ListF<CoolLentaFileItem>> t2 =
                    extendedBlock.items.partition(item ->
                            item.searchFileInfo.latitude.isPresent() && item.searchFileInfo.longitude.isPresent());

            ListF<CoolLentaFileItem> itemsWithGeo = t2._1;

            int countWithGeo = itemsWithGeo.size();

            if (countWithGeo == 0) {
                logger.debug("GEO BLOCK INFO: total={}, with_geo={}, geo_name=failed_no_geo_at_all",
                        extendedBlock.items.size(),
                        countWithGeo);
                return;
            }

            Tuple2<IntervalType, DateTime> intervalWithDate = BlockGeneratorUtils.getInterval(itemsWithGeo);

            if (intervalWithDate._1 == block.getIntervalType()) {
                ListF<Coordinates> coordinates = itemsWithGeo
                        .map(i -> new Coordinates(i.searchFileInfo.latitude.get(), i.searchFileInfo.longitude.get()));
                TitleGenerationContext generationContext = titleGenerationContextForBlock(block)
                        .withAttribute(GeoCoolLentaBlockTitlesGenerator.COORDINATES_ATTRIBUTE_NAME, coordinates)
                        .withAttribute(GeoCoolLentaBlockTitlesGenerator.LEGACY_TITLE_IS_GEO_ATTRIBUTE_NAME, config.isLegacyTitleWithGeo());

                BlockTexts texts = coolLentaBlockTitlesManager.generateTextsForBlock(new CoolLentaBlockTitlesManager.TitleParameters(
                        generationContext, LocalDate.now(), Option.empty(), uid
                ), BlockTitlesType.GEO).getOrNull();

                if (texts == null) {
                    logger.debug("GEO BLOCK INFO: total={}, with_geo={}, geo_name=failed_cant_generate",
                            extendedBlock.items.size(),
                            countWithGeo);
                } else {
                    logger.debug("GEO BLOCK INFO: total={}, with_geo={}, geo_name={}",
                            extendedBlock.items.size(),
                            countWithGeo,
                            texts.coverTitle.get(Language.RUSSIAN));
                }
            } else {
                logger.debug("GEO BLOCK INFO: total={}, with_geo={}, geo_name=failed_changed_interval",
                        extendedBlock.items.size(),
                        countWithGeo);
            }

        } catch (Exception e) {
            logger.debug("Some files are missing. id={}", block.id);
        }
    }

    private boolean isBlockNYearsAgo(MinimalCoolLentaBlock block, DateTime userTimeNow) {
        if (block.getIntervalType() != IntervalType.ONE_DAY) {
            return false;
        }
        LocalDate userDateNow = userTimeNow.toLocalDate();
        LocalDate blockDate = block.getIntervalStart().toLocalDate();
        if (userDateNow.isEqual(blockDate)) {
            return false;
        }
        return blockDate.getMonthOfYear() == userDateNow.getMonthOfYear() &&
                blockDate.getDayOfMonth() == userDateNow.getDayOfMonth();
    }

    private boolean blockIsValid(ListF<MinimalCoolLentaBlock> allBlocks, CoolLentaBlock extendedBlock) {
        MinimalCoolLentaBlock minimalBlock = extendedBlock.toMinimalBlock();
        Option<MinimalCoolLentaBlock> lastShownBlockO =
                allBlocks.filter(b -> b.getLegacyLentaLastShowDate().isPresent()).maxByO(b ->
                        b.getLegacyLentaLastShowDate().get());

        if(lastShownBlockO.isPresent()) {
            GenerationInterval lastGenerationInterval = lastShownBlockO.get().getGenerationInterval();
            GenerationInterval currentGenerationInterval = minimalBlock.getGenerationInterval();

            if (lastGenerationInterval.getDistance(currentGenerationInterval).getStandardDays() < 30) {
                logger.debug("Don't add block for interval {}. Too near to previous {}", currentGenerationInterval,
                        lastGenerationInterval);
                return false;
            }
        }
        if (blockContainsItemsWithoutI2tVector(extendedBlock)) {
            return true;
        }
        return blockIsValidAfterFiltering(extendedBlock);
    }

    private boolean blockContainsItemsWithoutI2tVector(CoolLentaBlock block) {
        return block.getItems().exists(item -> !CoolLentaManager.hasI2tVector(item));
    }

    private boolean isThematicBlockValid(MinimalCoolLentaBlock block, ListF<String> allAvailableThemes,
            ListF<String> recentlySentThemes) {
        if (allAvailableThemes.size() < 2) {
            logger.debug("The user has only one available theme {}", allAvailableThemes);
            return true;
        }
        if (recentlySentThemes.containsTs(getThemeIdFromThematicBlockId(block.getId()))) {
            logger.debug("Thematic block {} is invalid because a block with the same theme was sent recently. Recent themes: {}",
                    block.getId(), recentlySentThemes);
            return false;
        }
        logger.debug("Thematic block {} is valid because a block with the same theme was not sent recently. Recent themes: {}",
                block.getId(), recentlySentThemes);
        return true;
    }

    private boolean isThematic(String blockId) {
        return blockId.startsWith(ThematicBlocksGenerator.BLOCK_ID_PREFIX);
    }

    private boolean blockIsValidAfterFiltering(CoolLentaBlock block) {
        ListF<CoolLentaFileItem> items = Cf.toArrayList(block.getItems());
        if (!blockSavesIntervalAfterSimilarityCheck(block, items)) {
            return false;
        }
        return blockSavesIntervalAfterStopWordsCheck(block, items);
    }

    private boolean blockSavesIntervalAfterSimilarityCheck(CoolLentaBlock block,
            ListF<CoolLentaFileItem> itemsToCheck)
    {
        return blockSavesIntervalAfterCheck(block, itemsToCheck, enableSimilarityCheck,
                items -> BlockGeneratorUtils.removeSimilarItems(items, coolLentaManager.similarityThreshold.get()),
                "similarity");
    }

    private boolean blockSavesIntervalAfterStopWordsCheck(CoolLentaBlock block, ListF<CoolLentaFileItem> itemsToCheck) {
        return blockSavesIntervalAfterCheck(block, itemsToCheck, enableStopWordsCheck,
                items -> BlockGeneratorUtils.removeItemsByStopWords(items, coolLentaManager.getStopWords(), cvi2tProcessor),
                "stop words");
    }

    private boolean blockSavesIntervalAfterCheck(CoolLentaBlock block, ListF<CoolLentaFileItem> itemsToCheck,
            DynamicProperty<Boolean> enableCheck, Function<ListF<CoolLentaFileItem>, ListF<CoolLentaFileItem>> itemsRemover,
            String checkName)
    {
        String blockId = block.getId();
        if (!enableCheck.get()) {
            logger.debug("The block {} has been passed the {} filtering (The filtering is disabled)",
                    blockId, checkName);
            return true;
        }
        logger.debug("The size of block {} is {}", blockId, itemsToCheck.size());
        ListF<CoolLentaFileItem> itemsToRemove = itemsRemover.apply(itemsToCheck);
        if (itemsToRemove.isEmpty()) {
            logger.debug("The block {} has been passed the {} filtering (No files to remove)",
                    blockId, checkName);
            return true;
        }
        logger.debug("{} items has been removed from block {} during {} filtering", itemsToRemove.size(), blockId,
                checkName);
        CoolLentaBlockGenerator blockGenerator = blockGenerators
                .find(generator -> blockId.startsWith(generator.generatorIdPrefix()))
                .getOrThrow(() -> new IllegalStateException(String.format("Cannot find generator type for block '%s'",
                        blockId)));
        if (itemsToCheck.size() < blockGenerator.getMinSizeForBlock()) {
            logger.debug("The block's {} size has become lower than minimal during {} filtering. New size {}",
                    blockId, checkName, itemsToCheck.size());
            return false;
        }
        if (BlockGeneratorUtils.getInterval(itemsToCheck)._1 == block.toMinimalBlock().getIntervalType()) {
            logger.debug("The block {} has not changed the time interval after {} filtering. Block size {}", blockId,
                    checkName, itemsToCheck.size());
            return true;
        }
        logger.debug("The block {} has changed the time interval after {} filtering. Block size {}", blockId, checkName,
                itemsToCheck.size());
        return false;
    }

    public Option<ProcessingBlockContext> saveBlockToLenta(PassportUid uid, BlockGenerationConfig config, SaveBlockContext context) {
        CoolLentaBlock extendedBlock = context.getBlock();
        logger.debug("Save block to lenta for uid={}, block={}, config={}", uid, extendedBlock, config);

        Option<BlockTexts> textsO = Option.empty();
        ListF<CoolLentaFileItem> sourceItems = Cf.toArrayList(extendedBlock.getItems());
        SaveToLentaData saveToLentaData = new SaveToLentaData(extendedBlock.toMinimalBlock(), extendedBlock);
        boolean blockContainsItemsWithEmptyI2tVector = sourceItems.exists(item -> !CoolLentaManager.hasI2tVector(item));

        if (!blockContainsItemsWithEmptyI2tVector && enableSimilarityCheck.get()) {
            BlockGeneratorUtils.removeSimilarItems(sourceItems, coolLentaManager.similarityThreshold.get());
            CoolLentaBlock block = saveToLentaData.getExtendedBlock().withItems(sourceItems);
            saveToLentaData = new SaveToLentaData(block.toMinimalBlock(), block);
        }

        if (!blockContainsItemsWithEmptyI2tVector && enableStopWordsCheck.get()) {
            BlockGeneratorUtils.removeItemsByStopWords(sourceItems, coolLentaManager.getStopWords(), cvi2tProcessor);
            CoolLentaBlock block = saveToLentaData.getExtendedBlock().withItems(sourceItems);
            saveToLentaData = new SaveToLentaData(block.toMinimalBlock(), block);
        }

        if (config.geoConfig.isPresent() && context.getType() == BlockGenerationType.USUAL) {
            int totalCount = sourceItems.size();
            int countWithGeo = sourceItems.count(i -> i.searchFileInfo.hasGeo());

            GeoSelectionConfig geoConfig = config.geoConfig.get();

            if (countWithGeo > 0) {
                ListF<DateTime> etimesWithGeo = sourceItems
                        .filter(i -> i.searchFileInfo.hasGeo())
                        .map(CoolLentaFileItem::getUserEtime).sorted();

                DateTime firstWithGeo = etimesWithGeo.first();
                DateTime lastWithGeo = etimesWithGeo.last();

                // remove items before first and after last image with geo
                ListF<CoolLentaFileItem> items = sourceItems
                        .filter(i -> !i.getUserEtime().isBefore(firstWithGeo) && !i.getUserEtime().isAfter(lastWithGeo));

                if (geoConfig.removeWithoutGeo) {
                    // remove all without geo
                    items = items.filter(i -> i.searchFileInfo.hasGeo());
                }

                if (items.size() >= geoConfig.minFinalCount) {
                    Tuple2<IntervalType, DateTime> intervalWithDate = BlockGeneratorUtils.getInterval(items);

                    if (intervalWithDate._1 == saveToLentaData.getMinimalBlock().getIntervalType()) {
                        if (countWithGeo >= geoConfig.minGeoPercent * items.size()) {
                            Option<DateTime> photosliceDate = Option
                                    .when(saveToLentaData.getMinimalBlock().getIntervalType() != IntervalType.YEAR,
                                    firstWithGeo);
                            CoolLentaBlock geoExtendedBlock = extendedBlock
                                    .withItemsAndPhotosliceDate(items, photosliceDate);
                            ListF<Coordinates> coordinates = items.filter(i -> i.searchFileInfo.hasGeo())
                                    .map(i -> new Coordinates(i.searchFileInfo.latitude.get(),
                                            i.searchFileInfo.longitude.get()));
                            TitleGenerationContext generationContext =
                                    titleGenerationContextForBlock(saveToLentaData.getMinimalBlock())
                                            .withAttribute(GeoCoolLentaBlockTitlesGenerator.COORDINATES_ATTRIBUTE_NAME, coordinates)
                                            .withAttribute(GeoCoolLentaBlockTitlesGenerator.LEGACY_TITLE_IS_GEO_ATTRIBUTE_NAME, geoConfig.isLegacyTitleWithGeo());

                            textsO = coolLentaBlockTitlesManager.generateTextsForBlock(new CoolLentaBlockTitlesManager.TitleParameters(
                                    generationContext, context.getUserTimeNow().toLocalDate(), Option.empty(), uid
                                    ),
                                    BlockTitlesType.GEO);

                            if (textsO.isPresent()) {
                                saveToLentaData = new SaveToLentaData(geoExtendedBlock.toMinimalBlock(), geoExtendedBlock);
                                logger.debug("Removed photos from block to generate geo title. " +
                                                "total={}, after_filter={}, with_geo={}", totalCount, items.size(),
                                        countWithGeo);
                            } else {
                                logger.debug("Failed to generate geo text even after removing some photos. Get default");
                            }
                        } else {
                            logger.debug("Not enough images with geodata for removing images from block. " +
                                            "total={}, after_filter={}, with_geo={}", totalCount, items.size(), countWithGeo);
                        }
                    } else {
                        logger.debug("Can't remove images without geo. It changes block interval. " +
                                        "total={}, after_filter={}, with_geo={}, current={}, new={}", totalCount,
                                items.size(), countWithGeo, saveToLentaData.getMinimalBlock().getIntervalType(),
                                intervalWithDate._1);
                    }
                } else {
                    logger.debug("Not enough files after filtering. total={}, after_filter={}", totalCount, items.size());
                }
            } else {
                logger.debug("Not files with geo, total={}", totalCount);
            }
        }

        MinimalCoolLentaBlock minimalCoolLentaBlock = saveToLentaData.getMinimalBlock();
        if (context.getType() == BlockGenerationType.N_YEARS) {
            int nYearsAgoCount = Years.yearsBetween(saveToLentaData.getMinimalBlock().getUserMinDate().toLocalDate(),
                    context.getUserTimeNow().toLocalDate()).getYears();
            TitleGenerationContext titleGenerationContext = titleGenerationContextForBlock(minimalCoolLentaBlock)
                    .withAttribute(YEARS_ATTRIBUTE_NAME, nYearsAgoCount);
            textsO = coolLentaBlockTitlesManager.generateTextsForBlock(new CoolLentaBlockTitlesManager.TitleParameters(
                    titleGenerationContext, context.getUserTimeNow().toLocalDate(), Option.empty(), uid
            ), BlockTitlesType.N_YEARS);
        } else if (context.getType() == BlockGenerationType.THEMATIC) {
            String themeId = getThemeIdFromThematicBlock(saveToLentaData.getMinimalBlock());
            ThemeDefinition themeDefinition = themeDefinitionRegistry.getO(themeId)
                    .getOrThrow("Cannot find theme");
            TermDefinition termDefinition = themeDefinition.getForms();
            TitleGenerationContext titleGenerationContext = titleGenerationContextForBlock(minimalCoolLentaBlock)
                    .withTerm(termDefinition)
                    .withAttribute(THEME_ID_ATTRIBUTE_NAME, themeId);
            textsO = coolLentaBlockTitlesManager.generateTextsForBlock(new CoolLentaBlockTitlesManager.TitleParameters(
                    titleGenerationContext, context.getUserTimeNow().toLocalDate(), Option.of(themeId), uid
            ), BlockTitlesType.THEMATIC);
        }
        // not generated by geo and not is N years ago or thematic block - get default
        if (!textsO.isPresent()) {
            TitleGenerationContext titleGenerationContext = titleGenerationContextForBlock(minimalCoolLentaBlock);
            textsO = coolLentaBlockTitlesManager.generateTextsForBlock(new CoolLentaBlockTitlesManager.TitleParameters(
                    titleGenerationContext, context.getUserTimeNow().toLocalDate(), Option.empty(), uid
            ), BlockTitlesType.DEFAULT);
        }
        BlockTexts texts = textsO.get();
        int itemsSize = saveToLentaData.getMinimalBlock().getResourceIds().size();
        if (itemsSize >
                Cf.toSet(saveToLentaData.getMinimalBlock().getResourceIds().map(MpfsResourceId::serialize)).size()) {
            logger.debug("The block {} contains duplicates", saveToLentaData.getMinimalBlock().getId());
            return Option.empty();
        }
        ListF<String> platforms = config.platforms;
        Option<String> mordaBlockId = Option.empty();
        Option<String> desktopUrl = Option.empty();
        boolean desktopEnabled = config.platforms.containsTs(CoolLentaConfigurationManager.DESKTOP);
        if (desktopEnabled || coolLentaConfigurationManager.isGncPushesEnabled()) {
            String blockId = saveBlockForMorda(saveToLentaData.getMinimalBlock(),
                    getSpecificLentaMordaBlockFields(context.getType(), saveToLentaData.getMinimalBlock(), texts), uid);
            mordaBlockId = Option.of(blockId);
            if (desktopEnabled) {
                desktopUrl = Option.of(getDesktopUrl(blockId, uid));
            }
        }
        MapF<String, DataField> specific = getSpecificLentaBlockFields(context.getType(), saveToLentaData.getMinimalBlock(), texts,
                platforms, context.userTimeNow, desktopUrl);

        String groupKey = "cool_lenta_" + Random2.R.nextAlnum(10);
        LentaRecordType lentaRecordType = LentaRecordType.PHOTO_SELECTION_BLOCK;
        DataApiUserId userId = new DataApiPassportUserId(uid);
        FindOrCreateResult result = lentaManager.findOrCreateBlock(userId,
                new LentaBlockCreateData(lentaRecordType, groupKey, specific),
                ActionInfo.internal(ActionSource.photoReminders()));

        return Option.of(new ProcessingBlockContext(result.record, texts, context.getType(), saveToLentaData.getMinimalBlock(), mordaBlockId));
    }

    @SneakyThrows
    private String getDesktopUrl(String mordaBlockId, PassportUid uid) {
        return new URIBuilder(coolLentaConfigurationManager.getBaseMordaBlockLink(mordaBlockId))
                .setParameter("uid", uid.toString())
                .setParameter("login", blackbox2.oldHelper().loginByUid(uid).orElse(""))
                .setParameter("from", "desktop").build().toString();
    }

    private Option<LentaBlockRecord> processSelectedBlock(PassportUid uid, SaveBlockContext saveBlockContext,
            BlockGenerationConfig config)
    {
        Option<ProcessingBlockContext> processingBlockContext = saveBlockToLenta(uid, config, saveBlockContext);
        if (!processingBlockContext.isPresent()) {
            return Option.empty();
        }
        ProcessingBlockContext context = processingBlockContext.get();
        LentaBlockRecord record = context.getLentaBlockRecord();
        Instant now = Instant.now();
        Option<Instant> sendDate = Option.of(now);
        Option<Instant> pushDate = Option.empty();
        ListF<MinimalCoolLentaBlock> allBlocks = saveBlockContext.getAllBlocks();
        if (config.forceSendAndPush || (config.sendPush &&
                coolLentaBlockSendPushManager.shouldDoPush(
                        new CoolLentaBlockSendPushManager.PushBlockConfiguration(
                                allBlocks, config.getMaxPushesPerWeek(), config.getMaxPushesPerDay(),
                                config.getMaxDaysWithSendsPerWeek(), config.getMaxSendsPerDay(), userTimezoneHelper.getUserTimezone(uid)
                        )
                ))) {
            lentaNotificationManager.scheduleReminderBlockNotification(new DataApiPassportUserId(uid), record.id,
                    config.sendPushNow);
            pushDate = Option.of(now);
        }
        coolLentaManager.updateLegacyLentaDates(uid, saveBlockContext.getBlock().getId(), sendDate, pushDate);
        return Option.of(record);
    }

    private String getThemeIdFromThematicBlock(MinimalCoolLentaBlock block) {
        return getThemeIdFromThematicBlockId(block.getId());
    }

    private String getThemeIdFromThematicBlockId(String blockId) {
        String[] blockIdParts = blockId.split(CoolLentaBlockGenerator.BLOCK_ID_SEPARATOR);
        return blockIdParts[blockIdParts.length - 1];
    }

    @NotNull
    private TitleGenerationContext titleGenerationContextForBlock(MinimalCoolLentaBlock block) {
        Random2 R = new Random2(block.getBestResourceId().hashCode());

        GenerationInterval interval = block.getGenerationInterval();

        IntervalType intervalType = interval.type;
        DateTime intervalStart = interval.start;

        return new TitleGenerationContext(R, intervalType, intervalStart);
    }

    public static MapF<String, DataField> getSpecificLentaBlockFields(BlockGenerationType generationType, MinimalCoolLentaBlock block,
                                                                      BlockTexts texts, ListF<String> platforms, DateTime userTimeNow,
            Option<String> desktopUrl)
    {
        GenerationInterval interval = block.getGenerationInterval();

        DateTime intervalStart = interval.start;
        DateTime intervalEnd = interval.end;

        String subtype = genSubtype(generationType, texts, block);

        MapF<String, DataField> specific = Cf.toHashMap(Cf.list(
                PhotoSelectionFields.RESOURCE_IDS.toData(block.resourceIds.map(MpfsResourceId::serialize)),
                PhotoSelectionFields.BEST_RESOURCE_ID.toData(block.bestResourceId.serialize()),

                PhotoSelectionFields.INTERVAL_START.toData(intervalStart.toInstant()),
                PhotoSelectionFields.INTERVAL_END.toData(intervalEnd.toInstant()),
                PhotoSelectionFields.USER_TIMEZONE_ID.toData(block.userTimezoneId),
                PhotoSelectionFields.USER_GENERATION_TIME.toData(userTimeNow.toInstant()),

                PhotoSelectionFields.NOTIFICATION_TYPE.toData(PhotoSelectionNotificationType.NOTIFICATION_COOL_LENTA),
                PhotoSelectionFields.SUBTYPE.toData(subtype),
                PhotoSelectionFields.ENABLED_PLATFORMS.toData(platforms.mkString(","))
        ).plus(PhotoSelectionFields.TITLE.toData(I18nValue.of(texts.title))
        ).plus(PhotoSelectionFields.COVER_TITLE.toData(I18nValue.of(texts.coverTitle))
        ).plus(PhotoSelectionFields.COVER_SUBTITLE.toData(I18nValue.of(texts.coverSubtitle))
        ).plus(PhotoSelectionFields.BUTTON_TEXT.toData(I18nValue.of(texts.buttonText))));

        if (block.photosliceDate.isPresent()) {
            specific.put(PhotoSelectionFields.PHOTOSLICE_DATE.toData(block.photosliceDate.get()));
        }
        if (texts.geoIds.isNotEmpty()) {
            specific.put(PhotoSelectionFields.GEO_REGION_IDS.toData(texts.geoIds));
        }
        desktopUrl.ifPresent(url -> specific.put(PhotoSelectionFields.DESKTOP_URL.toData(url)));
        return specific;
    }

    public static MapF<String, DataField> getSpecificLentaMordaBlockFields(BlockGenerationType generationType,
            MinimalCoolLentaBlock block, BlockTexts texts) {

        String subtype = genSubtype(generationType, texts, block);

        MapF<String, DataField> specific = Cf.toHashMap(Cf.list(
                MordaPhotoSelectionFields.RESOURCE_IDS.toData(block.resourceIds.map(MpfsResourceId::serialize)),
                MordaPhotoSelectionFields.BEST_RESOURCE_ID.toData(block.bestResourceId.serialize()),
                MordaPhotoSelectionFields.USER_TIMEZONE_ID.toData(block.userTimezoneId),
                MordaPhotoSelectionFields.GENERATION_TYPE.toData(block.generationType),
                MordaPhotoSelectionFields.MIN_DATE.toData(block.minDate),
                MordaPhotoSelectionFields.MAX_DATE.toData(block.maxDate),
                MordaPhotoSelectionFields.SUBTYPE.toData(subtype),
                MordaPhotoSelectionFields.MTIME.toData(Instant.now())
        ).plus(MordaPhotoSelectionFields.TITLE.toData(I18nValue.of(texts.coverTitle))
        ).plus(MordaPhotoSelectionFields.SUBTITLE.toData(I18nValue.of(texts.coverSubtitle))
        ).plus(MordaPhotoSelectionFields.PHOTOSLICE_LINK_TEXT.toData(I18nValue.of(texts.buttonText))));
        block.photosliceDate.ifPresent(photosliceDate -> specific.put(MordaPhotoSelectionFields.PHOTOSLICE_DATE.toData(photosliceDate)));
        return specific;
    }

    private String saveBlockForMorda(MinimalCoolLentaBlock block, MapF<String, DataField> data, PassportUid uid) {
        logger.debug("Saving block for morda. uid={}, id={}", uid, block.getId());
        int maxMordaBlocksCount = coolLentaConfigurationManager.maxMordaBlocksCount.get();

        String recordId = OrderedUUID.generateOrderedUUID();
        Instant now = Instant.now();

        Database mordaDatabase = coolLentaManager.getOrCreateMordaBlocksDatabase(uid);

        // we need it to remove same block if present
        ListF<DataRecord> existingMordaBlocks = dataApiManager.getRecords(mordaDatabase.spec(), RecordsFilter.DEFAULT
                .withCollectionId(CoolLentaManager.MORDA_BLOCKS_COLLECTION_ID)
                .withRecordOrder(ByIdRecordOrder.RECORD_ID_DESC)
        );

        MapF<String, String> idByAllBlocksId
                = existingMordaBlocks
                .filter(r -> CoolLentaBlockFields.ALL_BLOCKS_ID.getO(r).isPresent())
                .toMap(r -> Tuple2.tuple(CoolLentaBlockFields.ALL_BLOCKS_ID.get(r), r.id.recordId()));
        MapF<String, DataRecord> blockByAllBlocksId =
                existingMordaBlocks.filter(r -> CoolLentaBlockFields.ALL_BLOCKS_ID.getO(r).isPresent())
                        .toMapMappingToKey(CoolLentaBlockFields.ALL_BLOCKS_ID::get);
        MapF<String, DataRecord> blockByBlocksId = existingMordaBlocks.toMapMappingToKey(DataRecord::getRecordId);

        ListF<RecordChange> changes = Cf.arrayList(RecordChange.set(
                CoolLentaManager.MORDA_BLOCKS_COLLECTION_ID,
                recordId,
                data
        ));
        ListF<CoolLentaBlockEvent> events = Cf.arrayList(new CoolLentaBlockEvent(
                uid, CoolLentaEventType.MORDA_BLOCKS_ADD, block.toMordaBlock(recordId, now)
        ));

        int targetSize = existingMordaBlocks.size() + 1;

        if (blockByAllBlocksId.containsKeyTs(block.getId())) {
            // Found block with save id in list of existing. Remove it.
            String mordaBlockId = idByAllBlocksId.getTs(block.getId());
            changes.add(CoolLentaModelUtils.removeMordaBlockChange(mordaBlockId));
            events.add(new CoolLentaBlockEvent(
                    uid, CoolLentaEventType.MORDA_BLOCKS_DELETE,
                    CoolLentaModelUtils.recordToMordaBlock(blockByAllBlocksId.getTs(block.id))
            ));
            targetSize--;
        }

        if (targetSize > maxMordaBlocksCount) {
            // max limit reached - remove blocks
            logger.debug("targetSize={} maxMordaBlocksCount={} existingMordaBlocksCount={}", targetSize, maxMordaBlocksCount, existingMordaBlocks.size());
            ListF<DataRecord> recordsToRemove = existingMordaBlocks
                    .sortedBy(DataRecord::getRecordId)
                    .take(targetSize - maxMordaBlocksCount);
            ListF<String> recordIdsToRemove = recordsToRemove.map(DataRecord::getRecordId);
            logger.debug("Morda block ids to remove: {}", recordIdsToRemove);
            changes.addAll(recordIdsToRemove.map(CoolLentaModelUtils::removeMordaBlockChange));
            events.addAll(recordIdsToRemove
                    .map(recordIdToRemove -> new CoolLentaBlockEvent(uid, CoolLentaEventType.MORDA_BLOCKS_DELETE,
                            CoolLentaModelUtils.recordToMordaBlock(blockByBlocksId.getTs(recordIdToRemove)))));
        }

        long rev = dataApiManager.applyDelta(mordaDatabase, RevisionCheckMode.PER_RECORD, new Delta(changes)).rev;
        CoolLentaEventLogger.log(events.map(event -> event.withRevision(rev)));
        coolLentaManager.updateLastShowDate(uid, block.id, Instant.now());
        return recordId;
    }

    @NotNull
    private static String genSubtype(BlockGenerationType type, BlockTexts texts, MinimalCoolLentaBlock block) {
        switch (type) {
            case USUAL:
                return "cool_lenta_"
                        + ((texts.geoIds.isNotEmpty()) ? "geo" : "")
                        + "interval_"
                        + block.getIntervalType();
            case N_YEARS:
                return N_YEARS_AGO_SUBTYPE_PREFIX + block.getIntervalType() + "_"
                        + texts.context.getAttributes().getOrThrow(YEARS_ATTRIBUTE_NAME, "No attribute for years in context");
            case THEMATIC:
                return THEMATIC_SUBTYPE_PREFIX
                        + block.getIntervalType()
                        + "_"
                        + texts.context.getAttributes().getOrThrow(THEME_ID_ATTRIBUTE_NAME, "No attribute for theme id in context");
            default:
                throw new IllegalStateException("Unknown gen type " + type);
        }
    }

    public GeoSelectionConfig defaultGeoConfig() {
        return new GeoSelectionConfig(
                geoRemoveOther.get(),
                false,
                geoMinPercent.get(),
                5
        );
    }

    @Data
    @AllArgsConstructor
    public static class BlockGenerationConfig {
        public final ListF<String> platforms;
        public final boolean sendPush;
        public final boolean sendPushNow;
        public final int maxPushesPerWeek;
        public final int maxDaysWithSendsPerWeek;
        public final int maxSendsPerDay;
        public final int maxPushesPerDay;

        public final Option<GeoSelectionConfig> geoConfig;
        public final Option<BlockGenerationType> blockType;
        public final boolean forceSendAndPush;
        public final boolean experimental;
    }

    @Data
    @AllArgsConstructor
    public static class GeoSelectionConfig {
        public final boolean removeWithoutGeo;
        public final boolean legacyTitleWithGeo;
        public final double minGeoPercent;
        public final int minFinalCount;
    }

    @Data
    private static class SaveToLentaData {
        private final MinimalCoolLentaBlock minimalBlock;
        private final CoolLentaBlock extendedBlock;
    }

    public enum BlockGenerationType {
        USUAL,
        N_YEARS,
        THEMATIC
    }

    static class BlockGenerationTypeSelector {

        private static final Logger logger = LoggerFactory.getLogger(BlockGenerationTypeSelector.class);

        private int exclusiveMax = 0;

        private final MapF<BlockGenerationType, BlockGenerationTypeProbabilityBorders> weightBordersMap = Cf.hashMap();

        private final MapF<BlockGenerationType, ListF<MinimalCoolLentaBlock>> blockTypesToBlocksMap = Cf.hashMap();

        private final MapF<BlockGenerationType, Integer> blockGenerationTypesWeights = Cf.hashMap();

        private final MapF<BlockGenerationType, Function<ListF<MinimalCoolLentaBlock>, MinimalCoolLentaBlock>>
                additionalSelectionFunctionsForBlocksMap = Cf.hashMap();

        public Option<BlockWithGenerationType> selectBlock() {
            if (exclusiveMax == 0) {
                return Option.empty();
            }
            int randomValue = Random2.R.nextInt(exclusiveMax);
            return weightBordersMap.filterValues(blockGenerationTypeProbabilityBorders ->
                    blockGenerationTypeProbabilityBorders.getInclusiveMin() <= randomValue &&
                            blockGenerationTypeProbabilityBorders.getExclusiveMax() > randomValue)
                    .keySet().toList().firstO().map(this::selectBlockByType);
        }

        public void addBlockToSelect(ListF<MinimalCoolLentaBlock> blocks, BlockGenerationType type, int typeWeight,
                Option<Function<ListF<MinimalCoolLentaBlock>, MinimalCoolLentaBlock>> additionalSelectionFunction)
        {
            if (blocks.isEmpty()) {
                return;
            }
            if (weightBordersMap.containsKeyTs(type)) {
                throw new IllegalArgumentException(String
                        .format("Block generation type selector contains the blocks with type `%s` already", type));
            }
            blockGenerationTypesWeights.put(type, typeWeight);
            int inclusiveMin = exclusiveMax;
            int typeProbability = blockGenerationTypesWeights.getOrElse(type, 0);
            exclusiveMax += typeProbability;
            weightBordersMap.put(type, new BlockGenerationTypeProbabilityBorders(inclusiveMin, exclusiveMax));
            blockTypesToBlocksMap.put(type, blocks);
            Function<ListF<MinimalCoolLentaBlock>, MinimalCoolLentaBlock> selectionOptionsFunction =
                    additionalSelectionFunction.getOrElse(Random2.R::randomElement);
            additionalSelectionFunctionsForBlocksMap.put(type, selectionOptionsFunction);
        }

        private BlockWithGenerationType selectBlockByType(BlockGenerationType type) {
            BlockWithGenerationType result = new BlockWithGenerationType(type,
                    additionalSelectionFunctionsForBlocksMap.getTs(type).apply(blockTypesToBlocksMap.getTs(type)));
            logger.debug("Block Type selected is {}. Available types {}. Weights map {}. Counts of blocks for types {}",
                    result.getType(), weightBordersMap.keys(), weightBordersMap,
                    blockTypesToBlocksMap.mapValues(ListF::size));
            return result;
        }
    }

    @Data
    private static class BlockGenerationTypeProbabilityBorders {
        private final int inclusiveMin;
        private final int exclusiveMax;
    }

    @Data
    private static class BlockWithGenerationType {
        private final BlockGenerationType type;
        private final MinimalCoolLentaBlock block;
    }

    @Data
    public static class SaveBlockContext {
        private final CoolLentaBlock block;
        private final BlockGenerationType type;
        private final DateTime userTimeNow;
        private final ListF<MinimalCoolLentaBlock> allBlocks;
    }

    @Data
    private static class BlocksBucketByTypes {
        private final ListF<MinimalCoolLentaBlock> usualBlocks;
        private final ListF<MinimalCoolLentaBlock> thematicBlocks;
    }

    @Data
    private static class ProcessingBlockContext {
        private final LentaBlockRecord lentaBlockRecord;
        private final BlockTexts texts;
        private final BlockGenerationType blockGenerationType;
        private final MinimalCoolLentaBlock minimalCoolLentaBlock;
        private final Option<String> mordaBlockIdO;
    }
}
