package ru.yandex.chemodan.app.lentaloader.reminder;

import org.joda.time.DateTime;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Test;
import org.mockito.ArgumentMatcher;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.dataapi.core.manager.DataApiManager;
import ru.yandex.chemodan.app.lentaloader.cool.CoolLentaConfigurationManager;
import ru.yandex.chemodan.app.lentaloader.cool.CoolLentaManager;
import ru.yandex.chemodan.app.lentaloader.cool.ExperimentMembersByOverrideManager;
import ru.yandex.chemodan.app.lentaloader.cool.MordaPushManager;
import ru.yandex.chemodan.app.lentaloader.cool.generator.BestOfYearBlockGenerator;
import ru.yandex.chemodan.app.lentaloader.cool.generator.DefaultBlockGenerator;
import ru.yandex.chemodan.app.lentaloader.cool.generator.ThematicBlocksGenerator;
import ru.yandex.chemodan.app.lentaloader.cool.generator.ThemeDefinition;
import ru.yandex.chemodan.app.lentaloader.cool.generator.ThemeDefinitionRegistry;
import ru.yandex.chemodan.app.lentaloader.cool.model.CoolLentaBlock;
import ru.yandex.chemodan.app.lentaloader.cool.model.MinimalCoolLentaBlock;
import ru.yandex.chemodan.app.lentaloader.cool.utils.BlockTitlesGenerator;
import ru.yandex.chemodan.app.lentaloader.cool.utils.GeoNamesSource;
import ru.yandex.chemodan.app.lentaloader.cool.utils.TankerTextGenerator;
import ru.yandex.chemodan.app.lentaloader.cool.utils.TextProcessor;
import ru.yandex.chemodan.app.lentaloader.lenta.FindOrCreateResult;
import ru.yandex.chemodan.app.lentaloader.lenta.LentaBlockRecord;
import ru.yandex.chemodan.app.lentaloader.lenta.LentaManager;
import ru.yandex.chemodan.app.lentaloader.lenta.LentaNotificationManager;
import ru.yandex.chemodan.app.lentaloader.lenta.LentaRecordType;
import ru.yandex.chemodan.app.lentaloader.lenta.update.LentaBlockCreateData;
import ru.yandex.chemodan.app.lentaloader.log.ActionInfo;
import ru.yandex.chemodan.app.lentaloader.reminder.sendpush.CoolLentaBlockSendPushManager;
import ru.yandex.chemodan.app.lentaloader.reminder.sendpush.GncClient;
import ru.yandex.chemodan.app.lentaloader.reminder.titles.CoolLentaBlockTitlesManager;
import ru.yandex.chemodan.app.lentaloader.reminder.titles.DefaultCoolLentaBlockTitlesGenerator;
import ru.yandex.chemodan.app.lentaloader.reminder.titles.NYearsAgoCoolLentaBlockTitlesGenerator;
import ru.yandex.chemodan.app.lentaloader.reminder.titles.SpecialDateTitleConfiguration;
import ru.yandex.chemodan.app.lentaloader.reminder.titles.SpecialDateTitleConfigurationRegistry;
import ru.yandex.chemodan.app.lentaloader.reminder.titles.ThematicCoolLentaBlockTitlesGenerator;
import ru.yandex.chemodan.app.lentaloader.test.TestUtils;
import ru.yandex.chemodan.app.uaas.experiments.ExperimentsManager;
import ru.yandex.chemodan.mpfs.UserBlockedException;
import ru.yandex.chemodan.util.blackbox.UserTimezoneHelper;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.inside.passport.blackbox2.Blackbox2;
import ru.yandex.misc.bender.config.BenderConfiguration;
import ru.yandex.misc.db.embedded.OS;
import ru.yandex.misc.db.embedded.sandbox.SandBoxTestResourceResolver;
import ru.yandex.misc.db.embedded.sandbox.SandboxResourcesRule;
import ru.yandex.misc.io.ClassPathResourceInputStreamSource;
import ru.yandex.misc.time.TimeUtils;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;
import static org.mockito.ArgumentMatchers.anyBoolean;
import static org.mockito.ArgumentMatchers.anyString;
import static org.mockito.ArgumentMatchers.argThat;
import static org.mockito.Mockito.any;
import static org.mockito.Mockito.doNothing;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;

public class CoolLentaReminderTest {

    private static final String MINIMAL_BLOCK_RESOURCE_ID = "1080714525";

    private static final String EXTENDED_BLOCK_RESOURCE_ID = "1080716202";

    private static final long DATE_WITHOUT_N_YEARS_BLOCKS = 1_565_082_044_000L; //6 August 2019, 9:00:44 GMT

    private static final long DATE_WITH_N_YEARS_BLOCKS = 1_562_630_400_000L; //9 July 2019, 0:00:00 GMT

    private static final String N_YEARS_BLOCK_ID = "default_one_day_2018_07_09";

    private static final String NON_ONE_DAY_BLOCK_ID = "default_year_2019";

    private static final String NON_N_YEARS_ONE_DAY_BLOCK_ID = "default_one_day_2019_06_10";

    private static final String THEMATIC_BLOCK_ID = "thematic_year_-2208997817000_nature";

    private static ListF<MinimalCoolLentaBlock> minimalBlocks = Cf.arrayList();
    private static ListF<CoolLentaBlock> extendedBlocks = Cf.arrayList();

    @ClassRule
    public static final SandboxResourcesRule resourcesRule =
            SandboxResourcesRule
                    .create(Cf.list(new SandBoxTestResourceResolver(Cf.map(
                            OS.ALL, MINIMAL_BLOCK_RESOURCE_ID
                    )), new SandBoxTestResourceResolver(Cf.map(
                            OS.ALL, EXTENDED_BLOCK_RESOURCE_ID
                    ))))
                    .withEnabledCache(true)
                    .withFilenames(Cf.list("mb.json", "eb.json"))
                    .withMaxRetryCount(10)
                    .build();

    private final LentaManager lentaManager = mock(LentaManager.class);
    private final LentaNotificationManager lentaNotificationManager = mock(LentaNotificationManager.class);
    private final CoolLentaManager coolLentaManager = mock(CoolLentaManager.class);
    private final Cvi2tProcessor cvi2tProcessor = mock(Cvi2tProcessor.class);
    private final ExperimentsManager experimentsManager = mock(ExperimentsManager.class);
    private final ThemeDefinitionRegistry themeDefinitionRegistry = mock(ThemeDefinitionRegistry.class);
    private final ExperimentMembersByOverrideManager experimentMembersByOverrideManager =
            mock(ExperimentMembersByOverrideManager.class);
    private final SpecialDateTitleConfigurationRegistry specialDateTitleConfigurationRegistry =
            mock(SpecialDateTitleConfigurationRegistry.class);
    private CoolLentaBlockTitlesManager coolLentaBlockTitlesManager;
    private UserTimezoneHelper userTimezoneHelper = mock(UserTimezoneHelper.class);
    private CoolLentaConfigurationManager coolLentaConfigurationManager = new CoolLentaConfigurationManager();
    private DataApiManager dataApiManager = mock(DataApiManager.class);
    private MordaPushManager mordaPushManager = mock(MordaPushManager.class);
    private Blackbox2 blackbox2 = mock(Blackbox2.class);
    private GncClient lentaGncClient = mock(GncClient.class);

    private final CoolLentaReminder.BlockGenerationConfig blockGenerationConfig =
            new CoolLentaReminder.BlockGenerationConfig(Cf.list("ios", "android"), true, true, 100500, 100500, 100500, 100500,
                    Option.empty(), Option.empty(), true, false);
    private final PassportUid uid = PassportUid.cons(50273844);

    @BeforeClass
    public static void init() {
        BenderConfiguration configuration = TestUtils.createBenderConfiguration();
        minimalBlocks.addAll(TestUtils.parseTestData(resourcesRule, MinimalCoolLentaBlock.class, configuration, "mb.json"));
        extendedBlocks.addAll(TestUtils.parseTestData(resourcesRule, CoolLentaBlock.class, configuration, "eb.json"));
    }

    @Before
    public void initMocks() {
        when(coolLentaManager.getAllBlocks(uid)).thenReturn(minimalBlocks);
        when(coolLentaManager.getExtendedBlock(any(PassportUid.class),
                argThat(new MinimalCoolLentaBlockMatcher(N_YEARS_BLOCK_ID)), any()))
                .thenReturn(extendedBlocks.first());
        when(coolLentaManager.getExtendedBlock(any(PassportUid.class),
                argThat(new MinimalCoolLentaBlockMatcher(NON_ONE_DAY_BLOCK_ID)), any()))
                .thenReturn(extendedBlocks.get(1));
        when(coolLentaManager.getExtendedBlock(any(PassportUid.class),
                argThat(new MinimalCoolLentaBlockMatcher(NON_N_YEARS_ONE_DAY_BLOCK_ID)), any()))
                .thenReturn(extendedBlocks.get(2));
        when(coolLentaManager.getExtendedBlock(any(PassportUid.class),
                argThat(new MinimalCoolLentaBlockMatcher(THEMATIC_BLOCK_ID)), any()))
                .thenReturn(extendedBlocks.get(3));
        doNothing().when(lentaNotificationManager)
                .scheduleReminderBlockNotification(any(DataApiUserId.class), anyString(), anyBoolean());
        when(lentaManager
                .findOrCreateBlock(any(DataApiUserId.class), any(LentaBlockCreateData.class), any(ActionInfo.class)))
                .thenReturn(new FindOrCreateResult(
                        new LentaBlockRecord("block-1", 0, LentaRecordType.PHOTO_SELECTION_BLOCK, "",
                                Option.empty(), Instant.now(), Option.empty(), Cf.map()),
                        Option.empty(),
                        Option.empty(),
                        FindOrCreateResult.Status.FOUND
                        ));
        when(experimentsManager.getFlags(uid.toUidOrZero().getUid()))
                .thenReturn(Cf.list());
        ThemeDefinition nature = TestUtils.createNatureThemeDefinition();
        when(themeDefinitionRegistry.getO("nature")).thenReturn(Option.of(nature));
        when(themeDefinitionRegistry.getAll()).thenReturn(Cf.list(nature));
        BlockTitlesGenerator blockTitlesGenerator = new BlockTitlesGenerator(GeoNamesSource.EMPTY);
        TankerTextGenerator tankerTextGenerator = new TankerTextGenerator(
                new ClassPathResourceInputStreamSource(TankerTextGenerator.class, "yadisk_backend.lenta.json"),
                new TextProcessor(GeoNamesSource.EMPTY));
        when(specialDateTitleConfigurationRegistry.getSpecialDateConfigurationForDateO(any(SpecialDateTitleConfiguration.DayWithMonth.class)))
                .thenReturn(Option.empty());
        coolLentaBlockTitlesManager = new CoolLentaBlockTitlesManager(Cf.list(
                new DefaultCoolLentaBlockTitlesGenerator(specialDateTitleConfigurationRegistry, tankerTextGenerator, blockTitlesGenerator, experimentsManager),
                new NYearsAgoCoolLentaBlockTitlesGenerator(specialDateTitleConfigurationRegistry, tankerTextGenerator, blockTitlesGenerator, experimentsManager),
                new ThematicCoolLentaBlockTitlesGenerator(specialDateTitleConfigurationRegistry, tankerTextGenerator, blockTitlesGenerator,
                        experimentsManager, themeDefinitionRegistry)));
        when(userTimezoneHelper.getUserTimezone(uid)).thenReturn(TimeUtils.EUROPE_MOSCOW_TIME_ZONE);
    }

    @Test
    public void testDefaultConfiguration() {
        when(experimentsManager.getFlags(uid.toUidOrZero().getUid()))
                .thenReturn(Cf.list());
        CoolLentaReminder reminder = createReminder(40, 20, 40);
        DateTime now = DateTime.now();
        CoolLentaReminder.SaveBlockContext saveBlockContext = reminder.selectBlockToSave(uid, now, blockGenerationConfig)
                .getOrThrow("Empty block");
        assertTrue(Cf.set(N_YEARS_BLOCK_ID, NON_N_YEARS_ONE_DAY_BLOCK_ID, NON_ONE_DAY_BLOCK_ID, THEMATIC_BLOCK_ID)
                .containsTs(saveBlockContext.getBlock().getId()));
        assertTrue(CoolLentaReminder.BlockGenerationType.N_YEARS == saveBlockContext.getType() ||
                CoolLentaReminder.BlockGenerationType.USUAL == saveBlockContext.getType() ||
                CoolLentaReminder.BlockGenerationType.THEMATIC == saveBlockContext.getType());
        assertTrue(reminder.generateBlock(uid, now, blockGenerationConfig).isPresent());
    }

    @Test
    public void testEnabledNYearsFeatureForNonNYearsDate() {
        CoolLentaReminder reminder = createReminder(40, 20, 0);
        DateTime dateTime = new DateTime(DATE_WITHOUT_N_YEARS_BLOCKS);
        CoolLentaReminder.SaveBlockContext saveBlockContext = reminder.selectBlockToSave(uid, dateTime, blockGenerationConfig)
                .getOrThrow("Empty block");
        assertTrue(Cf.set(N_YEARS_BLOCK_ID, NON_N_YEARS_ONE_DAY_BLOCK_ID, NON_ONE_DAY_BLOCK_ID)
                .containsTs(saveBlockContext.getBlock().getId()));
        assertEquals(CoolLentaReminder.BlockGenerationType.USUAL, saveBlockContext.getType());
        assertTrue(reminder.generateBlock(uid, dateTime, blockGenerationConfig).isPresent());
    }

    @Test
    public void testEnabledNYearsFeatureForNYearsDate() {
        CoolLentaReminder reminder = createReminder(40, 20, 0);
        DateTime dateTime = new DateTime(DATE_WITH_N_YEARS_BLOCKS);
        CoolLentaReminder.SaveBlockContext saveBlockContext = reminder.selectBlockToSave(uid, dateTime, blockGenerationConfig)
                .getOrThrow("Empty block");
        assertTrue(Cf.set(N_YEARS_BLOCK_ID, NON_N_YEARS_ONE_DAY_BLOCK_ID, NON_ONE_DAY_BLOCK_ID)
                .containsTs(saveBlockContext.getBlock().getId()));
        assertTrue(CoolLentaReminder.BlockGenerationType.N_YEARS == saveBlockContext.getType() ||
                CoolLentaReminder.BlockGenerationType.USUAL == saveBlockContext.getType());
        System.out.println(saveBlockContext.getType());
        assertTrue(reminder.generateBlock(uid, dateTime, blockGenerationConfig).isPresent());
    }

    @Test
    public void testEnabledNYearsFeatureForNYearsDateWithDisabledUsualBlocks() {
        CoolLentaReminder reminder = createReminder(0, 20, 0);
        DateTime dateTime = new DateTime(DATE_WITH_N_YEARS_BLOCKS);
        CoolLentaReminder.SaveBlockContext saveBlockContext = reminder.selectBlockToSave(uid, dateTime, blockGenerationConfig)
                .getOrThrow("Empty block");
        assertEquals(N_YEARS_BLOCK_ID, saveBlockContext.getBlock().getId());
        assertEquals(CoolLentaReminder.BlockGenerationType.N_YEARS, saveBlockContext.getType());
        assertTrue(reminder.generateBlock(uid, dateTime, blockGenerationConfig).isPresent());
    }

    @Test
    public void testEnabledNYearsFeatureForNYearsDateWithDisabledNYearsBlocks() {
        CoolLentaReminder reminder = createReminder(40, 0, 0);
        DateTime dateTime = new DateTime(DATE_WITH_N_YEARS_BLOCKS);
        CoolLentaReminder.SaveBlockContext saveBlockContext = reminder.selectBlockToSave(uid, dateTime, blockGenerationConfig)
                .getOrThrow("Empty block");
        assertTrue(Cf.set(N_YEARS_BLOCK_ID, NON_N_YEARS_ONE_DAY_BLOCK_ID, NON_ONE_DAY_BLOCK_ID)
                .containsTs(saveBlockContext.getBlock().getId()));
        assertEquals(CoolLentaReminder.BlockGenerationType.USUAL, saveBlockContext.getType());
        assertTrue(reminder.generateBlock(uid, dateTime, blockGenerationConfig).isPresent());
    }

    @Test
    public void testEnabledAllFeatures() {
        DateTime dateTime = new DateTime(DATE_WITH_N_YEARS_BLOCKS);
        CoolLentaReminder reminder = createReminder(40, 20, 40);
        CoolLentaReminder.SaveBlockContext saveBlockContext = reminder.selectBlockToSave(uid, dateTime, blockGenerationConfig)
                .getOrThrow("Empty block");
        assertTrue(Cf.set(N_YEARS_BLOCK_ID, NON_N_YEARS_ONE_DAY_BLOCK_ID, NON_ONE_DAY_BLOCK_ID, THEMATIC_BLOCK_ID)
                .containsTs(saveBlockContext.getBlock().getId()));
        CoolLentaReminder.BlockGenerationType type = saveBlockContext.getType();
        assertTrue(CoolLentaReminder.BlockGenerationType.USUAL == type ||
                CoolLentaReminder.BlockGenerationType.THEMATIC == type ||
                CoolLentaReminder.BlockGenerationType.N_YEARS == type);
        System.out.println(type);
        assertTrue(reminder.generateBlock(uid, dateTime, blockGenerationConfig).isPresent());
    }

    @Test
    public void testEnabledThematicOnly() {
        DateTime dateTime = new DateTime(DATE_WITHOUT_N_YEARS_BLOCKS);
        CoolLentaReminder reminder = createReminder(0, 0, 40);
        CoolLentaReminder.SaveBlockContext saveBlockContext = reminder.selectBlockToSave(uid, dateTime, blockGenerationConfig)
                .getOrThrow("Empty block");
        assertEquals(THEMATIC_BLOCK_ID, saveBlockContext.getBlock().getId());
        assertEquals(CoolLentaReminder.BlockGenerationType.THEMATIC, saveBlockContext.getType());
        assertTrue(reminder.generateBlock(uid, dateTime, blockGenerationConfig).isPresent());
    }

    @Test
    public void testMpfsUserBlocked() {
        when(coolLentaManager.getAllBlocks(uid)).thenThrow(UserBlockedException.class);
        DateTime dateTime = new DateTime(DATE_WITH_N_YEARS_BLOCKS);
        CoolLentaReminder reminder = createReminder(0, 0, 40);
        assertTrue(reminder.generateBlock(uid, dateTime, blockGenerationConfig).isEmpty());
    }

    private CoolLentaReminder createReminder(int usualBlocksProbability,
            int nYearsBlocksProbability, int thematicBlocksProbability)
    {
        return new CoolLentaReminder(coolLentaManager, lentaManager, lentaNotificationManager,
                cvi2tProcessor, experimentsManager, themeDefinitionRegistry,
                Cf.list(new BestOfYearBlockGenerator(), new DefaultBlockGenerator(),
                        new ThematicBlocksGenerator(themeDefinitionRegistry, cvi2tProcessor, experimentMembersByOverrideManager,
                                3, 20, false)),
                Duration.ZERO, Duration.ZERO, usualBlocksProbability, nYearsBlocksProbability,
                thematicBlocksProbability, 5, coolLentaBlockTitlesManager, new CoolLentaBlockSendPushManager(90, 3), userTimezoneHelper,
                coolLentaConfigurationManager, dataApiManager, mordaPushManager, blackbox2, lentaGncClient);
    }

    private static class MinimalCoolLentaBlockMatcher implements ArgumentMatcher<MinimalCoolLentaBlock> {

        private final String blockId;

        public MinimalCoolLentaBlockMatcher(String blockId) {
            this.blockId = blockId;
        }

        @Override
        public boolean matches(MinimalCoolLentaBlock argument) {
            return blockId.equals(argument.getId());
        }
    }
}
