package ru.yandex.chemodan.app.lentaloader.reminder;

import java.io.IOException;
import java.net.URI;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.apache.commons.lang3.StringEscapeUtils;
import org.apache.http.HttpResponse;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.ByteArrayEntity;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.bender.Bender;
import ru.yandex.misc.bender.parse.BenderJsonParser;
import ru.yandex.misc.codec.Hex;
import ru.yandex.misc.io.http.UriBuilder;
import ru.yandex.misc.io.http.apache.v4.Abstract200ExtendedResponseHandler;
import ru.yandex.misc.io.http.apache.v4.ApacheHttpClientUtils;
import ru.yandex.misc.io.http.apache.v4.ReadStringResponseHandler;
import ru.yandex.misc.lang.Check;
import ru.yandex.misc.time.InstantInterval;

/**
 * @author tolmalev
 */
public class DiskSearchClient {
    private static final BenderJsonParser<DiskSearchResponse> responseParser =
            Bender.jsonParser(DiskSearchResponse.class);

    private static final ObjectMapper mapper = new ObjectMapper();

    private static final String GET_FIELDS =
            "id,name,latitude,longitude,etime,beautiful,cost_disk_aethetic_0,aux_folder,i2t_keyword,md5"
                    + ",width,height"
           + ",binary_porn,mobile_porn,gruesome";

    private final HttpClient httpClient;
    private final String searchHost;

    public DiskSearchClient(HttpClient httpClient, String searchHost) {
        this.httpClient = httpClient;
        this.searchHost = searchHost;
    }

    public DiskSearchResponse findPhotos(PassportUid uid, InstantInterval interval, int maxCount) {
        return execute(host -> buildPhotoSearchUrl(uid, interval, maxCount, host, Option.empty()));
    }

    public DiskSearchResponse findPhotosWithQualifier(PassportUid uid, InstantInterval interval, int maxCount,
            Option<String> qualifier)
    {
        return execute(host -> buildPhotoSearchUrl(uid, interval, maxCount, host, qualifier));
    }

    public DiskSearchResponse findPhotosWithBeauty(PassportUid uid, InstantInterval interval,
            Option<String> experimentName)
    {
        return findPhotosWithBeauty(uid, interval, experimentName, 0, 1000);
    }

    public DiskSearchResponse findPhotosWithBeauty(PassportUid uid, InstantInterval interval,
            Option<String> experimentName, int offset, int maxCount)
    {
        return execute(host -> buildBeautySearchUrl(uid, interval, experimentName, host, offset, maxCount));
    }

    public ListF<DiskSearchFileInfo> getFilesInfoBatch(PassportUid uid, ListF<String> fileIds, Option<String> experimentName) {
        return execute(host -> buildBatchGetUrl(uid, fileIds, experimentName)).hitsArray;
    }

    public long findPhotosliceDocumentsCount(PassportUid uid) {
        return execute(hots -> buildPhotosliceDocumentsCountUrl(uid, hots)).hitsCount;
    }

    public MapF<Integer, Integer> getCountPhotosByDay(PassportUid uid) {
        URI url = UriBuilder.cons(searchHost)
                .appendPath("sequential/search")
                .addParam("service", "disk_queue")
                .addParam("text", "mediatype:9 AND (ext:jpg OR ext:heic) AND (aux_folder:disk OR aux_folder:photounlim)")
                .addParam("dp", "const(1 bad_etime)")
                .addParam("dp", "fallback(etime,bad_etime etime)")
                .addParam("dp", "increment(etime,10800 local_etime)")
                .addParam("dp", "cdiv(local_etime,86400 day)")
                .addParam("group", "day")
                .addParam("merge_func", "count")
                .addParam("get", "day")
                .addParam("collector", "sorted")
                .addParam("postfilter", "etime >= 2")
                .addParam("sort", "day")
                .addParam("IO_PRIO", "10000")
                .addParam("prefix", uid)
                .build();

        return ApacheHttpClientUtils.execute(new HttpGet(url), httpClient, new Abstract200ExtendedResponseHandler<MapF<Integer, Integer>>() {
            @Override
            protected MapF<Integer, Integer> handle200Response(HttpResponse response) throws IOException {
                JsonNode node = mapper.readTree(response.getEntity().getContent());
                node = node.get("hitsArray");
                Check.notNull(node, "bad response from search: no 'hitsArray'");
                Check.isTrue(node.isArray(), "hitsArray field must be array");

                MapF<Integer, Integer> result = Cf.hashMap();

                for (int i = 0; i < node.size(); i++) {
                    JsonNode partNode = node.get(i);

                    Check.notNull(partNode.get("day"), "no field 'word' in response");
                    int mergedDocsCount = 1;
                    if (partNode.get("merged_docs_count") != null) {
                        mergedDocsCount += partNode.get("merged_docs_count").asInt();
                    }

                    result.put(
                            Integer.parseInt(partNode.get("day").asText()),
                            mergedDocsCount
                    );
                }

                return result;
            }
        });
    }

    public MapF<String, byte[]> getTextCvVectors(ListF<String> words) {
        HttpPost request = new HttpPost(UriBuilder.cons(searchHost).appendPath("i2t").build());
        StringBuilder sb = new StringBuilder();

        sb.append("{\"words\": [");
        sb.append(words.map(StringEscapeUtils::escapeJson).mkString("\"", "\", \"", "\""));
        sb.append("]}");
        request.setEntity(new ByteArrayEntity(sb.toString().getBytes()));

        return ApacheHttpClientUtils.execute(request, httpClient, new Abstract200ExtendedResponseHandler<MapF<String, byte[]>>() {
            @Override
            protected MapF<String, byte[]> handle200Response(HttpResponse response) throws IOException {
                JsonNode node = mapper.readTree(response.getEntity().getContent());
                node = node.get("results");
                Check.notNull(node, "bad response from search: no 'results'");
                Check.isTrue(node.isArray(), "results field must be array");

                MapF<String, byte[]> result = Cf.hashMap();
                for (int i = 0; i < node.size(); i++) {
                    JsonNode partNode = node.get(i);

                    Check.notNull(partNode.get("word"), "no field 'word' in response");
                    Check.notNull(partNode.get("i2t"), "no field 'i2t' in response");

                    result.put(
                            partNode.get("word").asText(),
                            Hex.decode(partNode.get("i2t").asText())
                    );
                }

                return result;
            }
        });
    }

    private DiskSearchResponse execute(Function<String, String> consUriForHost) {
        return responseParser.parseJson(
                ApacheHttpClientUtils.execute(new HttpGet(consUriForHost.apply(searchHost)), httpClient, new ReadStringResponseHandler())
        );
    }

    public static DiskSearchResponse parseResponseJson(String json) {
        return responseParser.parseJson(json);
    }

    String buildPhotoSearchUrl(PassportUid uid, InstantInterval interval, int maxCount) {
        return buildPhotoSearchUrl(uid, interval, maxCount, searchHost, Option.empty());
    }

    String buildPhotoSearchUrl(PassportUid uid, InstantInterval interval, int maxCount, String searchHost,
            Option<String> qualifier)
    {
        UriBuilder builder = UriBuilder.cons(searchHost)
                .appendPath("listing")

                .addParam("uid", uid)
                .addParam("from-etime", interval.getStartMillis() / 1000)
                .addParam("to-etime", interval.getEndMillis() / 1000)
                .addParam("mimetype", "image/jpeg")
                .addParam("mimetype", "image/heif")
                .addParam("mediatype", 9)
                .addParam("length", maxCount)
                .addParam("get", GET_FIELDS)
                .addParam("skip-nulls", "true")
                .addParam("sort", "etime");

        qualifier.ifPresent((value) -> builder.addParam("search-qualifier", value));

        return builder.toUrl();
    }

    String buildPhotosliceDocumentsCountUrl(PassportUid uid, String searchHost) {
        UriBuilder builder = UriBuilder.cons(searchHost)
                .appendPath("listing")
                .addParam("uid", uid)
                .addParam("skip-nulls", "true")
                .addParam("apply-photoslice-filter", "1")
                .addParam("length", 0);

        return builder.toUrl();
    }

    String buildBeautySearchUrl(PassportUid uid, InstantInterval interval, Option<String> experimentName,
            String searchHost, int offset, int maxCount)
    {
        UriBuilder builder = UriBuilder.cons(searchHost)
                .appendPath("listing")

                .addParam("uid", uid)
                .addParam("from-etime", interval.getStartMillis() / 1000)
                .addParam("to-etime", interval.getEndMillis() / 1000)
                .addParam("mimetype", "image/jpeg")
                .addParam("mimetype", "image/heif")
                .addParam("mediatype", 9)
                .addParam("offset", offset)
                .addParam("length", maxCount)
                .addParam("skip-nulls", "true")
                .addParam("get", GET_FIELDS);

        experimentName.ifPresent((value) -> builder.addParam("search-qualifier", value));

        return builder.toUrl();
    }

    String buildBatchGetUrl(PassportUid uid, ListF<String> fileIds, Option<String> experimentName) {
        UriBuilder builder = UriBuilder.cons(searchHost)
                .appendPath("get")
                .addParam("uid", uid)
                .addParam("get", GET_FIELDS)
                .addParam("skip-nulls", "true");

        for (String fileId : fileIds) {
            builder.addParam("id", fileId);
        }

        experimentName.ifPresent((value) -> builder.addParam("search-qualifier", value));

        return builder.toUrl();
    }
}
