package ru.yandex.chemodan.app.lentaloader.reminder;

import lombok.Data;
import org.joda.time.DateTime;
import org.joda.time.LocalDate;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.lentaloader.cool.CoolLentaConfigurationManager;
import ru.yandex.chemodan.app.uaas.experiments.ExperimentsManager;
import ru.yandex.chemodan.util.blackbox.UserTimezoneHelper;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author tolmalev
 * @author dk666
 */
public class PhotoReminderManager {

    private static final Logger logger = LoggerFactory.getLogger(PhotoReminderManager.class);
    private static final String SEND_PUSHES_EXPERIMENT_FLAG_NAME_PREFIX = "disk_lenta_blocks_sends_pushes_";

    private final DynamicProperty<Integer> coolLentaBlockMaxDayWithSendsPerWeek =
            new DynamicProperty<>("lenta-cool-lenta-block-max-days-with-sends-per-week", 3);
    private final DynamicProperty<Integer> coolLentaBlockMaxPushesPerWeek =
            new DynamicProperty<>("lenta-cool-lenta-block-max-pushes-per-week", 3);

    private final ExperimentsManager experimentsManager;
    private final UserTimezoneHelper userTimezoneHelper;

    private final CoolLentaReminder coolLentaReminder;
    private final CoolLentaConfigurationManager coolLentaConfigurationManager;

    public PhotoReminderManager(
            UserTimezoneHelper userTimezoneHelper,
            ExperimentsManager experimentsManager,
            CoolLentaReminder coolLentaReminder, CoolLentaConfigurationManager coolLentaConfigurationManager)
    {
        this.experimentsManager = experimentsManager;
        this.userTimezoneHelper = userTimezoneHelper;
        this.coolLentaReminder = coolLentaReminder;
        this.coolLentaConfigurationManager = coolLentaConfigurationManager;
    }

    public void makeRemindBlock(PassportUid uid, Option<LocalDate> dayFromTask, int validTo, int maxSendsWithPushesPerDay) {
        DateTime userTime = DateTime.now(userTimezoneHelper.getUserTimezone(uid));
        if (dayFromTask.isPresent()) {
            LocalDate userDay =
                    userTime.toLocalDate();
            if (!userDay.isEqual(dayFromTask.get())) {
                logger.warn("Day from the task {} is not the same as day at user's timezone {}", dayFromTask.get(),
                        userDay);
                return;
            }
        }
        // Allow some tolerance
        if (userTime.minusMinutes(5).getHourOfDay() >= validTo) {
            logger.warn("Don't send notification because it's too late; user's time {}, validTo {}",
                    userTime, validTo);
            return;
        }
        makeRemindBlockForDateTime(uid, Option.empty(), maxSendsWithPushesPerDay);
    }

    private DateTime getUserTime(Option<DateTime> overrideValue, PassportUid uid) {
        return overrideValue.orElseGet(() -> DateTime.now(userTimezoneHelper.getUserTimezone(uid)));
    }

    private void makeRemindBlockForDateTime(PassportUid uid, Option<DateTime> forDateTime, int maxSendsWithPushesPerDay) {
        DateTime userTimeNow = getUserTime(forDateTime, uid);

        SendPushesConfiguration sendPushesConfiguration = coolLentaBlockConfig(uid, maxSendsWithPushesPerDay, maxSendsWithPushesPerDay);

        coolLentaReminder.generateBlock(uid, userTimeNow,
                new CoolLentaReminder.BlockGenerationConfig(coolLentaConfigurationManager.platformsToPush.get(), true, false,
                        sendPushesConfiguration.getMaxPushesPerWeek(), sendPushesConfiguration.getMaxDaysWithSendsPerWeek(),
                        sendPushesConfiguration.getMaxSendsPerDay(), sendPushesConfiguration.getMaxPushesPerDay(),
                        Option.of(coolLentaReminder.defaultGeoConfig()), Option.empty(), false, false));
    }

    private SendPushesConfiguration coolLentaBlockConfig(PassportUid uid, int maxSendsPerDay, int maxPushesPerDay) {
        SendPushesConfiguration defaultConfiguration = new SendPushesConfiguration(coolLentaBlockMaxDayWithSendsPerWeek.get(),
                coolLentaBlockMaxPushesPerWeek.get(), maxSendsPerDay, maxPushesPerDay);

        Option<String> flagO = experimentsManager.getFlags(uid.toUidOrZero().getUid())
                .find(flag -> flag.startsWith(SEND_PUSHES_EXPERIMENT_FLAG_NAME_PREFIX));
        if (!flagO.isPresent()) {
            return defaultConfiguration;
        }
        String[] values = flagO.get().substring(SEND_PUSHES_EXPERIMENT_FLAG_NAME_PREFIX.length()).split("_");
        if (values.length != 2) {
            return defaultConfiguration;
        }
        Option<Integer> maxSendsO = Cf.Integer.parseSafe(values[0]);
        if (!maxSendsO.isPresent()) {
            return defaultConfiguration;
        }
        Option<Integer> maxPushesO = Cf.Integer.parseSafe(values[1]);
        if (!maxPushesO.isPresent()) {
            return defaultConfiguration;
        }
        return new SendPushesConfiguration(maxSendsO.get(), maxPushesO.get(), maxSendsPerDay, maxPushesPerDay);
    }

    @Data
    private static class SendPushesConfiguration {
        private final int maxDaysWithSendsPerWeek;
        private final int maxPushesPerWeek;
        private final int maxSendsPerDay;
        private final int maxPushesPerDay;
    }
}
