package ru.yandex.chemodan.app.lentaloader.reminder;

import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import org.joda.time.Instant;
import org.joda.time.LocalDate;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiPassportUserId;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.dataapi.support.RecordField;
import ru.yandex.chemodan.app.lentaloader.blocks.ContentBlockAction;
import ru.yandex.chemodan.app.lentaloader.lenta.FieldPredicate;
import ru.yandex.chemodan.app.lentaloader.lenta.LentaManager;
import ru.yandex.chemodan.app.lentaloader.lenta.LentaRecordType;
import ru.yandex.chemodan.app.lentaloader.log.ActionInfo;
import ru.yandex.chemodan.app.lentaloader.log.ActionSource;
import ru.yandex.chemodan.app.uaas.experiments.ExperimentsManager;
import ru.yandex.chemodan.notifier.AddLentaNotificationBlockTask;
import ru.yandex.chemodan.util.blackbox.UserTimezoneHelper;
import ru.yandex.commune.bazinga.BazingaBender;
import ru.yandex.commune.bazinga.impl.JobStatus;
import ru.yandex.commune.bazinga.impl.OnetimeJob;
import ru.yandex.commune.bazinga.impl.TaskId;
import ru.yandex.commune.bazinga.impl.storage.BazingaStorage;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

public class PhotoReminderTaskManager {

    public static final String COOL_LENTA_REMOVE_PUSH_FOR_AUTOUPLOAD_EXPERIMENT = "disk_cool_lenta_remove_push_for_upload";
    private static final Logger logger = LoggerFactory.getLogger(PhotoReminderTaskManager.class);

    private final BazingaStorage bazingaStorage;

    private final ExperimentsManager experimentsManager;

    private final LentaManager lentaManager;

    private final UserTimezoneHelper userTimezoneHelper;

    public PhotoReminderTaskManager(BazingaStorage bazingaStorage, ExperimentsManager experimentsManager,
            LentaManager lentaManager, UserTimezoneHelper userTimezoneHelper)
    {
        this.bazingaStorage = bazingaStorage;
        this.experimentsManager = experimentsManager;
        this.lentaManager = lentaManager;
        this.userTimezoneHelper = userTimezoneHelper;
    }

    public boolean deleteAddNotificationForAutouploadTaskForBlock(PassportUid uid, Instant userTimeNow) {
        if (!experimentsManager.getFlags(uid.toUidOrZero().getUid()).containsTs(COOL_LENTA_REMOVE_PUSH_FOR_AUTOUPLOAD_EXPERIMENT)) {
            logger.debug("User is not connected to remove autoupload notification experiment uid={}", uid);
            return false;
        }
        DataApiUserId userId = new DataApiPassportUserId(uid);
        Option<DateTimeZone> userTimeZoneO = Option.of(userTimezoneHelper.getUserTimezone(uid));
        DateTime userTime = getDateTimeInUserTimezone(uid, userTimeNow, userTimeZoneO);
        DateTime now = getNowInUserTimezone(uid, userTimeZoneO);
        LocalDate today = userTime.toLocalDate();
        if (!now.toLocalDate().isEqual(today)) {
            logger.debug("Too late for deleting autoupload notification uid={}, now={}, userTime={}", uid, now, userTime);
            return false;
        }
        ListF<String> blockRecordIds = lentaManager.findBlocks(userId, LentaRecordType.CONTENT_BLOCK,
                FieldPredicate.field(RecordField.string("action")).in(Cf.set(ContentBlockAction.AUTOSAVE.value(),
                        ContentBlockAction.PHOTO_UNLIM.value())),
                ActionInfo.internal(ActionSource.photoReminders()))
                .filter(lentaBlockRecord -> lentaBlockRecord.mTill.map(
                        mTill -> isInstantInDay(mTill, today, uid) || isInstantInDay(mTill, today.minusDays(1), uid))
                        .getOrElse(Boolean.FALSE)
                ).map(lentaBlockRecord -> lentaBlockRecord.id);
        if (blockRecordIds.isEmpty()) {
            logger.debug("There is no autoupload blocks for user. uid={}", uid);
            return false;
        }
        boolean tasksHasBeenDeleted = false;
        AddLentaNotificationBlockTask taskPrototype = new AddLentaNotificationBlockTask();
        for (String blockRecordId : blockRecordIds)  {
            AddLentaNotificationBlockTask.ActiveUidParams activeUidParams = new AddLentaNotificationBlockTask
                    .ActiveUidParams(userId, blockRecordId);
            tasksHasBeenDeleted =
                    deleteTaskByActiveUid(activeUidParams, AddLentaNotificationBlockTask.idF().apply(taskPrototype)) ||
                            tasksHasBeenDeleted;
        }
        if (tasksHasBeenDeleted) {
            logger.debug("The push autoupload notification task has been removed for uid={}", uid);
        } else {
            logger.debug("The push autoupload notification task has not been removed for uid={}", uid);
        }
        return tasksHasBeenDeleted;
    }

    private DateTime getNowInUserTimezone(PassportUid uid, Option<DateTimeZone> userTimeZone) {
        DateTimeZone timeZone = userTimeZone.orElseGet(() -> userTimezoneHelper.getUserTimezone(uid));

        return DateTime.now(timeZone);
    }

    private DateTime getDateTimeInUserTimezone(PassportUid uid, Instant instant, Option<DateTimeZone> userTimeZone) {
        DateTimeZone timeZone = userTimeZone.orElseGet(() -> userTimezoneHelper.getUserTimezone(uid));

        return new DateTime(instant, timeZone);
    }

    private boolean isInstantInDay(Instant instant, LocalDate day, PassportUid uid) {
        return day.isEqual(getDateTimeInUserTimezone(uid, instant, Option.empty()).toLocalDate());
    }

    private boolean deleteTaskByActiveUid(Object activeUid, TaskId taskId) {
        Option<OnetimeJob> sendPushJobO = bazingaStorage
                .findOnetimeJobByActiveUid(OnetimeJob.generateActiveUniqueIdentifier(taskId,
                        new String(BazingaBender.mapper.serializeJson(activeUid))))
                .filter(job -> job.getValue().getStatus() == JobStatus.READY);
        if (!sendPushJobO.isPresent()) {
            logger.debug("{} has not been found for parameters and status READY activeUid={}", taskId, activeUid);
            return false;
        }
        logger.debug("{} has been found for parameters activeUid={}. Trying to delete", taskId, activeUid);
        bazingaStorage.deleteOnetimeJob(sendPushJobO.map(OnetimeJob::getId).get());
        return true;
    }
}
