package ru.yandex.chemodan.app.lentaloader.reminder;

import javax.annotation.Nullable;

import lombok.Data;
import org.joda.time.Duration;
import org.joda.time.LocalDate;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.lentaloader.worker.LentaTaskQueueName;
import ru.yandex.commune.bazinga.scheduler.ActiveUniqueIdentifierConverter;
import ru.yandex.commune.bazinga.scheduler.ExecutionContext;
import ru.yandex.commune.bazinga.scheduler.OnetimeTaskSupport;
import ru.yandex.commune.bazinga.scheduler.TaskQueueName;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.bender.annotation.BenderDefaultValue;

/**
 * @author tolmalev
 */
public class RemindPhotosTask extends OnetimeTaskSupport<RemindPhotosTask.RemindPhotosTaskParameters> {

    private PhotoReminderManager photoReminderManager;

    public RemindPhotosTask(PhotoReminderManager photoReminderManager) {
        super(RemindPhotosTaskParameters.class);
        this.photoReminderManager = photoReminderManager;
    }

    public RemindPhotosTask(long uid, LocalDate day, int validTo, int index, int countPerDay) {
        super(new RemindPhotosTaskParameters(new PassportUid(uid), day, validTo, index, countPerDay));
    }

    public RemindPhotosTask(PassportUid uid, LocalDate day, int validTo, int index, int countPerDay) {
        super(new RemindPhotosTaskParameters(uid, day, validTo, index, countPerDay));
    }

    public RemindPhotosTask(RemindPhotosTaskParameters parameters) {
        super(parameters);
    }

    @Override
    protected void execute(RemindPhotosTaskParameters parameters, ExecutionContext context) throws Exception {
        LocalDate day = parameters.day;
        // default value is January 1st 1970
        if (day.getYear() < 2000) {
            // old version of a task
            photoReminderManager.makeRemindBlock(parameters.uid, Option.empty(), parameters.validTo, parameters.countPerDay);
        } else {
            // new version of a task
            photoReminderManager.makeRemindBlock(parameters.uid, Option.of(day), parameters.validTo, parameters.countPerDay);
        }
    }

    @Override
    public TaskQueueName queueName() {
        return LentaTaskQueueName.LENTA_OFFLINE;
    }

    @Override
    public int priority() {
        return 0;
    }

    @Override
    public Duration timeout() {
        return Duration.standardMinutes(1);
    }

    @Nullable
    @Override
    public Class<? extends ActiveUniqueIdentifierConverter<?, ?>> getActiveUidConverter() {
        return RemindPhotosTaskParameters.Converter.class;
    }

    @BenderBindAllFields
    public static class RemindPhotosTaskParameters {
        public final PassportUid uid;

        @BenderDefaultValue("1970-01-01")
        public final LocalDate day;

        @BenderDefaultValue("20")
        public final int validTo;

        @BenderDefaultValue("0")
        public final int index;

        @BenderDefaultValue("1")
        public final int countPerDay;

        RemindPhotosTaskParameters(PassportUid uid, LocalDate day, int validTo, int index, int countPerDay) {
            this.uid = uid;
            this.day = day;
            this.validTo = validTo;
            this.index = index;
            this.countPerDay = countPerDay;
        }

        public static class Converter implements ActiveUniqueIdentifierConverter<RemindPhotosTaskParameters, ActiveUid> {

            @Override
            public Class<ActiveUid> getActiveUniqueIdentifierClass() {
                return ActiveUid.class;
            }

            @Override
            public ActiveUid convert(RemindPhotosTaskParameters parameters) {
                return new ActiveUid(parameters.uid, parameters.day, parameters.index);
            }
        }
    }

    @Data
    @BenderBindAllFields
    public static class ActiveUid {
        private final PassportUid uid;
        private final LocalDate day;
        private final int index;
    }
}
