package ru.yandex.chemodan.app.lentaloader.reminder.sendpush;

import lombok.Data;
import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import org.joda.time.Days;
import org.joda.time.Instant;
import org.joda.time.LocalDate;
import org.joda.time.ReadableInstant;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.chemodan.app.lentaloader.cool.model.MinimalCoolLentaBlock;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.random.Random2;

/**
 * @author messiahlap
 */
public class CoolLentaBlockSendPushManager {

    private static final Logger logger = LoggerFactory.getLogger(CoolLentaBlockSendPushManager.class);

    private final int minRepeatDays;

    private final int burstBlocksCount;

    private final DynamicProperty<Boolean> pushCalculationProbabilityEnabled = new DynamicProperty<>("cool-lenta-photo-selection-push-probability-enabled", false);

    public CoolLentaBlockSendPushManager(int minRepeatDays, int burstBlocksCount) {
        this.minRepeatDays = minRepeatDays;
        this.burstBlocksCount = burstBlocksCount;
    }

    public boolean shouldDoSend(SendBlockConfiguration configuration) {
        ListF<MinimalCoolLentaBlock> allBlocks = configuration.getAllBlocks();
        if (configuration.getMaxSendsPerDay() == 0) {
            return false;
        }
        MapF<Integer, Integer> blocksSentForLastWeekCounts = getActionStatisticsForLastWeek(allBlocks,
                MinimalCoolLentaBlock::getLegacyLentaLastShowDate, Instant.now(), configuration.getUserTimezone());
        logger.debug("Sends week statistics {}", blocksSentForLastWeekCounts);
        int sentBlocksTodayCount = blocksSentForLastWeekCounts.getOrElse(0, 0);
        if (sentBlocksTodayCount >= configuration.getMaxSendsPerDay()) {
            logger.debug("Reached limit for sent blocks per day. sentBlocksTodayCount={}, maxPushesPerDay={}",
                    sentBlocksTodayCount, configuration.getMaxSendsPerDay());
            return false;
        }
        int daysWithSendsForLastWeek = blocksSentForLastWeekCounts.size();
        int maxDaysWithSendsPerWeek = configuration.getMaxDaysWithSendsPerWeek();
        if (daysWithSendsForLastWeek >= maxDaysWithSendsPerWeek) {
            logger.debug("Reached limit for days with sends per week. daysWithSendsForLastWeek={}, maxDaysWithSendsPerWeek={}",
                    daysWithSendsForLastWeek, maxDaysWithSendsPerWeek);
            return false;
        }

        int shownBlocksCount = allBlocks.count(block -> block.legacyLentaLastShowDate.isPresent());
        if (shownBlocksCount < burstBlocksCount) {
            logger.debug("Burst is active - add block today. shownBlocksCount={}", shownBlocksCount);
            return true;
        }

        if (minRepeatDays == 0) {
            return true;
        }

        double probabilityLimit = Math.min(allBlocks.size() * 1.0 / minRepeatDays, maxDaysWithSendsPerWeek / 7.0);
        boolean result = Random2.R.nextDouble() <= probabilityLimit;

        logger.debug("Use probability to add block. Limit={}%, allBlocks={} repeatDays={}, maxDaysWithSendsPerWeek={}",
                Math.round(100 * probabilityLimit), allBlocks.size(), minRepeatDays, maxDaysWithSendsPerWeek);

        return result;
    }

    public boolean shouldDoPush(PushBlockConfiguration configuration) {
        if (!pushCalculationProbabilityEnabled.get()) {
            return true;
        }
        int maxPushesPerDay = configuration.getMaxPushesPerDay();
        if (maxPushesPerDay == 0) {
            return false;
        }
        ListF<MinimalCoolLentaBlock> allBlocks = configuration.getAllBlocks();
        MapF<Integer, Integer> blocksPushedForLastWeekCounts = getActionStatisticsForLastWeek(allBlocks,
                MinimalCoolLentaBlock::getLegacyLentaLastPushDate, Instant.now(), configuration.getUserTimezone());
        logger.debug("Pushes week statistics {}", blocksPushedForLastWeekCounts);
        int pushedBlocksTodayCount = blocksPushedForLastWeekCounts.getOrElse(0, 0);
        if (pushedBlocksTodayCount >= maxPushesPerDay) {
            logger.debug("Reached limit for pushed blocks per day. pushedBlocksTodayCount={}, maxPushesPerDay={}",
                    pushedBlocksTodayCount, maxPushesPerDay);
            return false;
        }
        int maxDaysWithPushesPerWeek = configuration.getMaxDaysWithPushesPerWeek();
        int daysWithPushesForLastWeek = blocksPushedForLastWeekCounts.size();
        if (daysWithPushesForLastWeek >= maxDaysWithPushesPerWeek) {
            logger.debug("Reached limit for days with pushes per week. daysWithPushesForLastWeek={}, maxDaysWithPushesPerWeek={}",
                    daysWithPushesForLastWeek, maxDaysWithPushesPerWeek);
            return false;
        }
        int maxDaysWithSendsPerWeek = configuration.getMaxDaysWithSendsPerWeek();
        int maxSendsPerDay = configuration.getMaxSendsPerDay();
        double probabilityLimit = Math.max(minRepeatDays * 1.0 * maxDaysWithPushesPerWeek * maxPushesPerDay / (allBlocks.size() * maxDaysWithSendsPerWeek * maxSendsPerDay),
                maxDaysWithPushesPerWeek * maxPushesPerDay * 1.0 / (maxDaysWithSendsPerWeek * maxSendsPerDay));
        boolean result = Random2.R.nextDouble() <= probabilityLimit;

        logger.debug("Use probability to push block. Result={} limit={}%, allBlocks={} repeatDays={}, " +
                        "maxDaysWithPushesPerWeek={} maxDaysWithSendsPerWeek={} maxPushesPerDay={} maxSendsPerDay={}",
                result, Math.round(100 * probabilityLimit), allBlocks.size(), minRepeatDays, maxDaysWithPushesPerWeek,
                maxDaysWithSendsPerWeek, maxPushesPerDay, maxSendsPerDay);
        return result;
    }

    public static <T> MapF<Integer, Integer> getActionStatisticsForLastWeek(ListF<T> blocks,
            Function<T, Option<Instant>> actionDateRetriever, ReadableInstant now, DateTimeZone userTimezone)
    {
        return blocks
                .filter(block -> filterBlockByActionDate(block, actionDateRetriever, now, userTimezone))
                .groupBy(block -> getDaysBeforeToday(block, actionDateRetriever, now, userTimezone))
                .mapValues(ListF::size);
    }

    private static <T> boolean filterBlockByActionDate(T block,
            Function<T, Option<Instant>> actionDateRetriever, ReadableInstant now,
            DateTimeZone userTimezone)
    {
        Option<Instant> actionDateO = actionDateRetriever.apply(block);
        LocalDate today = new DateTime(now, userTimezone).toLocalDate();
        return actionDateO.isPresent() &&
                Days.daysBetween(new DateTime(actionDateO.get(), userTimezone).toLocalDate(), today).isLessThan(Days.SEVEN);
    }

    private static <T> int getDaysBeforeToday(T block,
            Function<T, Option<Instant>> actionDateRetriever, ReadableInstant now,
            DateTimeZone userTimezone) {
        return Days.daysBetween(
                new DateTime(actionDateRetriever.apply(block).get(), userTimezone).toLocalDate(),
                new DateTime(now, userTimezone).toLocalDate()
        ).getDays();
    }

    @Data
    public static class SendBlockConfiguration {
        private final ListF<MinimalCoolLentaBlock> allBlocks;
        private final int maxDaysWithSendsPerWeek;
        private final int maxSendsPerDay;
        private final DateTimeZone userTimezone;
    }

    @Data
    public static class PushBlockConfiguration {
        private final ListF<MinimalCoolLentaBlock> allBlocks;
        private final int maxDaysWithPushesPerWeek;
        private final int maxPushesPerDay;
        private final int maxDaysWithSendsPerWeek;
        private final int maxSendsPerDay;
        private final DateTimeZone userTimezone;
    }

}
