package ru.yandex.chemodan.app.lentaloader.reminder.sendpush;

import lombok.Data;
import lombok.SneakyThrows;
import org.apache.http.client.utils.URIBuilder;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataField;
import ru.yandex.chemodan.app.dataapi.api.data.filter.RecordsFilter;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataRecord;
import ru.yandex.chemodan.app.dataapi.api.db.Database;
import ru.yandex.chemodan.app.dataapi.api.deltas.Delta;
import ru.yandex.chemodan.app.dataapi.api.deltas.RecordChange;
import ru.yandex.chemodan.app.dataapi.api.deltas.RevisionCheckMode;
import ru.yandex.chemodan.app.dataapi.core.manager.DataApiManager;
import ru.yandex.chemodan.app.lentaloader.cool.CoolLentaConfigurationManager;
import ru.yandex.chemodan.app.lentaloader.cool.CoolLentaManager;
import ru.yandex.chemodan.app.lentaloader.cool.SupClient;
import ru.yandex.chemodan.app.lentaloader.reminder.CoolLentaReminder;
import ru.yandex.chemodan.app.lentaloader.reminder.MordaPhotoSelectionFields;
import ru.yandex.chemodan.app.lentaloader.reminder.titles.BlockTexts;
import ru.yandex.chemodan.app.notifier.metadata.MetadataEntity;
import ru.yandex.chemodan.app.notifier.metadata.MetadataEntityType;
import ru.yandex.chemodan.app.notifier.metadata.MetadataWrapper;
import ru.yandex.chemodan.mpfs.MpfsClient;
import ru.yandex.chemodan.mpfs.MpfsFileInfo;
import ru.yandex.chemodan.mpfs.MpfsResourceId;
import ru.yandex.chemodan.mpfs.MpfsUser;
import ru.yandex.chemodan.util.exception.PermanentHttpFailureException;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.inside.passport.blackbox2.Blackbox2;
import ru.yandex.inside.passport.blackbox2.protocol.response.BlackboxDbFields;
import ru.yandex.inside.utils.Language;
import ru.yandex.misc.io.http.HttpException;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.net.LocalhostUtils;
import ru.yandex.misc.random.Random2;

public class GncClient {

    private static final Logger logger = LoggerFactory.getLogger(GncClient.class);
    private static final String LINK = "link";

    private final SupClient supClient;
    private final MpfsClient mpfsClient;
    private final CoolLentaConfigurationManager coolLentaConfigurationManager;
    private final Blackbox2 blackbox2;
    private final DataApiManager dataApiManager;
    private final CoolLentaManager coolLentaManager;

    private final DynamicProperty<ListF<String>> thematicBlockPushTemplates = new DynamicProperty<>("cool-lenta-gnc-push-templates-thematic",
            Cf.list("gnc_new_photo_selection_thematic_1", "gnc_new_photo_selection_thematic_2", "gnc_new_photo_selection_thematic_3"));
    private final DynamicProperty<ListF<String>> nonThematicBlockPushTemplates = new DynamicProperty<>("cool-lenta-gnc-push-templates-non-thematic",
            Cf.list("gnc_new_photo_selection_1", "gnc_new_photo_selection_2", "gnc_new_photo_selection_3", "gnc_new_photo_selection_4",
                    "gnc_new_photo_selection_5", "gnc_new_photo_selection_6"));

    public GncClient(SupClient supClient, MpfsClient mpfsClient, CoolLentaConfigurationManager coolLentaConfigurationManager,
                     Blackbox2 blackbox2, DataApiManager dataApiManager, CoolLentaManager coolLentaManager) {
        this.supClient = supClient;
        this.mpfsClient = mpfsClient;
        this.coolLentaConfigurationManager = coolLentaConfigurationManager;
        this.blackbox2 = blackbox2;
        this.dataApiManager = dataApiManager;
        this.coolLentaManager = coolLentaManager;
    }

    public void sendPush(GncPushContext context) {
        logger.debug("Sending push to GNC {}", context);
        PassportUid uid = context.getUid();
        final String groupKey = uid + "_new_photos_" + Random2.R.nextAlnum(10);
        String templateName = context.getBlockGenerationType() == CoolLentaReminder.BlockGenerationType.THEMATIC ?
                Random2.R.randomElement(thematicBlockPushTemplates.get()) :
                Random2.R.randomElement(nonThematicBlockPushTemplates.get());
        logger.debug("GNC template name {}", templateName);
        final MetadataWrapper meta = MetadataWrapper.createEmpty();
        MetadataEntity photoSelectionTitleData = new MetadataEntity(MetadataEntityType.TEXT);
        photoSelectionTitleData.put("text", context.getTexts().getCoverTitle().get(getUserLanguage(uid)));
        meta.meta.put("photo_selection_title", photoSelectionTitleData);
        meta.meta.put("action", getMetadataLinkEntity(buildActionLink(context.getMordaBlockId())));
        getPreviewEntity(uid, context.getBestResourceId()).ifPresent(entity -> meta.meta.put("entity", entity));
        logger.debug("GNC meta {}", meta);
        supClient.sendPush(new String(SupGncPayloadData.serializer
                .serializeJson(SupGncPayloadData.forNewNotification(uid, templateName, meta, groupKey))
        ));
        saveGncDataForMordaBlock(context.getMordaBlockId(), groupKey, templateName, uid);
    }

    public void deleteGncNotification(PassportUid uid, String groupKey, String templateName) {
        supClient.deletePush(new String(SupGncPayloadData.serializer
                .serializeJson(SupGncPayloadData.forNotificationDeletion(uid, templateName, groupKey))));
    }

    public static MetadataEntity getMetadataLinkEntity(String linkValue) {
        MetadataEntity entity = new MetadataEntity(MetadataEntityType.LINK);
        entity.put(LINK, linkValue);
        return entity;
    }

    private void saveGncDataForMordaBlock(String mordaBlockId, String groupKey, String templateName, PassportUid uid) {
        Database mordaDatabase = coolLentaManager.getOrCreateMordaBlocksDatabase(uid);
        ListF<DataRecord> mordaBlocks = dataApiManager.getRecords(mordaDatabase.spec(), RecordsFilter.DEFAULT
                .withCollectionId(CoolLentaManager.MORDA_BLOCKS_COLLECTION_ID)
                .withRecordId(mordaBlockId));
        if (mordaBlocks.isEmpty()) {
            logger.warn("Cannot find morda block by id uid={} id={}", uid, mordaBlockId);
            return;
        }
        MapF<String, DataField> data = Cf.toHashMap(Cf.list(
                MordaPhotoSelectionFields.GNC_TEMPLATE_NAME.toData(templateName),
                MordaPhotoSelectionFields.GNC_GROUP_KEY.toData(groupKey)))
                .plus(mordaBlocks.first().data().data());
        ListF<RecordChange> changes = Cf.arrayList(RecordChange.set(
                CoolLentaManager.MORDA_BLOCKS_COLLECTION_ID,
                mordaBlockId,
                data
        ));
        dataApiManager.applyDelta(mordaDatabase, RevisionCheckMode.PER_RECORD, new Delta(changes));
    }

    @SneakyThrows
    private String buildActionLink(String mordaBlockId) {
        return new URIBuilder(coolLentaConfigurationManager.getBaseMordaBlockLink(mordaBlockId))
                .setParameter("from", "gnc").build().toString();
    }

    private Language getUserLanguage(PassportUid uid) {
        return blackbox2.query().userInfo(LocalhostUtils.localAddress(), uid, Cf.list(BlackboxDbFields.LANG))
                .getDbFields()
                .getO(BlackboxDbFields.LANG)
                .flatMapO(Language.R::fromValueO)
                .getOrElse(Language.RUSSIAN);
    }

    private Option<MetadataEntity> getPreviewEntity(PassportUid uid, MpfsResourceId mpfsResourceId) {
        return getFileInfo(uid, mpfsResourceId).map(fileInfo -> {
            MetadataEntity entityData = new MetadataEntity(MetadataEntityType.RESOURCE);
            entityData.put("preview", fileInfo.getMeta().getPreview().get().concat("&preview_type=jpg"));
            entityData.put("preview_file_path", fileInfo.path.get());
            entityData.put("preview_file_resource_id", mpfsResourceId.serialize());
            return entityData;
        });
    }

    private Option<MpfsFileInfo> getFileInfo(PassportUid uid, MpfsResourceId resourceId) {
        try {
            return mpfsClient.getFileInfoOByFileId(MpfsUser.of(uid.toString()), resourceId.owner.toString(), resourceId.fileId);
        } catch (HttpException | PermanentHttpFailureException e) {
            logger.warn("Unable to retrieve file preview for resource_id: {}, user: {}", resourceId, uid);
            return Option.empty();
        }
    }

    @Data
    public static class GncPushContext {
        private final PassportUid uid;
        private final String mordaBlockId;
        private final CoolLentaReminder.BlockGenerationType blockGenerationType;
        private final BlockTexts texts;
        private final MpfsResourceId bestResourceId;
    }
}

