package ru.yandex.chemodan.app.lentaloader.reminder.sendpush.modeling;

import org.joda.time.DateTime;
import org.joda.time.Duration;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.lentaloader.reminder.sendpush.CoolLentaBlockSendPushManager;
import ru.yandex.misc.random.Random2;
import ru.yandex.misc.time.TimeUtils;

/**
 * @author messiahlap
 */
public class SendPushProcessor {

    private static final int BURST_COUNT = 3;
    private static final int REPEAT_DAYS = 90;

    private final int sendsCount;

    private final int pushesCount;

    public SendPushProcessor(int sendsCount, int pushesCount) {
        this.sendsCount = sendsCount;
        this.pushesCount = pushesCount;
    }

    public boolean shouldSendToday(DateTime today, ListF<SendPushable> allItems) {
        MapF<Integer, Integer> itemsSentForLastWeekCounts = CoolLentaBlockSendPushManager
                .getActionStatisticsForLastWeek(allItems, SendPushable::getSendTime, today.toInstant(), TimeUtils.EUROPE_MOSCOW_TIME_ZONE);

        int daysWithSendsForLastWeek = itemsSentForLastWeekCounts.size();
        if (daysWithSendsForLastWeek >= sendsCount) {
            return false;
        }

        int shownBlocksCount = allItems.count(item -> item.getSendTime().isPresent());
        if (shownBlocksCount < BURST_COUNT) {
            return true;
        }

        double probabilityLimit = Math.min(allItems.size() * 1.0 / REPEAT_DAYS, sendsCount / 7.0);
        return Random2.R.nextDouble() <= probabilityLimit;
    }

    public Option<SendPushable> processSending(DateTime today, ListF<SendPushable> allItems) {
        ListF<SendPushable> items = allItems
                .filter(item -> !item.getSendTime().isPresent()
                        || new Duration(item.getSendTime().get(), today).isLongerThan(Duration.standardDays(REPEAT_DAYS)));
        if (items.isEmpty()) {
            return Option.empty();
        }
        SendPushable item = Random2.R.randomElement(items);
        item.setSendTime(Option.of(today.toInstant()));
        return Option.of(item);
    }

    public boolean shouldPushToday(DateTime today, ListF<SendPushable> allItems) {
        MapF<Integer, Integer> blocksPushedForLastWeekCounts = CoolLentaBlockSendPushManager
                .getActionStatisticsForLastWeek(allItems, SendPushable::getPushTime, today.toInstant(), TimeUtils.EUROPE_MOSCOW_TIME_ZONE);
        int daysWithPushesForLastWeek = blocksPushedForLastWeekCounts.size();
        if (daysWithPushesForLastWeek >= pushesCount) {
            return false;
        }
        double probabilityLimit = Math.max(REPEAT_DAYS * 1.0 * pushesCount / (allItems.size() * sendsCount),
                pushesCount * 1.0 / (sendsCount));
        //double probabilityLimit = (double) pushesCount / (7 * Math.min(allItems.size() * 1.0 / REPEAT_DAYS, sendsCount / 7.0));
        return Random2.R.nextDouble() <= probabilityLimit;
    }

    public void processPush(DateTime today, SendPushable item) {
        item.setPushTime(Option.of(today.toInstant()));
    }
}
