package ru.yandex.chemodan.app.lentaloader.reminder.special;

import org.joda.time.DateTime;
import org.joda.time.DateTimeConstants;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataField;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiPassportUserId;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.dataapi.support.I18nValue;
import ru.yandex.chemodan.app.lentaloader.cool.CoolLentaFileItem;
import ru.yandex.chemodan.app.lentaloader.cool.CoolLentaManager;
import ru.yandex.chemodan.app.lentaloader.cool.generator.BeautySelectorType;
import ru.yandex.chemodan.app.lentaloader.cool.generator.BestOfYearBlockGenerator;
import ru.yandex.chemodan.app.lentaloader.cool.generator.BlockGeneratingResult;
import ru.yandex.chemodan.app.lentaloader.cool.generator.BlockGeneratorUtils;
import ru.yandex.chemodan.app.lentaloader.cool.utils.IntervalType;
import ru.yandex.chemodan.app.lentaloader.lenta.FindOrCreateResult;
import ru.yandex.chemodan.app.lentaloader.lenta.LentaManager;
import ru.yandex.chemodan.app.lentaloader.lenta.LentaNotificationManager;
import ru.yandex.chemodan.app.lentaloader.lenta.LentaRecordType;
import ru.yandex.chemodan.app.lentaloader.lenta.update.LentaBlockCreateData;
import ru.yandex.chemodan.app.lentaloader.log.ActionInfo;
import ru.yandex.chemodan.app.lentaloader.log.ActionSource;
import ru.yandex.chemodan.app.lentaloader.reminder.PhotoSelectionFields;
import ru.yandex.chemodan.app.lentaloader.reminder.PhotoSelectionNotificationType;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.time.InstantInterval;

/**
 * @author tolmalev
 */
public class NY2018Project {
    private static final Logger logger = LoggerFactory.getLogger(NY2018Project.class);

    private final DynamicProperty<Boolean> totalEnabled = new DynamicProperty<>("lenta-by-2018-enabled", false);

    private final int weekOfYear;

    private final CoolLentaManager coolLentaManager;
    private final BestOfYearBlockGenerator bestOfYearBlockGenerator;
    private final LentaManager lentaManager;
    private final LentaNotificationManager lentaNotificationManager;

    public NY2018Project(int weekOfYear, CoolLentaManager coolLentaManager,
            BestOfYearBlockGenerator bestOfYearBlockGenerator,
            LentaManager lentaManager, LentaNotificationManager lentaNotificationManager)
    {
        this.weekOfYear = weekOfYear;
        this.coolLentaManager = coolLentaManager;
        this.bestOfYearBlockGenerator = bestOfYearBlockGenerator;
        this.lentaManager = lentaManager;
        this.lentaNotificationManager = lentaNotificationManager;
    }

    public boolean generateBlock(PassportUid uid, DateTime dateTime) {
        if (!isTimeToGenerateBlock(uid, dateTime)) {
            return false;
        }
        return generateBlockImpl(uid, dateTime);
    }

    public boolean generateBlockImpl(PassportUid uid, DateTime dateTime) {
        InstantInterval interval = new InstantInterval(IntervalType.YEAR.getIntervalStart(dateTime),
                IntervalType.YEAR.getIntervalEnd(dateTime));

        ListF<CoolLentaFileItem> files = coolLentaManager.getFiles(uid, interval, Option.of("ny-2018-project"),
                bestOfYearBlockGenerator.getBeautyLimit());
        ListF<BlockGeneratingResult> selection =
                coolLentaManager.generateBlockContentWithRetries(bestOfYearBlockGenerator, files, uid);

        if (selection.isEmpty()) {
            return false;
        }

        ListF<CoolLentaFileItem> selectedFiles = selection.first().getItems();

        ListF<String> resourceIds =
                selectedFiles.map(file -> file.mpfsFileInfo.get().getMeta().getResourceId().get().serialize());
        CoolLentaFileItem bestFile = BlockGeneratorUtils.selectBest(selectedFiles, 1, BeautySelectorType.BOTH).first();

        MapF<String, DataField> specific = Cf.toMap(Cf.list(
                PhotoSelectionFields.RESOURCE_IDS.toData(resourceIds),
                PhotoSelectionFields.BEST_RESOURCE_ID
                        .toData(bestFile.mpfsFileInfo.get().getMeta().getResourceId().get().serialize()),
                PhotoSelectionFields.GEO_COORDINATES
                        .toData(selectedFiles.map(file -> file.searchFileInfo.getGeoCoordinates())),
                PhotoSelectionFields.INTERVAL_START.toData(interval.getStart()),
                PhotoSelectionFields.INTERVAL_END.toData(interval.getEnd()),
                PhotoSelectionFields.NOTIFICATION_TYPE
                        .toData(PhotoSelectionNotificationType.NOTIFICATION_NY_2018),
                PhotoSelectionFields.SUBTYPE.toData("ny_2018"),

                PhotoSelectionFields.ICON_TYPE.toData("new_year_2018_magic_wand"),
                PhotoSelectionFields.ENABLED_PLATFORMS.toData("ios,android")
        ).plus(
                PhotoSelectionFields.TITLE.toData(new I18nValue<>(
                        "2018 в фотографиях", "2018 in pictures", "2018 у фотографіях", "Fotoğraflarla 2018"))
        ));

        String groupKey = "ny_2018";
        LentaRecordType lentaRecordType = LentaRecordType.PHOTO_SELECTION_BLOCK;
        DataApiUserId userId = new DataApiPassportUserId(uid);

        FindOrCreateResult result = lentaManager.findOrCreateBlock(userId,
                new LentaBlockCreateData(lentaRecordType, groupKey, specific),
                ActionInfo.internal(ActionSource.photoReminders()));

        lentaNotificationManager.scheduleReminderBlockNotification(userId, result.getRecordId(), false);

        logger.info("Generated ny_2018 block for uid={}, start={}, end={}, photos_count={}", uid,
                interval.getStart().getMillis(),
                interval.getEnd().getMillis(),
                selectedFiles.length());

        return true;
    }

    public boolean isTimeToGenerateBlock(PassportUid uid, DateTime dateTime) {
        if (!isEnabledForDate(dateTime)) {
            return false;
        }
        int todayDayOfWeek = dateTime.getDayOfWeek();
        if (todayDayOfWeek < DateTimeConstants.SATURDAY && todayDayOfWeek - 1 == uid.getUid() % 5) {
            return true;
        }
        return false;
    }

    public boolean isEnabledForDate(DateTime dateTime) {
        if (!totalEnabled.get()) {
            return false;
        }
        return dateTime.getYear() == 2018
                && dateTime.getMonthOfYear() == 12
                && dateTime.getWeekOfWeekyear() == weekOfYear;
    }
}
