package ru.yandex.chemodan.app.lentaloader.reminder.titles;

import org.joda.time.LocalDate;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.lentaloader.cool.utils.BlockTitlesGenerator;
import ru.yandex.chemodan.app.lentaloader.cool.utils.TankerTextGenerator;
import ru.yandex.chemodan.app.lentaloader.cool.utils.TitleGenerationContext;
import ru.yandex.chemodan.app.uaas.experiments.ExperimentsManager;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.inside.utils.DynamicLocalizedString;
import ru.yandex.inside.utils.LocalizedString;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author messiahlap
 */
public abstract class AbstractCoolLentaBlockTitlesGenerator {

    public static final String SPECIAL_DATE_EXPERIMENT_FLAG = "disk_cool_lenta_titles_special_dates";

    private static final Logger logger = LoggerFactory.getLogger(AbstractCoolLentaBlockTitlesGenerator.class);

    private final SpecialDateTitleConfigurationRegistry specialDateTitleConfigurationRegistry;

    private final TankerTextGenerator tankerTextGenerator;

    private final BlockTitlesGenerator blockTitlesGenerator;

    private final ExperimentsManager experimentsManager;

    private final DynamicProperty<Boolean> enableSpecialDatesTexts = new DynamicProperty<>("special_dates_texts_enable", false);

    public AbstractCoolLentaBlockTitlesGenerator(SpecialDateTitleConfigurationRegistry specialDateTitleConfigurationRegistry,
            TankerTextGenerator tankerTextGenerator, BlockTitlesGenerator blockTitlesGenerator,
            ExperimentsManager experimentsManager)
    {
        this.specialDateTitleConfigurationRegistry = specialDateTitleConfigurationRegistry;
        this.tankerTextGenerator = tankerTextGenerator;
        this.blockTitlesGenerator = blockTitlesGenerator;
        this.experimentsManager = experimentsManager;
    }

    public Option<BlockTexts> generateBlockTexts(CoolLentaBlockTitlesManager.TitleParameters titleParameters) {
        Option<BlockTexts> result = enableSpecialDatesTexts.get() || experimentsManager.getFlags(titleParameters.getUid().toUidOrZero().getUid())
                .containsTs(SPECIAL_DATE_EXPERIMENT_FLAG) ? generateTextsForSpecialDate(titleParameters) : Option.empty();
        if (result.isPresent()) {
            return result;
        }
        TitleGenerationContext context = titleParameters.getTitleGenerationContext();
        Option<LocalizedString> coverTitleO = getCoverTitle(titleParameters);
        if (!coverTitleO.isPresent()) {
            return Option.empty();
        }
        return Option.of(new BlockTexts(context, getLegacyTitle(context, coverTitleO.get()), coverTitleO.get(), getCoverSubtitle(context),
                getButtonText(context), getGeoIds(context)));
    }

    abstract public BlockTitlesType getTitlesType();

    protected ListF<String> getKeysSuffixQueue(CoolLentaBlockTitlesManager.TitleParameters titleParameters) {
        return Cf.list();
    }

    protected LocalizedString getCoverSubtitle(TitleGenerationContext context) {
        return blockTitlesGenerator.generateBlockSubtitle(context);
    }

    protected LocalizedString getLegacyTitle(TitleGenerationContext context, LocalizedString coverTitle) {
        return blockTitlesGenerator.generateLegacyBlockTitle(context);
    }

    protected LocalizedString getButtonText(TitleGenerationContext context) {
        return blockTitlesGenerator.generatePhotosliceLinkText(context);
    }

    protected ListF<Integer> getGeoIds(TitleGenerationContext context) {
        return Cf.list();
    }

    protected TankerTextGenerator getTankerTextGenerator() {
        return tankerTextGenerator;
    }

    protected BlockTitlesGenerator getBlockTitlesGenerator() {
        return blockTitlesGenerator;
    }

    abstract protected Option<LocalizedString> getCoverTitle(CoolLentaBlockTitlesManager.TitleParameters titleParameters);

    /**
     * The common template for title and subtitle tanker keys:
     *      - cool_lenta_(title|subtitle)_<special_date_name>_[thematic]_[theme]_v(version_number)
     *  Examples:
     *      - cool_lenta_title_new_year_thematic_cats_v1 - will be used only for thematic blocks with theme cats
     *      - cool_lenta_subtitle_new_year_thematic_v1 - will be used for all thematic blocks
     *      - cool_lenta_title_new_year_v1 - will be used for all blocks
     * @param titleParameters if date is "special" then BlockTexts will be generated for this date
     * @return BlockTexts if the date is "special" else Option.empty()
     */
    protected Option<BlockTexts> generateTextsForSpecialDate(CoolLentaBlockTitlesManager.TitleParameters titleParameters) {
        LocalDate date = titleParameters.getDate();
        Option<SpecialDateTitleConfiguration> configurationO = specialDateTitleConfigurationRegistry
                .getSpecialDateConfigurationForDateO(new SpecialDateTitleConfiguration.DayWithMonth(date.getMonthOfYear(), date.getDayOfMonth()));
        logger.debug("special date configuration {}", configurationO);
        if (!configurationO.isPresent()) {
            return Option.empty();
        }
        String dateName = configurationO.get().getSpecialDateName();
        ListF<String> keysTemplateQueue = getKeysSuffixQueue(titleParameters).map(suffix -> "lenta/cool_lenta_%s_" + dateName + "_" + suffix)
                .plus("lenta/cool_lenta_%s_" + dateName);
        logger.debug("special date templates queue {}", keysTemplateQueue);
        TitleGenerationContext context = titleParameters.getTitleGenerationContext();
        Option<LocalizedString> coverTitleO = buildCoverTitle(dateName, keysTemplateQueue, context, titleParameters);
        if (!coverTitleO.isPresent()) {
            return Option.empty();
        }
        LocalizedString coverTitle = coverTitleO.get();
        LocalizedString coverSubtitle = buildCoverSubtitle(dateName, keysTemplateQueue, context);
        return Option.of(new BlockTexts(context, getLegacyTitle(context, coverTitle), coverTitle, coverSubtitle, getButtonText(context), getGeoIds(context)));
    }

    private Option<LocalizedString> buildCoverTitle(String specialDateId, ListF<String> keysQueue,
            TitleGenerationContext context, CoolLentaBlockTitlesManager.TitleParameters titleParameters) {
        String appendKey = "lenta/cool_lenta_title_" + specialDateId + "_append";
        if (tankerTextGenerator.containsAnyVersionOfKey(appendKey)) {
            logger.debug("Found append key {}", appendKey);
            Option<String> lineToAppendO = tankerTextGenerator.processOneFromSetOrTemplatesByKeyPrefix(appendKey, context).getAll()
                    .values().findNot(StringUtils::isBlank);
            if (lineToAppendO.isPresent()) {
                String lineToAppend = lineToAppendO.get();
                return getCoverTitle(titleParameters).map(localizedString ->
                        new DynamicLocalizedString(localizedString.getAll().mapValues(value -> value + " " + lineToAppend)));
            }
            logger.debug("Append line is empty");
        }
        Option<LocalizedString> coverTitleO = getLocalizedStringForKeysQueue(keysQueue, "title", context);
        if (!coverTitleO.isPresent()) {
            return getCoverTitle(titleParameters);
        }
        return coverTitleO;
    }

    private LocalizedString buildCoverSubtitle(String specialDateId, ListF<String> keysQueue,
            TitleGenerationContext context) {
        String appendKey = "lenta/cool_lenta_subtitle_" + specialDateId + "_append";
        if (tankerTextGenerator.containsAnyVersionOfKey(appendKey)) {
            logger.debug("Found append key {}", appendKey);
            Option<String> lineToAppendO = tankerTextGenerator.processOneFromSetOrTemplatesByKeyPrefix(appendKey, context).getAll()
                    .values().findNot(StringUtils::isBlank);
            if (lineToAppendO.isPresent()) {
                String lineToAppend = lineToAppendO.get();
                return new DynamicLocalizedString(getCoverSubtitle(context).getAll().mapValues(value -> value + " " + lineToAppend));
            }
            logger.debug("Append line is empty");
        }
        return getLocalizedStringForKeysQueue(keysQueue, "subtitle", context)
                .getOrElse(() -> getCoverSubtitle(context));
    }

    private Option<LocalizedString> getLocalizedStringForKeysQueue(ListF<String> keysQueue, String type,
            TitleGenerationContext context)
    {
        for (String keyTemplate : keysQueue) {
            String keyPrefix = String.format(keyTemplate, type);
            logger.debug("special date key prefix {}", keyPrefix);
            if (!tankerTextGenerator.containsAnyVersionOfKey(keyPrefix)) {
                continue;
            }
            return Option.of(tankerTextGenerator.processOneFromSetOrTemplatesByKeyPrefix(keyPrefix, context));
        }
        return Option.empty();
    }
}
