package ru.yandex.chemodan.app.lentaloader.reminder.titles;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.chemodan.app.lentaloader.cool.utils.BlockTitlesGenerator;
import ru.yandex.chemodan.app.lentaloader.cool.utils.TankerTextGenerator;
import ru.yandex.chemodan.app.lentaloader.cool.utils.TitleGenerationContext;
import ru.yandex.chemodan.app.uaas.experiments.ExperimentsManager;
import ru.yandex.inside.geobase.Geobase6;
import ru.yandex.inside.geobase.LinguisticsItem;
import ru.yandex.inside.geobase.RegionNode;
import ru.yandex.inside.geobase.RegionType;
import ru.yandex.inside.utils.DynamicLocalizedString;
import ru.yandex.inside.utils.Language;
import ru.yandex.inside.utils.LocalizedString;
import ru.yandex.misc.geo.Coordinates;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author messiahlap
 */
public class GeoCoolLentaBlockTitlesGenerator extends AbstractCoolLentaBlockTitlesGenerator {

    public static final String COORDINATES_ATTRIBUTE_NAME = "coordinates";
    public static final String LEGACY_TITLE_IS_GEO_ATTRIBUTE_NAME = "legacy_title_is_geo";

    private static final Logger logger = LoggerFactory.getLogger(GeoCoolLentaBlockTitlesGenerator.class);

    private final Geobase6 geobase;

    public GeoCoolLentaBlockTitlesGenerator(SpecialDateTitleConfigurationRegistry specialDateTitleConfigurationRegistry,
            TankerTextGenerator tankerTextGenerator, BlockTitlesGenerator blockTitlesGenerator, Geobase6 geobase, ExperimentsManager experimentsManager)
    {
        super(specialDateTitleConfigurationRegistry, tankerTextGenerator, blockTitlesGenerator, experimentsManager);
        this.geobase = geobase;
    }

    @Override
    public BlockTitlesType getTitlesType() {
        return BlockTitlesType.GEO;
    }

    @Override
    protected Option<LocalizedString> getCoverTitle(CoolLentaBlockTitlesManager.TitleParameters titleParameters) {
        return getFinalGeoName(titleParameters.getTitleGenerationContext());
    }

    @Override
    protected LocalizedString getLegacyTitle(TitleGenerationContext context, LocalizedString coverTitle) {
        LocalizedString defaultLegacyTitle = super.getLegacyTitle(context, coverTitle);
        if (!(Boolean) context.getAttributes().getOrThrow(LEGACY_TITLE_IS_GEO_ATTRIBUTE_NAME)) {
            return defaultLegacyTitle;
        }
        return coverTitle.append(". ").append(defaultLegacyTitle);
    }

    private Option<LocalizedString> getFinalGeoName(TitleGenerationContext titleGenerationContext) {
        try {
            ListF<RegionNode> nodes = getNodes(titleGenerationContext);

            if (nodes.isEmpty()) {
                logger.debug("Not Generated block with geo: No good geo nodes found.");
                return Option.empty();
            }

            ListF<MapF<Language, LinguisticsItem>> names = nodes.map(n -> {
                LinguisticsItem defaultName = geobase.getLinguisticsItemByRegionId(n.getId(), "ru").get();

                return Cf.x(Language.values()).toMapMappingToValue(lang ->
                        geobase.getLinguisticsItemByRegionId(n.getId(), lang.value()).getOrElse(defaultName));
            });

            MapF<Language, String> concatenators = Cf.map(
                    Language.RUSSIAN, " и ",
                    Language.ENGLISH, " and ",
                    Language.UKRAINIAN, " i ",
                    Language.TURKISH, " ve "
            );

            MapF<Language, String> finalGeoName = Cf.x(concatenators.keySet()).toMapMappingToValue(
                    lang -> names.map(m -> m.getTs(lang).getNominativeCase()).mkString(concatenators.getTs(lang))
            );
            return Option.of(new DynamicLocalizedString(finalGeoName));
        } catch (RuntimeException e) {
            logger.error("Failed to generate geo titles for block: {}", e);
            return Option.empty();
        }
    }

    @Override
    protected ListF<Integer> getGeoIds(TitleGenerationContext context) {
        return getNodes(context).map(RegionNode::getId);
    }

    private ListF<RegionNode> getNodes(TitleGenerationContext context) {
        ListF<Coordinates> coordinates = context.getAttributes()
                .getO(COORDINATES_ATTRIBUTE_NAME)
                .filter(ListF.class::isInstance)
                .map(ListF.class::cast).getOrThrow("Cannot get the coordinates")
                .map(Coordinates.class::cast);

        ListF<RegionNode> regionNodes = coordinates
                .flatMap(geobase::getRegionIdByCoordinates)
                .flatMap(geobase::getRegionById);

        logger.debug("ALL GEO INFORMATION");

        MapF<RegionType, MapF<Integer, Integer>> nodesByType = regionNodes
                .flatMap(n -> geobase.getParentsById(n.getId()).plus(n))
                .groupBy(RegionNode::getType)
                .mapValues(v -> v.map(RegionNode::getId).countBy());

        MapF<Integer, Integer> countByNode =
                regionNodes.flatMap(n -> geobase.getParentsById(n.getId()).plus(n)).countBy(RegionNode::getId);

        Tuple2List<RegionType, Integer> goodGeoTypes = Tuple2List.fromPairs(
//                    skipped for now
//                    RegionType.CITY_MICRODISTRICT, 1,
                RegionType.VILLAGE, 2,
                RegionType.CITY, 2,
                RegionType.FEDERAL_SUBJECT, 1,
                RegionType.COUNTRY, 2
        );

        ListF<RegionNode> nodes = Cf.arrayList();

        for (Tuple2<RegionType, Integer> geoT2 : goodGeoTypes) {
            RegionType geoType = geoT2._1;
            int maxDifferentNodes = geoT2._2;
            MapF<Integer, Integer> regionIds = nodesByType.getTs(geoType);

            if (regionIds != null
                    && regionIds.size() <= maxDifferentNodes
                    && regionIds.values().sum(Cf.Integer) == coordinates.size()) {
                nodes = regionIds.keySet().flatMap(geobase::getRegionById)
                        .filter(rn -> geobase.getLinguisticsItemByRegionId(rn.getId(), "ru").isPresent())
                        .sortedByDesc(n -> countByNode.getOrElse(n.getId(), 0));

                if (!nodes.isEmpty()) {
                    break;
                }
            }
        }

        nodesByType.keySet().sortedByDesc(RegionType::value).forEach(type -> {
            logger.debug("{}: {}", type, nodesByType.getTs(type)
                    .filter((n, cnt) -> geobase.getLinguisticsItemByRegionId(n, "ru").isPresent())
                    .mapEntries((n, cnt) -> geobase.getLinguisticsItemByRegionId(n, "ru").get()
                            .getNominativeCase() + "(" + cnt + ")").mkString(", "));
        });

        return nodes;
    }
}
