package ru.yandex.chemodan.app.lentaloader.reminder.titles;

import org.joda.time.DateTime;
import org.joda.time.LocalDate;
import org.junit.Before;
import org.junit.Test;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.lentaloader.cool.generator.ThemeDefinition;
import ru.yandex.chemodan.app.lentaloader.cool.generator.ThemeDefinitionRegistry;
import ru.yandex.chemodan.app.lentaloader.cool.utils.BlockTitlesGenerator;
import ru.yandex.chemodan.app.lentaloader.cool.utils.GeoNamesSource;
import ru.yandex.chemodan.app.lentaloader.cool.utils.IntervalType;
import ru.yandex.chemodan.app.lentaloader.cool.utils.TankerTextGenerator;
import ru.yandex.chemodan.app.lentaloader.cool.utils.TermDefinition;
import ru.yandex.chemodan.app.lentaloader.cool.utils.TermLanguageDefinition;
import ru.yandex.chemodan.app.lentaloader.cool.utils.TextProcessor;
import ru.yandex.chemodan.app.lentaloader.cool.utils.TitleGenerationContext;
import ru.yandex.chemodan.app.lentaloader.test.TestUtils;
import ru.yandex.chemodan.app.uaas.experiments.ExperimentsManager;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.inside.utils.Language;
import ru.yandex.misc.io.ClassPathResourceInputStreamSource;
import ru.yandex.misc.random.Random2;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.ArgumentMatchers.anyLong;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;
import static ru.yandex.chemodan.app.lentaloader.reminder.CoolLentaReminder.THEME_ID_ATTRIBUTE_NAME;

/**
 * @author messiahlap
 */
public class SpecialDatesTitlesTest {

    private static final int TEST_EVENT_MONTH = 10;

    private static final int TEST_EVENT_DAY = 31;

    private static final String TEST_EVENT_NAME = "test_event";

    private final SpecialDateTitleConfigurationRegistry specialDateTitleConfigurationRegistry =
            mock(SpecialDateTitleConfigurationRegistry.class);

    private final TankerTextGenerator tankerTextGenerator = new TankerTextGenerator(
            new ClassPathResourceInputStreamSource(AbstractCoolLentaBlockTitlesGenerator.class, "special_dates_titles_test.json"),
            new TextProcessor(GeoNamesSource.EMPTY));

    private final BlockTitlesGenerator blockTitlesGenerator = new BlockTitlesGenerator(GeoNamesSource.EMPTY);

    private final ThemeDefinitionRegistry themeDefinitionRegistry = mock(ThemeDefinitionRegistry.class);

    private final ExperimentsManager experimentsManager = mock(ExperimentsManager.class);

    private final ThematicCoolLentaBlockTitlesGenerator thematicCoolLentaBlockTitlesGenerator =
            new ThematicCoolLentaBlockTitlesGenerator(specialDateTitleConfigurationRegistry, tankerTextGenerator,
                    blockTitlesGenerator, experimentsManager, themeDefinitionRegistry);

    private final DefaultCoolLentaBlockTitlesGenerator defaultCoolLentaBlockTitlesGenerator =
            new DefaultCoolLentaBlockTitlesGenerator(specialDateTitleConfigurationRegistry, tankerTextGenerator,
                    blockTitlesGenerator, experimentsManager);

    private final PassportUid uid = PassportUid.cons(2L);

    @Before
    public void initMocks() {
        when(specialDateTitleConfigurationRegistry
                .getSpecialDateConfigurationForDateO(any(SpecialDateTitleConfiguration.DayWithMonth.class)))
                .thenAnswer(invocation -> {
                    if (new SpecialDateTitleConfiguration.DayWithMonth(TEST_EVENT_MONTH, TEST_EVENT_DAY).equals(invocation.getArgument(0))) {
                        return Option.of(new SpecialDateTitleConfiguration(TEST_EVENT_NAME,
                                Cf.list(new SpecialDateTitleConfiguration.DayWithMonth(TEST_EVENT_MONTH, TEST_EVENT_DAY))));
                    }
                    return Option.empty();
                });
        when(themeDefinitionRegistry.getO("nature")).thenReturn(Option.of(TestUtils.createNatureThemeDefinition()));
        when(themeDefinitionRegistry.getO("people")).thenReturn(Option.of(new ThemeDefinition("people", Cf.list(), createPeopleTermDefinition(),
                true, Option.of(false), Option.of(false), Option.of(false), Option.empty())));
        when(experimentsManager.getFlags(anyLong())).thenReturn(Cf.list(AbstractCoolLentaBlockTitlesGenerator.SPECIAL_DATE_EXPERIMENT_FLAG));
    }

    @Test
    public void testSpecialDateTitleForSpecificTheme() {
        TitleGenerationContext context = new TitleGenerationContext(Random2.R, IntervalType.MONTH, new DateTime(1107205200000L))
                .withTerm(createPeopleTermDefinition())
                .withAttribute(THEME_ID_ATTRIBUTE_NAME, "people");
        Option<BlockTexts> blockTextsO = thematicCoolLentaBlockTitlesGenerator.generateBlockTexts(new CoolLentaBlockTitlesManager.TitleParameters(
                context, new LocalDate(2019, TEST_EVENT_MONTH, TEST_EVENT_DAY), Option.of("people"), uid
                ));
        assertTrue(blockTextsO.isPresent());
        BlockTexts blockTexts = blockTextsO.get();
        assertEquals("Тестовый тайтл для Встреч", blockTexts.getCoverTitle().get(Language.RUSSIAN));
        assertEquals("Тестовый субтайтл для Встреч", blockTexts.getCoverSubtitle().get(Language.RUSSIAN));
    }

    @Test
    public void testSpecialDateTitleForCommonTheme() {
        TitleGenerationContext context = new TitleGenerationContext(Random2.R, IntervalType.MONTH, new DateTime(1107205200000L))
                .withTerm(TestUtils.createNatureThemeDefinition().getForms())
                .withAttribute(THEME_ID_ATTRIBUTE_NAME, "nature");
        Option<BlockTexts> blockTextsO = thematicCoolLentaBlockTitlesGenerator.generateBlockTexts(new CoolLentaBlockTitlesManager.TitleParameters(
                context, new LocalDate(2019, TEST_EVENT_MONTH, TEST_EVENT_DAY), Option.of("nature"), uid
        ));
        assertTrue(blockTextsO.isPresent());
        BlockTexts blockTexts = blockTextsO.get();
        assertEquals("Тестовый тайтл для тематических подборок", blockTexts.getCoverTitle().get(Language.RUSSIAN));
        assertEquals("Тестовый субтайтл для тематических подборок", blockTexts.getCoverSubtitle().get(Language.RUSSIAN));
    }

    @Test
    public void testSpecialDateTitleForDefaultBlock() {
        TitleGenerationContext context = new TitleGenerationContext(Random2.R, IntervalType.MONTH, new DateTime(1107205200000L));
        Option<BlockTexts> blockTextsO = defaultCoolLentaBlockTitlesGenerator.generateBlockTexts(new CoolLentaBlockTitlesManager.TitleParameters(
                context, new LocalDate(2019, TEST_EVENT_MONTH, TEST_EVENT_DAY), Option.empty(), uid
        ));
        assertTrue(blockTextsO.isPresent());
        BlockTexts blockTexts = blockTextsO.get();
        assertEquals("Тестовый тайтл для всех подборок", blockTexts.getCoverTitle().get(Language.RUSSIAN));
        assertEquals("Тестовый субтайтл для всех подборок", blockTexts.getCoverSubtitle().get(Language.RUSSIAN));
    }

    @Test
    public void testNonSpecialDateTitleForThematicBlock() {
        TitleGenerationContext context = new TitleGenerationContext(Random2.R, IntervalType.MONTH, new DateTime(1107205200000L))
                .withTerm(createPeopleTermDefinition())
                .withAttribute(THEME_ID_ATTRIBUTE_NAME, "people");
        Option<BlockTexts> blockTextsO = thematicCoolLentaBlockTitlesGenerator.generateBlockTexts(new CoolLentaBlockTitlesManager.TitleParameters(
                context, new LocalDate(2019, 10, 30), Option.of("people"), uid
        ));
        assertTrue(blockTextsO.isPresent());
        BlockTexts blockTexts = blockTextsO.get();
        assertEquals("Подборка ваших фото с людьми", blockTexts.getCoverTitle().get(Language.RUSSIAN));
    }

    @Test
    public void testSpecialDateAppendTitle() {
        TankerTextGenerator tankerTextGenerator = new TankerTextGenerator(
                new ClassPathResourceInputStreamSource(AbstractCoolLentaBlockTitlesGenerator.class, "special_dates_with_appends.json"),
                new TextProcessor(GeoNamesSource.EMPTY));
        TitleGenerationContext context = new TitleGenerationContext(Random2.R, IntervalType.MONTH, new DateTime(1107305200000L))
                .withTerm(createPeopleTermDefinition())
                .withAttribute(THEME_ID_ATTRIBUTE_NAME, "people");
        Option<BlockTexts> blockTextsO = new ThematicCoolLentaBlockTitlesGenerator(specialDateTitleConfigurationRegistry, tankerTextGenerator,
                blockTitlesGenerator, experimentsManager, themeDefinitionRegistry).generateBlockTexts(new CoolLentaBlockTitlesManager.TitleParameters(
                context, new LocalDate(2019, TEST_EVENT_MONTH, TEST_EVENT_DAY), Option.of("people"), uid
        ));
        assertTrue(blockTextsO.isPresent());
        BlockTexts blockTexts = blockTextsO.get();
        assertEquals("Подборка ваших фото с людьми ❄", blockTexts.getCoverTitle().get(Language.RUSSIAN));
        assertEquals("Февраль 2005 \uD83C\uDF84️", blockTexts.getCoverSubtitle().get(Language.RUSSIAN));
    }


    private TermDefinition createPeopleTermDefinition() {
        TermLanguageDefinition russianTD = new TermLanguageDefinition("встречи", "людей", "о", "людях", "людьми", "людей", "к людям", Option.empty());
        TermLanguageDefinition turkishTD = new TermLanguageDefinition("buluşmalar", "insanların", "", "insanları", "insanlara", "insanlarda", "insanlardan", Option.empty());
        TermLanguageDefinition englishTD = new TermLanguageDefinition("get-togethers", "of people", "of", "people", "people", "people", "to people", Option.empty());
        TermLanguageDefinition ukranianTD = new TermLanguageDefinition("зустрічі", "людей", "про", "людей", "людьми", "людей", "до людей", Option.empty());
        return new TermDefinition(Cf.map(Language.RUSSIAN, russianTD,
                Language.TURKISH, turkishTD,
                Language.ENGLISH, englishTD,
                Language.UKRAINIAN, ukranianTD));
    }
}

