package ru.yandex.chemodan.app.lentaloader.reminder.titles;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.lentaloader.cool.generator.ThematicBlocksGenerator;
import ru.yandex.chemodan.app.lentaloader.cool.generator.ThemeDefinition;
import ru.yandex.chemodan.app.lentaloader.cool.generator.ThemeDefinitionRegistry;
import ru.yandex.chemodan.app.lentaloader.cool.utils.BlockTitlesGenerator;
import ru.yandex.chemodan.app.lentaloader.cool.utils.IntervalType;
import ru.yandex.chemodan.app.lentaloader.cool.utils.TankerTextGenerator;
import ru.yandex.chemodan.app.lentaloader.cool.utils.TitleGenerationContext;
import ru.yandex.chemodan.app.uaas.experiments.ExperimentsManager;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.inside.utils.DynamicLocalizedString;
import ru.yandex.inside.utils.Language;
import ru.yandex.inside.utils.LocalizedString;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.random.Random2;

import static ru.yandex.chemodan.app.lentaloader.reminder.CoolLentaReminder.THEME_ID_ATTRIBUTE_NAME;

/**
 * @author messiahlap
 */
public class ThematicCoolLentaBlockTitlesGenerator extends AbstractCoolLentaBlockTitlesGenerator {

    private static final String THEMATIC_TITLE_TEXT_TANKER_KEYS_PREFIX =
            "lenta/thematic_blocks_text";

    private final ThemeDefinitionRegistry themeDefinitionRegistry;

    private final DynamicProperty<Integer> nonCustomTitleProbability =
            new DynamicProperty<>("cool-lenta-thematic-non-custom-titles-probability-percent", 30);

    public ThematicCoolLentaBlockTitlesGenerator(SpecialDateTitleConfigurationRegistry specialDateTitleConfigurationRegistry,
            TankerTextGenerator tankerTextGenerator, BlockTitlesGenerator blockTitlesGenerator, ExperimentsManager experimentsManager,
            ThemeDefinitionRegistry themeDefinitionRegistry)
    {
        super(specialDateTitleConfigurationRegistry, tankerTextGenerator, blockTitlesGenerator, experimentsManager);
        this.themeDefinitionRegistry = themeDefinitionRegistry;
    }

    @Override
    public BlockTitlesType getTitlesType() {
        return BlockTitlesType.THEMATIC;
    }

    @Override
    protected ListF<String> getKeysSuffixQueue(CoolLentaBlockTitlesManager.TitleParameters titleParameters) {
        Validate.isTrue(titleParameters.getThemeName().isPresent());
        String theme = titleParameters.getThemeName().get();
        return Cf.list(ThematicBlocksGenerator.BLOCK_ID_PREFIX + "_" + theme,
                ThematicBlocksGenerator.BLOCK_ID_PREFIX);
    }

    @Override
    protected LocalizedString getCoverSubtitle(TitleGenerationContext context) {
        if (IntervalType.SEASON == context.getIntervalType()) {
            return getBlockTitlesGenerator().generateLegacyBlockTitle(context);
        }
        if (IntervalType.YEAR == context.getIntervalType()) {
            return getTankerTextGenerator().processTemplateByKey("lenta/thematic_blocks_year_subtitle_v1", context);
        }
        return getBlockTitlesGenerator().generateBlockSubtitle(context);
    }

    @Override
    protected Option<LocalizedString> getCoverTitle(CoolLentaBlockTitlesManager.TitleParameters titleParameters) {
        Validate.isTrue(titleParameters.getThemeName().isPresent());
        TitleGenerationContext titleGenerationContext = titleParameters.getTitleGenerationContext();
        LocalizedString nonCustomTitle = getTankerTextGenerator()
                .processOneFromSetOrTemplatesByKeyPrefix(THEMATIC_TITLE_TEXT_TANKER_KEYS_PREFIX, titleGenerationContext);
        String customTitleKeyPrefix = String.format("lenta/thematic_custom_title_%s",
                titleGenerationContext.getAttributes().getOrThrow(THEME_ID_ATTRIBUTE_NAME));
        String themeName = titleParameters.getThemeName().get();
        ThemeDefinition theme = themeDefinitionRegistry.getO(themeName).getOrThrow("Cannot find definition for theme " + themeName);
        if (!getTankerTextGenerator().containsAnyVersionOfKey(customTitleKeyPrefix)) {
            if (theme.useCustomTitlesOnly()) {
                throw new IllegalStateException(String.format("No custom title for theme id {}",
                        titleGenerationContext.getAttributes().getOrThrow(THEME_ID_ATTRIBUTE_NAME)));
            }
            return Option.of(nonCustomTitle);
        }
        LocalizedString customTitle = getTankerTextGenerator()
                .processOneFromSetOrTemplatesByKeyPrefix(customTitleKeyPrefix, titleGenerationContext);
        if (theme.useCustomTitlesOnly()) {
            checkCustomTitle(customTitle, nonCustomTitle);
            return Option.of(customTitle);
        }
        if (Random2.R.nextInt(100) < nonCustomTitleProbability.get()) {
            return Option.of(nonCustomTitle);
        }
        for (Language language : nonCustomTitle.getAll().keys()) {
            if (StringUtils.isBlank(customTitle.getO(language).getOrElse(""))) {
                return Option.of(mergeTitles(customTitle, nonCustomTitle));
            }
        }
        return Option.of(customTitle);
    }

    private void checkCustomTitle(LocalizedString customTitle, LocalizedString nonCustomTitle) {
        for (Language language : nonCustomTitle.getAll().keys()) {
            if (StringUtils.isBlank(customTitle.getO(language).getOrElse(""))) {
                throw new IllegalStateException(String.format("Invalid custom title title=%s language=%s", customTitle, language));
            }
        }
    }

    private LocalizedString mergeTitles(LocalizedString customTitle, LocalizedString nonCustomTitle) {
        MapF<Language, String> values = Cf.hashMap();
        for (Language language : nonCustomTitle.getAll().keys()) {
            if (StringUtils.isBlank(customTitle.getO(language).getOrElse(""))) {
                values.put(language, nonCustomTitle.get(language));
            }
        }
        return new DynamicLocalizedString(values.plus(customTitle.getAll()));
    }
}
