package ru.yandex.chemodan.app.lentaloader.web;

import lombok.Data;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple3;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataField;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.dataapi.support.I18nValue;
import ru.yandex.chemodan.app.lentaloader.LentaEventsLogListener;
import ru.yandex.chemodan.app.lentaloader.blocks.FolderCreationBlockManager;
import ru.yandex.chemodan.app.lentaloader.lenta.LentaBlockRecord;
import ru.yandex.chemodan.app.lentaloader.lenta.LentaManager;
import ru.yandex.chemodan.app.lentaloader.lenta.update.DeleteHandler;
import ru.yandex.chemodan.app.lentaloader.log.ActionInfo;
import ru.yandex.chemodan.app.lentaloader.log.ActionReason;
import ru.yandex.chemodan.app.lentaloader.log.ActionSource;
import ru.yandex.chemodan.app.lentaloader.log.ActionSourceHolder;
import ru.yandex.chemodan.app.lentaloader.log.LentaBlockEvent;
import ru.yandex.chemodan.app.lentaloader.log.LentaEventType;
import ru.yandex.chemodan.app.lentaloader.reminder.GenerateBlockTask;
import ru.yandex.chemodan.app.lentaloader.reminder.PhotoSelectionFields;
import ru.yandex.chemodan.mpfs.MpfsResourceId;
import ru.yandex.chemodan.util.exception.BadRequestException;
import ru.yandex.chemodan.util.exception.NotFoundException;
import ru.yandex.chemodan.util.tskv.TskvUtils;
import ru.yandex.chemodan.util.web.ErrorPojo;
import ru.yandex.chemodan.util.web.OkPojo;
import ru.yandex.commune.a3.action.Action;
import ru.yandex.commune.a3.action.ActionContainer;
import ru.yandex.commune.a3.action.HttpMethod;
import ru.yandex.commune.a3.action.Path;
import ru.yandex.commune.a3.action.parameter.bind.annotation.RequestParam;
import ru.yandex.commune.a3.action.parameter.bind.annotation.SpecialParam;
import ru.yandex.commune.a3.action.result.pojo.ActionResultPojo;
import ru.yandex.commune.bazinga.BazingaTaskManager;
import ru.yandex.commune.bazinga.impl.FullJobId;
import ru.yandex.commune.bazinga.impl.JobInfoValue;
import ru.yandex.commune.bazinga.impl.JobStatus;
import ru.yandex.commune.bazinga.impl.OnetimeJob;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.inside.utils.Language;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.web.servlet.HttpServletRequestX;

/**
 * @author dbrylev
 */
@ActionContainer
public class LentaActions {
    private static final Logger logger = LoggerFactory.getLogger(LentaActions.class);

    private final LentaManager lentaManager;
    private final FolderCreationBlockManager folderCreationBlockManager;
    private final LentaEventsLogListener lentaEventsLogListener;
    private final BazingaTaskManager bazingaTaskManager;

    private final DynamicProperty<Integer> generateRandomBlockRetryCount =
            new DynamicProperty<>("lenta-action-generation-block-retry-count", 10);

    public LentaActions(LentaManager lentaManager,
            FolderCreationBlockManager folderCreationBlockManager,
            LentaEventsLogListener lentaEventsLogListener,
            BazingaTaskManager bazingaTaskManager)
    {
        this.lentaManager = lentaManager;
        this.folderCreationBlockManager = folderCreationBlockManager;
        this.lentaEventsLogListener = lentaEventsLogListener;
        this.bazingaTaskManager = bazingaTaskManager;
    }

    @Action
    public OkPojo deleteEmptyBlock(
            @RequestParam("uid") DataApiUserId uid,
            @RequestParam("blockId") Option<String> blockId)
    {
        ActionInfo actionInfo = ActionInfo.internal(ActionSource.lentaApi());

        blockId.flatMapO(id -> lentaManager.findBlock(uid, id, actionInfo))
                .forEach(block -> lentaManager.deleteEmptyBlockDelayed(
                        uid, block.getIdAndType(), actionInfo.withReason(ActionReason.BAD_BLOCK_COMPLAIN)));

        return new OkPojo();
    }

    @Action
    public OkPojo deleteBlock(
            @RequestParam("uid") DataApiUserId uid,
            @RequestParam("blockId") String blockId)
    {
        lentaManager.deleteBlock(uid, blockId, rec -> {
            return DeleteHandler.delete(ActionReason.USER_REQUEST);

        }, ActionInfo.internal(ActionSource.lentaApi()));

        return new OkPojo();
    }

    @Action
    // maybe should be removed. It should be decided during working on https://wiki.yandex-team.ru/users/slukyanenko/Scenarii-dlja-raboty-s-Alisojj/
    public OkPojo reportBlockVisit(@RequestParam("uid") DataApiUserId uid, @RequestParam("blockId") String blockId,
            @RequestParam("os") String os)
    {
        return new OkPojo();
    }

    @Action
    @Path(value = "/blocks/types/folder_creation", methods = HttpMethod.POST)
    public OkPojo createFolderCreationBlock(
            @RequestParam("uid") DataApiUserId uid,
            @RequestParam("folderId") MpfsResourceId folderId)
    {
        folderCreationBlockManager.createDirectly(uid, folderId, ActionInfo.internal(ActionSource.lentaApi()));

        return new OkPojo();
    }

    @Action
    public OkPojo processLogLine(@SpecialParam HttpServletRequestX req) {
        ActionSourceHolder.invokeWith(ActionSource.lentaApi(),
                () -> lentaEventsLogListener.processLogLine(new String(req.getInputStreamX().readBytes())));

        return new OkPojo();
    }

    @Action
    public CreatedOrUpdatedBlockInfo startSavingFolderFromPublic(
            @RequestParam("uid") DataApiUserId uid,
            @RequestParam("folderId") MpfsResourceId folderId,
            @RequestParam("folderPath") Option<String> folderPath)
    {
        return resolveCreatedBlockInfo(folderCreationBlockManager.handleStartSavingFolderFromPublic(
                uid, folderId, folderPath, ActionInfo.internal(ActionSource.lentaApi())), Option.of(uid));
    }

    @Action
    public CreatedOrUpdatedBlockInfo saveFileFromPublic(@SpecialParam HttpServletRequestX req) {
        return processLogLineAndReturnCreatedBlockId(req);
    }

    @Action
    public CreatedOrUpdatedBlockInfo processLogLineAndReturnCreatedBlockId(@SpecialParam HttpServletRequestX req) {
        String logLine = new String(req.getInputStreamX().readBytes());
        Option<DataApiUserId> uidO = TskvUtils.extractTskv(logLine).getO("uid").map(DataApiUserId::parse);

        try {
            ListF<LentaBlockEvent> blocks = ActionSourceHolder.invokeWithResult(ActionSource.lentaApi(),
                    () -> lentaEventsLogListener.processLogLineAndStoreResult(logLine));

            return resolveCreatedBlockInfo(blocks, uidO);
        } catch (Throwable t) {
            ExceptionUtils.throwIfUnrecoverable(t);
            logger.error("Failed to resolve created block id from logline: {}", logLine);
            throw t;
        }
    }

    @Action
    public CoolLentaBlockInfo getCoolLentaBlockInfo(
            @RequestParam("uid") DataApiUserId uid,
            @RequestParam("blockId") String blockId,
            @RequestParam("lang") String lang)
    {
        ActionInfo actionInfo = ActionInfo.internal(ActionSource.lentaApi());
        LentaBlockRecord block = lentaManager.findBlock(uid, blockId, actionInfo)
                .getOrThrow(() -> new NotFoundException("Block not found: " + blockId));

        MapF<String, DataField> data = block.toData();

        Language language = Language.R.valueOfO(lang).getOrElse(Language.RUSSIAN);

        return new CoolLentaBlockInfo(
                block.id,
                block.type.value(),
                PhotoSelectionFields.SUBTYPE.get(data),

                PhotoSelectionFields.BEST_RESOURCE_ID.get(data),
                PhotoSelectionFields.RESOURCE_IDS.get(data),

                PhotoSelectionFields.PHOTOSLICE_DATE.getO(data),

                getOrDefault(PhotoSelectionFields.TITLE.get(data), language),
                getOrDefault(PhotoSelectionFields.COVER_TITLE.get(data), language),
                getOrDefault(PhotoSelectionFields.COVER_SUBTITLE.get(data), language),
                getOrDefault(PhotoSelectionFields.BUTTON_TEXT.get(data), language)
        );
    }

    @Action
    public Object generateBlock(@RequestParam("uid") DataApiUserId uid, @RequestParam("count") int count) {
        FullJobId jobId = bazingaTaskManager.schedule(new GenerateBlockTask(uid, count, Instant.now()));
        for (int i = 0; i < generateRandomBlockRetryCount.get(); i++) {
            try {
                Thread.sleep(500);
            } catch (InterruptedException e) {
                logger.warn("The generate block thread is interrupted uid={} count={}", uid, count);
                return new ErrorPojo("The processing has been interrupted");
            }
            Option<OnetimeJob> jobO = bazingaTaskManager.getOnetimeJob(jobId);
            if (!jobO.isPresent()) {
                return new ErrorPojo("The task has not been scheduled");
            }
            JobInfoValue info = jobO.get().getValue();
            if (info.getStatus() == JobStatus.COMPLETED) {
                return new OkPojo();
            }
            if (info.getStatus() == JobStatus.FAILED) {
                return new ErrorPojo("The generation has been failed: " + info.getExceptionMessage().getOrElse(""));
            }
        }
        return new ErrorPojo("The waiting time for processing is exceeded");
    }

    private String getOrDefault(I18nValue<String> i18nValue, Language language) {
        return i18nValue.translations.getOrElse(language, i18nValue.translations.getTs(Language.RUSSIAN));
    }

    private final static ListF<LentaEventType> CREATE_OR_UPDATE = Cf.list(
            LentaEventType.BLOCK_CREATE,
            LentaEventType.BLOCK_UPDATE_AND_UP,
            LentaEventType.BLOCK_UPDATE,
            LentaEventType.BLOCK_CREATE_PINNED,
            LentaEventType.BLOCK_TASK_SCHEDULED,
            LentaEventType.BLOCK_TASK_MERGED);

    private CreatedOrUpdatedBlockInfo resolveCreatedBlockInfo(
            ListF<LentaBlockEvent> events, Option<DataApiUserId> uidO)
    {
        String blockId;
        try {
            blockId = events
                    .filter(event -> uidO.isSome(event.uid))
                    .filter(event -> CREATE_OR_UPDATE.containsTs(event.type))
                    .filterMap(event -> event.blockId).first();
        } catch (IndexOutOfBoundsException exception) {
            String errorMessage = "Failed to resolve created blockId for uid: " + uidO
                    + ". Found blocks: " + events.map(e -> Tuple3.tuple(e.uid, e.type, e.blockId));
            logger.error(errorMessage);
            throw new BadRequestException(errorMessage);
        }

        return new CreatedOrUpdatedBlockInfo(blockId);
    }

    @ActionResultPojo
    @BenderBindAllFields
    public static class CreatedOrUpdatedBlockInfo {
        public final String blockId;

        public CreatedOrUpdatedBlockInfo(String blockId) {
            this.blockId = blockId;
        }
    }

    @Data
    @ActionResultPojo
    @BenderBindAllFields
    public static class CoolLentaBlockInfo {
        public final String id;
        public final String type;
        public final String subtype;

        public final String bestResourceId;
        public final ListF<String> resourceIds;

        public final Option<Instant> photosliceDate;

        public final String title;
        public final String coverTitle;
        public final String coverSubtitle;
        public final String buttonText;
    }
}
