package ru.yandex.chemodan.app.lentaloader.worker.tasks;

import org.apache.commons.lang3.mutable.MutableInt;

import ru.yandex.bolts.collection.IteratorF;
import ru.yandex.chemodan.app.dataapi.api.data.filter.condition.DatabaseCondition;
import ru.yandex.chemodan.app.dataapi.api.db.Database;
import ru.yandex.chemodan.app.dataapi.core.datasources.disk.DiskDataSource;
import ru.yandex.chemodan.app.lentaloader.DynamicVars;
import ru.yandex.chemodan.app.lentaloader.lenta.LentaManager;
import ru.yandex.chemodan.app.lentaloader.lenta.LentaManagerImpl;
import ru.yandex.chemodan.app.lentaloader.log.ActionReason;
import ru.yandex.chemodan.app.lentaloader.worker.LentaCronTaskSupport;
import ru.yandex.chemodan.app.lentaloader.worker.LentaTaskQueueName;
import ru.yandex.commune.bazinga.scheduler.ExecutionContext;
import ru.yandex.commune.bazinga.scheduler.TaskQueueName;
import ru.yandex.commune.bazinga.scheduler.schedule.Schedule;
import ru.yandex.commune.bazinga.scheduler.schedule.ScheduleCron;
import ru.yandex.commune.bazinga.scheduler.schedule.ScheduleWithRetry;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.db.masterSlave.MasterSlaveContextHolder;
import ru.yandex.misc.db.masterSlave.MasterSlavePolicy;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.thread.ThreadUtils;
import ru.yandex.misc.time.MoscowTime;

/**
 * @author dbrylev
 */
public class CleanupLentaBlocksCronTask extends LentaCronTaskSupport {

    private static final Logger logger = LoggerFactory.getLogger(CleanupLentaBlocksCronTask.class);

    private final DiskDataSource nativeDataApiManager;
    private final LentaManager lentaManager;

    public CleanupLentaBlocksCronTask(
            DiskDataSource nativeDataApiManager, LentaManager lentaManager)
    {
        this.nativeDataApiManager = nativeDataApiManager;
        this.lentaManager = lentaManager;
    }

    @Override
    public Schedule cronExpression() {
        return new ScheduleWithRetry(new ScheduleCron("0 1 * * *", MoscowTime.TZ), 3);
    }

    @Override
    public void doExecute(ExecutionContext executionContext) throws Exception {
        DynamicVars.CleanupBlocksConfiguration configuration = DynamicVars.cleanupBlocksTaskConf.get();

        IteratorF<Database> databases = nativeDataApiManager.getDatabases(
                LentaManagerImpl.DB_REF, DatabaseCondition.recordsCount().gt(configuration.blocksSearchLimit));

        MutableInt users = new MutableInt(), blocks = new MutableInt(), errors = new MutableInt();

        MasterSlaveContextHolder.PolicyHandle handle = MasterSlaveContextHolder.push(MasterSlavePolicy.R_SM);
        try {
            databases.forEachRemaining(db -> MasterSlaveContextHolder.withPolicy(MasterSlavePolicy.RW_M, () -> {

                int limit = (int) (db.meta.recordsCount - configuration.blocksEnough);

                if (limit > 0) {
                    try {
                        blocks.add(lentaManager.deleteEldestBlocks(db.uid, limit,
                                getActionInfo().withReason(ActionReason.LENTA_SIZE_LIMIT_EXCEEDED)));
                        users.increment();

                    } catch (Exception e) {
                        ThreadUtils.rethrowIfControl(e);
                        logger.warn("Failed to delete eldest blocks for user {}: {}", db.uid, e);

                        errors.increment();
                    }
                }
            }));
        } finally {
            handle.popSafely();
            executionContext.setExecutionInfo(new Result(users.getValue(), blocks.getValue(), errors.getValue()));
        }
    }

    @Override
    public TaskQueueName queueName() {
        return LentaTaskQueueName.LENTA_CRON;
    }

    @BenderBindAllFields
    public static class Result {
        public final int users;
        public final int blocks;
        public final int errors;

        public Result(int users, int blocks, int errors) {
            this.users = users;
            this.blocks = blocks;
            this.errors = errors;
        }
    }
}
