package ru.yandex.chemodan.app.lentaloader.worker.tasks;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.lentaloader.blocks.ContentBlockFields;
import ru.yandex.chemodan.app.lentaloader.blocks.ContentBlockManager;
import ru.yandex.chemodan.app.lentaloader.blocks.FolderCreationBlockAction;
import ru.yandex.chemodan.app.lentaloader.blocks.FolderCreationBlockFields;
import ru.yandex.chemodan.app.lentaloader.blocks.FolderCreationBlockManager;
import ru.yandex.chemodan.app.lentaloader.lenta.LentaManager;
import ru.yandex.chemodan.app.lentaloader.lenta.LentaRecordType;
import ru.yandex.chemodan.app.lentaloader.lenta.update.LentaBlockBaseData;
import ru.yandex.chemodan.app.lentaloader.lenta.update.UpdateOrDeleteHandler;
import ru.yandex.chemodan.app.lentaloader.log.ActionReason;
import ru.yandex.chemodan.app.lentaloader.log.DataOrRefusal;
import ru.yandex.chemodan.app.lentaloader.log.ReasonedAction;
import ru.yandex.chemodan.mpfs.MpfsResourceId;
import ru.yandex.chemodan.util.exception.PermanentHttpFailureException;
import ru.yandex.commune.bazinga.scheduler.ExecutionContext;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author dbrylev
 */
public class DeleteOrUpdateLentaBlockTask extends ModifyBlockTaskSupport {

    private static final Logger logger = LoggerFactory.getLogger(DeleteOrUpdateLentaBlockTask.class);

    private ContentBlockManager contentBlockManager;
    private FolderCreationBlockManager folderCreationBlockManager;

    public DeleteOrUpdateLentaBlockTask(DataApiUserId uid, String blockId, ReasonedAction actionInfo) {
        super(uid, blockId, Option.of(actionInfo.reason), Option.of(actionInfo.actionInfo));
    }

    public DeleteOrUpdateLentaBlockTask(
            LentaManager lentaManager,
            ContentBlockManager contentBlockManager,
            FolderCreationBlockManager folderCreationBlockManager)
    {
        super(lentaManager);
        this.contentBlockManager = contentBlockManager;
        this.folderCreationBlockManager = folderCreationBlockManager;
    }

    @Override
    protected void execute(DataApiUserId uid, String blockId, Option<ActionReason> reason, ExecutionContext context) {
        lentaManager.updateOrDeleteBlock(uid, blockId, record -> {
            try {
                if (record.type == LentaRecordType.CONTENT_BLOCK) {
                    DataOrRefusal<Integer> filesCount = contentBlockManager.getFilesCount(uid, record);

                    if (!filesCount.isRefusal()) {
                        return UpdateOrDeleteHandler.update(
                                new LentaBlockBaseData(record.type, record.groupKey),
                                record.specific.plus(Cf.toMap(Cf.list(
                                        ContentBlockFields.FILES_COUNT.toData(filesCount.getData())))));
                    } else {
                        return UpdateOrDeleteHandler.delete(reason.getOrElse(filesCount.getRefusalReason()));
                    }

                } else if (record.type == LentaRecordType.FOLDER_CREATION) {
                    if (FolderCreationBlockFields.ACTION.getO(record).isSome(FolderCreationBlockAction.SAVING_PUBLIC)) {
                        return UpdateOrDeleteHandler.ignore();
                    }

                    MpfsResourceId folderId = MpfsResourceId.parse(FolderCreationBlockFields.FOLDER_ID.get(record));

                    if (!folderCreationBlockManager.folderExists(uid, folderId)) {
                        return UpdateOrDeleteHandler.delete(reason.getOrElse(ActionReason.UNSPECIFIED));
                    } else {
                        return UpdateOrDeleteHandler.ignore();
                    }

                } else {
                    return UpdateOrDeleteHandler.delete(reason.getOrElse(ActionReason.UNSPECIFIED));
                }
            } catch (PermanentHttpFailureException e) {
                if (e.getStatusCode().filter(code -> code == 403).isPresent()) {
                    logger.warn("The 403 status code has been got from MPFS", e);
                    return UpdateOrDeleteHandler.ignore();
                }
                throw e;
            }
        }, getActionInfo());
    }
}
