package ru.yandex.chemodan.app.lentaloader.worker.tasks;

import javax.annotation.Nullable;

import org.joda.time.Duration;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.lentaloader.DynamicVars;
import ru.yandex.chemodan.app.lentaloader.lenta.FieldPredicate;
import ru.yandex.chemodan.app.lentaloader.lenta.LentaBlockRecord;
import ru.yandex.chemodan.app.lentaloader.lenta.LentaManager;
import ru.yandex.chemodan.app.lentaloader.lenta.LentaRecordType;
import ru.yandex.chemodan.app.lentaloader.log.ActionInfo;
import ru.yandex.chemodan.app.lentaloader.log.ReasonedAction;
import ru.yandex.chemodan.app.lentaloader.worker.LentaOnetimeTaskSupport;
import ru.yandex.chemodan.http.YandexCloudRequestIdHolder;
import ru.yandex.commune.bazinga.scheduler.ActiveUidBehavior;
import ru.yandex.commune.bazinga.scheduler.ActiveUidDropType;
import ru.yandex.commune.bazinga.scheduler.ActiveUidDuplicateBehavior;
import ru.yandex.commune.bazinga.scheduler.ActiveUniqueIdentifierConverter;
import ru.yandex.commune.bazinga.scheduler.ExecutionContext;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.bender.annotation.BenderFlatten;
import ru.yandex.misc.support.tl.ThreadLocalHandle;

/**
 * @author dbrylev
 */
public class FindAndDeleteOrUpdateLentaBlocksTask
        extends LentaOnetimeTaskSupport<FindAndDeleteOrUpdateLentaBlocksTask.Parameters>
{
    private final DynamicProperty<Integer> maxUpdateBlocks = new DynamicProperty<>("FindAndDeleteOrUpdateLentaBlocksTask-max-update-blocks", 5);
    private LentaManager lentaManager;

    public FindAndDeleteOrUpdateLentaBlocksTask(
            DataApiUserId uid, LentaRecordType blockType,
            FieldPredicate filter, ReasonedAction actionInfo)
    {
        super(new Parameters(
                new ActiveUidParams(uid, blockType, filter),
                new ActionInfoParameters(YandexCloudRequestIdHolder.getO(), actionInfo)));
    }

    public FindAndDeleteOrUpdateLentaBlocksTask(LentaManager lentaManager) {
        super(Parameters.class);
        this.lentaManager = lentaManager;
    }

    @Override
    protected void doExecute(Parameters params, ExecutionContext context) {
        Option<ThreadLocalHandle> handler = params.info.ycrid.map(YandexCloudRequestIdHolder::setAndPushToNdc);
        try {
            ActionInfo actionInfo = getActionInfo();
            ReasonedAction reasonedAction = actionInfo.withReason(params.info.reason.get());

            ListF<LentaBlockRecord> blocks = lentaManager.findBlocks(
                    params.active.uid, params.active.blockType, params.active.filter, actionInfo);

            blocks = blocks
                    .sortedByDesc(block -> block.mTime)
                    .take(maxUpdateBlocks.get());

            blocks.zipWithIndex().forEach((block, idx) -> {
                Duration delay = DynamicVars.blocksFindAndModifyGap.get().multipliedBy(idx);

                lentaManager.deleteOrUpdateBlockDelayed(
                        params.active.uid, block.getIdAndType(), delay, reasonedAction);
            });

        } finally {
            handler.forEach(ThreadLocalHandle::popSafely);
        }
    }

    @Override
    protected ActionInfo getActionInfo() {
        return getParametersTyped().info.getActionInfo(super.getActionInfo());
    }

    @Override
    public ActiveUidBehavior activeUidBehavior() {
        return new ActiveUidBehavior(ActiveUidDropType.WHEN_RUNNING, ActiveUidDuplicateBehavior.DO_NOTHING);
    }

    @Override
    public Duration timeout() {
        return Duration.standardMinutes(1);
    }

    @Nullable
    @Override
    public Class<? extends ActiveUniqueIdentifierConverter<?, ?>> getActiveUidConverter() {
        return FindAndDeleteOrUpdateLentaBlocksTask.ActiveUidParams.Converter.class;
    }

    @BenderBindAllFields
    public static class ActiveUidParams {
        public final DataApiUserId uid;

        public final LentaRecordType blockType;
        public final FieldPredicate filter;

        public ActiveUidParams(
                DataApiUserId uid, LentaRecordType blockType, FieldPredicate filter)
        {
            this.uid = uid;
            this.blockType = blockType;
            this.filter = filter;
        }

        public static class Converter implements ActiveUniqueIdentifierConverter<Parameters, ActiveUidParams> {
            @Override
            public Class<ActiveUidParams> getActiveUniqueIdentifierClass() {
                return FindAndDeleteOrUpdateLentaBlocksTask.ActiveUidParams.class;
            }

            @Override
            public ActiveUidParams convert(Parameters parameters) {
                return parameters.active;
            }
        }
    }

    @BenderBindAllFields
    public static class Parameters {
        @BenderFlatten
        public final ActiveUidParams active;
        @BenderFlatten
        public final ActionInfoParameters info;

        public Parameters(ActiveUidParams active, ActionInfoParameters info) {
            this.active = active;
            this.info = info;
        }
    }
}
