package ru.yandex.chemodan.app.logreader.accesslog;

import org.junit.Before;
import org.junit.Test;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;

import ru.yandex.chemodan.app.countersapi.dao.PublicPageViewsDao;

import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.verifyZeroInteractions;

/**
 * @author nshmakov
 */
public class MpfsPublicInfoLogListenerTest {

    private MpfsPublicInfoLogListener sut = new MpfsPublicInfoLogListener();

    @Mock
    private PublicPageViewsDao publicPageViewsDaoMock;

    @Before
    public void init() {
        MockitoAnnotations.initMocks(this);
        sut.setPublicPageViewsDao(publicPageViewsDaoMock);
    }

    @Test
    public void shouldRecognizePublicInfoAccess() {
        String logRecord = "tskv\ttskv_format=ydisk-mpfs-access-log\thost=mpfs1g.disk.yandex.net\tname=mpfs.access\tappname=disk"
                + "\tunixtime=1485167415\ttimestamp=2017-01-23 13:30:15,926\ttimezone=+0300"
                + "\tycrid=web-d1386663fa476e868541b1d125a71381-ufo02d\trequest_id=509383_100040\tpid=509383"
                + "\tmodule=__init__\theaders=\"Content-Length: \"\tmethod=GET\tuid=None"
                + "\turi=/json/public_info?meta=blocked%2Cblockings%2Ccomment_ids%2Cdownload_counter%2Cdrweb%2Cfrom%2Cmediatype%2Cmimetype%2Coffice_online_url%2Cpublic%2Cscreenshot%2Cshort_url%2Csize%2Csizes%2Cspeed_limited%2Cuid%2Cviews_counter%2Cfile_id&private_hash=UVjMrX7%2F1AvSosK6pjue2uPS%2BN0CdkafW6zJmGSno6U%3D&increment_views=1&tld=sk"
                + "\tproto=HTTP/1.1\tstatus=200\trequest_time=0.087";

        String hashUrlDecoded = "UVjMrX7/1AvSosK6pjue2uPS+N0CdkafW6zJmGSno6U=";

        sut.processLogLine(logRecord);

        verify(publicPageViewsDaoMock).increaseCounter(hashUrlDecoded, 1);
    }

    @Test
    public void shouldIgnoreNonPublicInfoAccess() {
        String logRecord = "tskv\ttskv_format=ydisk-mpfs-access-log\thost=mpfs1g.disk.yandex.net\tname=mpfs.access\tappname=disk"
                + "\tunixtime=1485167407\ttimestamp=2017-01-23 13:30:07,758\ttimezone=+0300"
                + "\tycrid=web-3a4c145b4709efa0d6876bb4857379d5-ufo02d\trequest_id=509282_100000\tpid=509282"
                + "\tmodule=__init__\theaders=\"Content-Length: \"\tmethod=GET\tuid=27835513"
                + "\turi=/json/status?oid=300007f6353baa14532d3b6c766c6e33c4f8dd7fd8bf984738a08e9e2196530e&meta=blocked%2Cdownload_counter%2Cviews_counter%2Cdrweb%2Cfolder_type%2Cfrom%2Cfullname%2Cgroup%2Cmediatype%2Cmid%2Cmimetype%2Coriginal_parent_id%2Cpublic%2Csearch_scope%2Cshared_rights%2Cshort_url%2Csize%2Csizes%2Csubject%2Cto%2Cwith_shared%2Clast_import_time%2Cappend_time%2Cphotoslice_time%2Cfile_id%2Coffice_online_url%2Ccustom_properties%2Ccomment_ids%2Cviews_counter%2Ctotal_results_count&uid=27835513&tld=ua&decode_custom_properties=1"
                + "\tproto=HTTP/1.1\tstatus=200\trequest_time=0.192";
        sut.processLogLine(logRecord);

        verifyZeroInteractions(publicPageViewsDaoMock);
    }

    @Test
    public void shouldIncrementCounterWithOnlyOneParameterInPublicInfo() {
        String logRecord = "tskv\ttskv_format=ydisk-mpfs-access-log\thost=mpfs1g.disk.yandex.net\tname=mpfs.access\tappname=disk"
                + "\tunixtime=1485167415\ttimestamp=2017-01-23 13:30:15,926\ttimezone=+0300"
                + "\tycrid=web-d1386663fa476e868541b1d125a71381-ufo02d\trequest_id=509383_100040\tpid=509383"
                + "\tmodule=__init__\theaders=\"Content-Length: \"\tmethod=GET\tuid=None"
                + "\turi=/json/public_info?private_hash=UVjMrX7%2F1AvSosK6pjue2uPS%2BN0CdkafW6zJmGSno6U%3D&increment_views=1"
                + "\tproto=HTTP/1.1\tstatus=200\trequest_time=0.087";

        String hashUrlDecoded = "UVjMrX7/1AvSosK6pjue2uPS+N0CdkafW6zJmGSno6U=";

        sut.processLogLine(logRecord);

        verify(publicPageViewsDaoMock).increaseCounter(hashUrlDecoded, 1);
    }

    @Test
    public void shouldIncrementCounterWhenHashIsNotEncodedInPublicInfo() {
        String logRecord = "tskv\ttskv_format=ydisk-mpfs-access-log\thost=mpfs1g.disk.yandex.net\tname=mpfs.access\tappname=disk"
                + "\tunixtime=1485167415\ttimestamp=2017-01-23 13:30:15,926\ttimezone=+0300"
                + "\tycrid=web-d1386663fa476e868541b1d125a71381-ufo02d\trequest_id=509383_100040\tpid=509383"
                + "\tmodule=__init__\theaders=\"Content-Length: \"\tmethod=GET\tuid=None"
                + "\turi=/json/public_info?private_hash=UVjMrX7/1AvSosK6pjue2uPSN0CdkafW6zJmGSno6U=&increment_views=1"
                + "\tproto=HTTP/1.1\tstatus=200\trequest_time=0.087";

        String hashUrlDecoded = "UVjMrX7/1AvSosK6pjue2uPSN0CdkafW6zJmGSno6U=";

        sut.processLogLine(logRecord);

        verify(publicPageViewsDaoMock).increaseCounter(hashUrlDecoded, 1);
    }

    @Test
    public void shouldIgnoreNon200PublicInfoRequests() {
        String logRecord = "tskv\ttskv_format=ydisk-mpfs-access-log\thost=mpfs1g.disk.yandex.net\tname=mpfs.access\tappname=disk"
                + "\tunixtime=1485167416\ttimestamp=2017-01-23 13:30:16,986\ttimezone=+0300"
                + "\tycrid=web-cba2ec951cee461191b11cc726a99c78-ufo07d\trequest_id=509238_100040\tpid=509238"
                + "\tmodule=__init__\theaders=\"Content-Length: \"\tmethod=GET\tuid=None"
                + "\turi=/json/public_info?meta=blocked%2Cblockings%2Ccomment_ids%2Cdownload_counter%2Cdrweb%2Cfrom%2Cmediatype%2Cmimetype%2Coffice_online_url%2Cpublic%2Cscreenshot%2Cshort_url%2Csize%2Csizes%2Cspeed_limited%2Cuid%2Cviews_counter%2Cfile_id&private_hash=VJYwcSG4c7qgNWX%2BFtA5DIhBGyoBGIO%2BVav%2B%2BdnjhPQ%3D&increment_views=1&tld=sk"
                + "\tproto=HTTP/1.1\tstatus=404\trequest_time=0.035";

        sut.processLogLine(logRecord);

        verifyZeroInteractions(publicPageViewsDaoMock);
    }

    @Test
    public void shouldIgnorePublicFolderFiles() {
        String logRecord = "tskv\ttskv_format=ydisk-mpfs-access-log\thost=mpfs1g.disk.yandex.net\tname=mpfs.access\tappname=disk"
                + "\tunixtime=1485167434\ttimestamp=2017-01-23 13:30:34,822\ttimezone=+0300"
                + "\tycrid=web-1b4d2f19a4271605bddb788e1e99544f-ufo08d\trequest_id=509375_100121\tpid=509375"
                + "\tmodule=__init__\theaders=\"Content-Length: \"\tmethod=GET\tuid=None"
                + "\turi=/json/public_info?meta=blocked%2Cblockings%2Ccomment_ids%2Cdownload_counter%2Cdrweb%2Cfrom%2Cmediatype%2Cmimetype%2Coffice_online_url%2Cpublic%2Cscreenshot%2Cshort_url%2Csize%2Csizes%2Cspeed_limited%2Cuid%2Cviews_counter%2Cfile_id&private_hash=nEJyPRZ2D%2FSS7HBFt3N11L0CVsMS78hBGnse%2BI%2B%2BGgo%3D%3A%2FDSC_8997.JPG&increment_views=0&tld=sk"
                + "\tproto=HTTP/1.1\tstatus=200\trequest_time=0.075";

        sut.processLogLine(logRecord);

        verifyZeroInteractions(publicPageViewsDaoMock);
    }

    @Test
    public void shouldIgnoreBrokenLogLine() {
        String logRecord = "2014-07-18 18:35:48,135 [16755] ios-RSEh_8L_676-1-webdav3h 16755_13534 __init__ POST /json/setting_set?uid=206729751&namespace=photostream&key=f572bc3eea2f19559ad3a094a0e940a8&value=<><autoupload xmlns=\"\" type=\"all\">always</autoupload></> HTTP/1.1 200 0.012";

        sut.processLogLine(logRecord);

        verifyZeroInteractions(publicPageViewsDaoMock);
    }

    @Test
    public void shouldRecognizePublicAlbumAccess() {
        String logRecord = "tskv\ttskv_format=ydisk-mpfs-access-log\thost=mpfs1g.disk.yandex.net\tname=mpfs.access\tappname=disk"
                + "\tunixtime=1485168154\ttimestamp=2017-01-23 13:42:34,809\ttimezone=+0300"
                + "\tycrid=web-6c7dc2a145897d89cc9b43c52d92ed39-ufo05d\trequest_id=509293_103617\tpid=509293"
                + "\tmodule=__init__\theaders=\"Content-Length: \"method=GET\tuid=None\turi=/json/public_album_get?public_key=aKtoMkWXFvdr2v9SOztvechicSlS5c7oCbspQFaPrHRaOUopaSgKOmsnvptQqawmq%2FJ6bpmRyOJonT3VoXnDag%3D%3D&meta=size%2Csizes%2Cmediatype%2Cfile_id&amount=0&increment_views=1&tld=sk"
                + "\tproto=HTTP/1.1\tstatus=200\trequest_time=0.067";

        String publicKey = "aKtoMkWXFvdr2v9SOztvechicSlS5c7oCbspQFaPrHRaOUopaSgKOmsnvptQqawmq/J6bpmRyOJonT3VoXnDag==";

        sut.processLogLine(logRecord);

        verify(publicPageViewsDaoMock).increaseCounter(publicKey, 1);
    }
}
