package ru.yandex.chemodan.app.logreader.event_history;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.eventlog.events.StandardMediaType;
import ru.yandex.chemodan.mpfs.MpfsResourceId;
import ru.yandex.chemodan.uploader.docviewer.DocviewerClient;
import ru.yandex.chemodan.util.tskv.TskvUtils;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.commune.salr.logreader.LogListener;
import ru.yandex.inside.passport.PassportUidOrZero;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author akirakozov
 */
public class MpfsEventHistoryLogListener implements LogListener {
    private static final Logger logger = LoggerFactory.getLogger(MpfsEventHistoryLogListener.class);

    public static final ListF<String> REQUIRED_FIELDS_FOR_SET_PUBLIC_EVENT = Cf.list(
            "tgt_rawaddress", "public_key", "resource_media_type");

    private DynamicProperty<ListF<String>> extensionsToWarnUpDocviewerCache =
        new DynamicProperty<>("log-reader-extensions-to-warn-up-dv-cache", Cf.list("ppt", "pptx"));

    private final DocviewerClient docviewerClient;

    public MpfsEventHistoryLogListener(DocviewerClient docviewerClient) {
        this.docviewerClient = docviewerClient;
    }

    @Override
    public void processLogLine(String line) {
        Option<SetPublicEventInfo> eventInfo = parseLine(line);
        if (eventInfo.map(this::needToWarnUpDvCache).getOrElse(false)) {
            docviewerClient.startConvertToHtml(
                    PassportUidOrZero.zero(), "ya-disk-public://" + eventInfo.get().publicKey, Option.empty(), true);
        }
    }

    boolean needToWarnUpDvCache(SetPublicEventInfo info) {
        if (info.mediaType == StandardMediaType.DOCUMENT) {
            String fileId = info.resourceId.fileId;
            String path = fileId.startsWith("/attach") ?
                    StringUtils.substringBeforeLast(fileId, "_") : fileId;
            String extension = StringUtils.substringAfterLast(path, ".");
            return extensionsToWarnUpDocviewerCache.get().containsTs(extension);
        } else {
            return false;
        }
    }

    Option<SetPublicEventInfo> parseLine(String line) {
        MapF<String, String> m = TskvUtils.extractTskv(line);

        if (!"fs-set-public".equals(m.getTs("event_type")) || !"file".equals(m.getTs("resource_type"))) {
            // not supported event_type
            return Option.empty();
        }

        ListF<String> notFoundRequiredFields = REQUIRED_FIELDS_FOR_SET_PUBLIC_EVENT
                .zipWith(m::getO)
                .filterBy2(f -> !f.isPresent())
                .get1();

        if (!notFoundRequiredFields.isEmpty()) {
            logger.debug("Not found required fields: " + notFoundRequiredFields);
            return Option.empty();
        }


        return Option.of(new SetPublicEventInfo(
                MpfsResourceId.parse(m.getTs("tgt_rawaddress")), m.getTs("public_key"),
                StandardMediaType.R.fromValue(m.getTs("resource_media_type"))));
    }

    public static class SetPublicEventInfo {
        public final MpfsResourceId resourceId;
        public final String publicKey;
        public final StandardMediaType mediaType;

        public SetPublicEventInfo(
                MpfsResourceId resourceId,
                String publicKey,
                StandardMediaType mediaType)
        {
            this.resourceId = resourceId;
            this.publicKey = publicKey;
            this.mediaType = mediaType;
        }
    }
}
