package ru.yandex.chemodan.app.logreader.event_history;

import org.junit.Before;
import org.junit.Test;
import org.mockito.Mock;
import org.mockito.Mockito;
import org.mockito.MockitoAnnotations;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.eventlog.events.StandardMediaType;
import ru.yandex.chemodan.mpfs.MpfsResourceId;
import ru.yandex.chemodan.uploader.docviewer.DocviewerClient;
import ru.yandex.inside.passport.PassportUidOrZero;
import ru.yandex.misc.test.Assert;

/**
 * @author akirakozov
 */
public class MpfsEventHistoryLogListenerTest {
    private static final String CORRECT_DISK_FILE_LOG_LINE =
            "tskv\tunixtime=1502176926\thostname=mpfs1g\tevent_type=fs-set-public\t" +
            "req_id=web-7d70b4032ba50606f4f91eb0aab964ac-ufo06f\ttskv_format=ydisk-event-history-log\t" +
            "tgt_folder_id=78952817:/disk\t" +
            "tgt_rawaddress=78952817:/disk/file.pptx\t" +
            "uid=78952817\towner_uid=78952817\tshort_url=https://yadi.sk/i/asdfadf\t" +
            "resource_file_id=019365f9ea63e638bfb6cf15ce04e816b2db8407db25fa63336b881e0cad36c0\tsubtype=None\t" +
            "lenta_media_type=document\tpublic_key=1c0zhwJca86VyVf2U5NqnvVjWBA+nsNmo3QDu5DlTKE=\t" +
            "resource_media_type=document\ttype=None\tresource_type=file";

    private static final String CORRECT_MAIL_ATTACH_LOG_LINE =
            "tskv\tunixtime=1502271575\thostname=mpfs1g\tevent_type=fs-set-public\t" +
            "req_id=mpfs-8e0d9e9e80318d656aec0185b2f8d2f3-mpfs17g\ttskv_format=ydisk-event-history-log\t" +
            "tgt_folder_id=392489900:/attach\ttgt_rawaddress=392489900:/attach/coffee.pptx_1502271550.35\t" +
            "uid=392489900\towner_uid=392489900\t" +
            "short_url=https://yadi.sk/mail/?hash=2tsMCamMiXUpwx8ZPa330Q%2F1XI6QRJb8%2FrF6VgaD8b0%3D\t" +
            "resource_file_id=366a965bb4476261808d219e1654045bd6797978d351a3c7ab4513abed3441db\tsubtype=attach\t" +
            "lenta_media_type=document\tpublic_key=2tsMCasf32rafwx8ZPa330Q/1XI6QRJb8/rF6VgaD8b0=\t" +
            "resource_media_type=document\ttype=store\tresource_type=file";

    @Mock
    private DocviewerClient docviewerClient;

    private MpfsEventHistoryLogListener listener;

    @Before
    public void init() {
        MockitoAnnotations.initMocks(this);
        listener =  new MpfsEventHistoryLogListener(docviewerClient);
    }

    @Test
    public void needToWarnUpDvCache() throws Exception {
        Assert.isTrue(listener.needToWarnUpDvCache(
                new MpfsEventHistoryLogListener.SetPublicEventInfo(
                        MpfsResourceId.parse("123:/disk/file.ppt"), "", StandardMediaType.DOCUMENT)));

        Assert.isFalse(listener.needToWarnUpDvCache(
                new MpfsEventHistoryLogListener.SetPublicEventInfo(
                        MpfsResourceId.parse("123:/disk/file.txt"), "", StandardMediaType.DOCUMENT)));

        Assert.isFalse(listener.needToWarnUpDvCache(
                new MpfsEventHistoryLogListener.SetPublicEventInfo(
                        MpfsResourceId.parse("123:/disk/file.ppt"), "", StandardMediaType.UNKNOWN)));
    }

    @Test
    public void parseRegularDiskLogLine() throws Exception {
        MpfsEventHistoryLogListener.SetPublicEventInfo info = listener.parseLine(CORRECT_DISK_FILE_LOG_LINE)
                .getOrThrow("Correct log line wasn't parsed");

        Assert.equals(StandardMediaType.DOCUMENT, info.mediaType);
        Assert.equals("1c0zhwJca86VyVf2U5NqnvVjWBA+nsNmo3QDu5DlTKE=", info.publicKey);
        Assert.equals(MpfsResourceId.parse("78952817:/disk/file.pptx"), info.resourceId);

    }

    @Test
    public void parseMailAttachLogLine() throws Exception {
        MpfsEventHistoryLogListener.SetPublicEventInfo info = listener.parseLine(CORRECT_MAIL_ATTACH_LOG_LINE)
                .getOrThrow("Correct log line wasn't parsed");

        Assert.equals("2tsMCasf32rafwx8ZPa330Q/1XI6QRJb8/rF6VgaD8b0=", info.publicKey);
        Assert.equals(MpfsResourceId.parse("392489900:/attach/coffee.pptx_1502271550.35"), info.resourceId);
    }

    @Test
    public void parseBadLines() throws Exception {
        // no public_key field
        Assert.none(listener.parseLine(
                "tskv\tevent_type=fs-set-public\ttgt_rawaddress=78952817:/disk/file.pptx\t" +
                "resource_media_type=document"));

        // not supported event_type
        Assert.none(listener.parseLine(
                "tskv\tevent_type=some-type\ttgt_rawaddress=78952817:/disk/file.pptx\t" +
                        "resource_media_type=document\tpublic_key=1c0zhwJca86VyVf2U5N"));

    }

    @Test
    public void processLogLineForDiskFile() {
        processLogLine(CORRECT_DISK_FILE_LOG_LINE,
                "ya-disk-public://1c0zhwJca86VyVf2U5NqnvVjWBA+nsNmo3QDu5DlTKE=");
    }

    @Test
    public void processLogLineForMailAttachment() {
        processLogLine(CORRECT_MAIL_ATTACH_LOG_LINE,
                "ya-disk-public://2tsMCasf32rafwx8ZPa330Q/1XI6QRJb8/rF6VgaD8b0=");
    }

    private void processLogLine(String line, String exptectedUrl) {
        listener.processLogLine(line);
        Mockito.verify(docviewerClient, Mockito.times(1))
                .startConvertToHtml(PassportUidOrZero.zero(), exptectedUrl, Option.empty(), true);
    }
}
