package ru.yandex.chemodan.app.logreader.preview;

import java.net.UnknownHostException;

import com.mongodb.MongoClient;
import com.mongodb.MongoClientOptions;
import com.mongodb.ReadPreference;
import com.mongodb.ServerAddress;
import com.mongodb.WriteConcern;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;
import org.springframework.context.annotation.Profile;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.logreader.util.LogReaderFactory;
import ru.yandex.chemodan.bazinga.http.PgHttpBazingaClientContextConfiguration;
import ru.yandex.chemodan.uploader.docviewer.DocviewerClient;
import ru.yandex.commune.admin.web.AdminApp;
import ru.yandex.commune.mongo.MongoUtils;
import ru.yandex.commune.mongo.admin.browser.MongoBrowserAdminPage;
import ru.yandex.commune.salr.logreader.LogReader;
import ru.yandex.inside.admin.conductor.Conductor;
import ru.yandex.inside.admin.conductor.GroupOrHost;
import ru.yandex.misc.io.file.File2;
import ru.yandex.misc.ip.IpPort;
import ru.yandex.misc.lang.tsb.YandexToStringBuilder;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author nshmakov
 */
@Profile("preview")
@Configuration
@Import({
        PgHttpBazingaClientContextConfiguration.class,
})
public class MpfsLogReaderPreviewContextConfiguration {

    private static final Logger log = LoggerFactory.getLogger(MpfsLogReaderPreviewContextConfiguration.class);

    @Bean
    public MpfsPreviewLogListener mpfsPreviewLogListener(
            @Value("${mongo.mpfs.meta.db.name}")
            String metaDbName,
            @Value("${mongo.mpfs.meta.collection.name}")
            String metaCollectionName,
            @Qualifier("mpfsMetaMongoClient")
            MongoClient mongoClient,
            DocviewerClient docviewerClient)
    {
        return new MpfsPreviewLogListener(docviewerClient,
                mongoClient.getDB(metaDbName).getCollection(metaCollectionName));
    }

    @Bean
    public LogReader mpfsPreviewLogReader(
            MpfsPreviewLogListener mpfsPreviewLogListener,
            @Value("${mpfs.preview.log}")
            File2 logFile,
            @Value("${mpfs.preview.log.state.file}")
            File2 logReaderStateFile,
            @Value("${log-reader.max-log-line-data-size}")
            int maxLogLineDataSize)
    {
        return LogReaderFactory.createLogReader(
                logFile, logReaderStateFile, mpfsPreviewLogListener, maxLogLineDataSize);
    }

    @Bean
    public MongoClient mpfsMetaMongoClient(
            @Value("${meta-mongo.hosts:-}")
            GroupOrHost hosts,
            @Value("${meta-mongo.port:-27028}")
            IpPort port,
            @Value("${mongo.connections.per.host:-50}")
            int connectionsPerHost,
            @Value("${mongo.connect.timeout.ms:-5000}")
            int connectTimeoutMs,
            @Value("${mongo.socket.timeout.ms:-5000}")
            int socketTimeoutMs,
            @Value("${mongo.pool.timeout.ms:-5000}")
            int poolWaitTimeoutMs,
            @Value("${mongo.threads.block.multiplier:-20}")
            int threadsAllowedToBlockForConnectionMultiplier,
            @Value("${mongo.w:-2}")
            int w,
            @Value("${mongo.fsync:-false}")
            boolean fsync,
            @Value("${mongo.wtimeout.ms:-2500}")
            int wtimeoutMs,
            Conductor conductor)
            throws UnknownHostException
    {
        ListF<ServerAddress> hostList = conductor.hostsFromString(hosts).map(MongoUtils.consServerAddressF(port));

        MongoClientOptions mongoOptions = MongoClientOptions.builder()
                .connectionsPerHost(connectionsPerHost)
                .connectTimeout(connectTimeoutMs)
                .socketTimeout(socketTimeoutMs)
                .maxWaitTime(poolWaitTimeoutMs)
                .threadsAllowedToBlockForConnectionMultiplier(threadsAllowedToBlockForConnectionMultiplier)
                .writeConcern(new WriteConcern(w, wtimeoutMs, fsync))
                .readPreference(ReadPreference.primaryPreferred())
                .build();

        log.info("Starting mongo client to hosts {} with options: {}", hostList,
                YandexToStringBuilder.reflectionToStringValueObject(mongoOptions));

        return new MongoClient(hostList, mongoOptions);
    }

    @Bean
    public MongoBrowserAdminPage mpfsMongoBrowserAdminPage(
            @Qualifier("mpfsMetaMongoClient") MongoClient mongoClient,
            AdminApp adminApp)
    {
        MongoBrowserAdminPage mongoBrowserAdminPage = new MongoBrowserAdminPage(mongoClient);
        adminApp.registerAdminPage("Mpfs meta mongo browser", Option.of("Db"), mongoBrowserAdminPage);
        return mongoBrowserAdminPage;
    }
}
