package ru.yandex.chemodan.app.logreader.preview;

import com.mongodb.BasicDBObject;
import com.mongodb.DBCollection;
import com.mongodb.DBObject;
import org.apache.commons.lang3.StringUtils;
import org.bson.types.Binary;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.bazinga.http.PgHttpBazingaTaskManager;
import ru.yandex.chemodan.core.worker.tasks.RegenerateDocumentPreview;
import ru.yandex.chemodan.core.worker.tasks.RegenerateImagePreview;
import ru.yandex.chemodan.core.worker.tasks.RegeneratePreviewParameters;
import ru.yandex.chemodan.core.worker.tasks.RegenerateVideoPreview;
import ru.yandex.chemodan.uploader.docviewer.DocviewerClient;
import ru.yandex.chemodan.uploader.preview.PreviewImageManager;
import ru.yandex.chemodan.util.http.ContentTypeUtils;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.commune.salr.logreader.LogListener;
import ru.yandex.inside.mulca.MulcaId;
import ru.yandex.misc.codec.Hex;
import ru.yandex.misc.io.http.UrlUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.random.Random2;

/**
 * @author akirakozov
 */
public class MpfsPreviewLogListener implements LogListener {
    private static final Logger logger = LoggerFactory.getLogger(MpfsPreviewLogListener.class);

    private final DynamicProperty<Boolean> enableScheduling =
            new DynamicProperty<>("log-reader-enable-scheduling", false);

    private final DynamicProperty<Boolean> enableSchedulingFromQueue =
            new DynamicProperty<>("log-reader-enable-from-queue", false);

    private final DynamicProperty<Boolean> enableSchedulingVideo =
            new DynamicProperty<>("log-reader-enable-video", false);

    private final DynamicProperty<Boolean> enableSchedulingDocuments =
            new DynamicProperty<>("log-reader-enable-documents", false);

    private final DynamicProperty<Double> schedulingDocumentsPercent =
            new DynamicProperty<>("log-reader-enable-documents-percent", 0.0);

    private final DynamicProperty<Boolean> enableSchedulingImages =
            new DynamicProperty<>("log-reader-enable-images", false);

    private final DynamicProperty<Long> schedulingDelay =
            new DynamicProperty<>("log-reader-scheduling-delay", 5000L);

    private final DynamicProperty<Long> modificationDelay =
            new DynamicProperty<>("log-reader-modification-delay", Duration.standardMinutes(5).getMillis());

    private final DynamicProperty<Instant> lastUpdateTimestamp =
            new DynamicProperty<>("log-reader-last-update-timestamp", new Instant(0));

    @Autowired
    private PgHttpBazingaTaskManager httpBazingaTaskManager;

    private final DocviewerClient docviewerClient;
    private final DBCollection dbCollection;

    public MpfsPreviewLogListener(DocviewerClient docviewerClient, DBCollection collection) {
        this.docviewerClient = docviewerClient;
        this.dbCollection = collection;
    }

    private Option<DBObject> findInMeta(String hid) {
        return Option.ofNullable(dbCollection.findOne(new BasicDBObject("hid", new Binary((byte) 2, hid.getBytes()))));
    }

    static boolean isQueueRequestId(String requestId) {
        return !requestId.contains("_");
    }

    static String extractMpfsRequestId(String line) {
        return line.split(" ")[4];
    }

    static byte[] decodeHid(String hid) {
        return Hex.decode(hid);
    }

    static RegeneratePreviewParameters extractRegeneratePreviewParameters(String line) {
        MapF<String, String> paramsMap = Cf.hashMap();
        String[] parts = line.split(" ");
        for (int i = 0; i < parts.length; i++) {
            if (parts[i].endsWith(":") && i + 1 < parts.length && !parts[i + 1].endsWith(":")) {
                paramsMap.put(StringUtils.substringBefore(parts[i], ":"), parts[++i]);
            }
        }

        return new RegeneratePreviewParameters(
                paramsMap.getOrThrow("uid"),
                paramsMap.getOrThrow("hid"),
                MulcaId.fromSerializedString(paramsMap.getOrThrow("stid")),
                paramsMap.getOrThrow("mimetype"),
                Long.parseLong(paramsMap.getOrThrow("size")),
                UrlUtils.urlDecode(paramsMap.getOrThrow("path")),
                new Instant(Long.parseLong(paramsMap.getOrThrow("mtime")) * 1000),
                new Instant(Long.parseLong(paramsMap.getOrThrow("utime")) * 1000));
    }

    public void processLogLine(String line) {
        RegeneratePreviewParameters params = extractRegeneratePreviewParameters(line);

        if (params.mtime.isPresent() && new Duration(params.mtime.get(), Instant.now())
                .isShorterThan(new Duration(modificationDelay.get())))
        {
            logger.info("Modified less than {} millis ago. Not schedulling. {}", modificationDelay.get(), params);
            return;
        }

        String correctMimeType = docviewerClient
                .correctMimeTypeByFileNameIfUnknown(params.mimeType, params.getFilenameO());
        if (!params.mimeType.equals(correctMimeType)) {
            logger.info("Corrected mimetype for {}. {} -> {}",
                    params.getFilename(), params.mimeType, correctMimeType);
            params = params.withCorrectedMimeType(correctMimeType);
        }

        if (scheduleAsVideo(params.mimeType)
            || scheduleAsDocument(params.mimeType, params.fileSize)
            || scheduleAsImage(params.mimeType))
        {
            if (!enableSchedulingFromQueue.get() && isQueueRequestId(extractMpfsRequestId(line))) {
                logger.debug("Disabled preview regeneration from queue: {}", params);
            } else {
                schedulePreviewRegeneration(params);
            }
        }
    }

    private boolean scheduleAsVideo(String mimeType) {
        return enableSchedulingVideo.get() && ContentTypeUtils.isVideoContentType(mimeType);
    }

    private boolean scheduleAsImage(String mimeType) {
        return enableSchedulingImages.get() && PreviewImageManager.isSupportedMimeType(mimeType);
    }

    private boolean scheduleAsDocument(String mimeType, long fileSize) {
        return enableSchedulingDocuments.get()
                && (Random2.R.nextInt(100000) / 1000.0 < schedulingDocumentsPercent.get())
                && docviewerClient.isDocumentSupportedByDocviewer(Option.of(mimeType), Option.of(fileSize));
    }

    private void schedulePreviewRegeneration(RegeneratePreviewParameters params) {
        try {
            Option<DBObject> fromMetaO = findInMeta(params.hid);
            if (fromMetaO.isPresent()) {
                DBObject fromMeta = fromMetaO.get();
                logger.debug("Found in meta: {}", fromMeta);
                if (fromMeta.get("failed") instanceof Number &&
                        new Instant(((Number) fromMeta.get("failed")).longValue() * 1000)
                            .isAfter(lastUpdateTimestamp.get()))
                {
                    logger.debug("This file was already failed to be regenerate: {}", params);
                    return;
                }
            } else {
                logger.debug("Not found in meta: {}", params.hid);
            }
        } catch (RuntimeException e) {
            logger.error("Failed search in meta-table: {}", e);
        }
        logger.info("Schedulling preview regeneration: {}", params);
        if (enableScheduling.get()) {
            Instant time = new Instant().plus(schedulingDelay.get());
            if (ContentTypeUtils.isVideoContentType(params.mimeType)) {
                httpBazingaTaskManager.schedule(new RegenerateVideoPreview(params), time);
            } else if (PreviewImageManager.isSupportedMimeType(params.mimeType)) {
                httpBazingaTaskManager.schedule(new RegenerateImagePreview(params), time);
            } else {
                httpBazingaTaskManager.schedule(new RegenerateDocumentPreview(params), time);
            }
        } else {
            logger.info("Scheduling is disabled");
        }
    }
}
