package ru.yandex.chemodan.app.migrator.control;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.atomic.AtomicInteger;

import lombok.AllArgsConstructor;
import lombok.Data;
import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function0V;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.dataapi.core.dao.usermeta.migration.MigrationControl;
import ru.yandex.chemodan.app.migrator.logging.MigrationStageStatus;
import ru.yandex.chemodan.app.migrator.migration.MigrationResult;
import ru.yandex.chemodan.app.migrator.migration.UserMigrationManager;
import ru.yandex.chemodan.app.migrator.tasks.DatasyncMigrationCleanUpTask;
import ru.yandex.chemodan.app.migrator.tasks.DatasyncMigrationLogsCheckTask;
import ru.yandex.chemodan.app.migrator.tasks.DatasyncMigrationTask;
import ru.yandex.chemodan.app.migrator.tasks.DatasyncMigrationYtSupplyTask;
import ru.yandex.commune.bazinga.BazingaTaskManager;
import ru.yandex.commune.bazinga.impl.FullJobId;
import ru.yandex.commune.bazinga.impl.OnetimeJob;
import ru.yandex.commune.bazinga.impl.TaskOverridesManager;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.thread.ThreadUtils;

/**
 * @author yashunsky
 */
@AllArgsConstructor
public class BazingaDatasyncUserMigrationManager {
    private static final Logger logger = LoggerFactory.getLogger(BazingaDatasyncUserMigrationManager.class);

    private final UserMigrationManager userMigrationManager;
    private final MigrationControl migrationControl;
    private final BazingaTaskManager bazingaTaskManager;
    private final TaskOverridesManager taskOverridesManager;
    private final BazingaHelper bazingaHelper;
    private final String sharpeiId;

    public void scheduleMigrationNow(DataApiUserId uid, int destinationShard) {
        int userShard = userMigrationManager.getUserShard(uid, migrationControl.getSharpeiSemaphore());
        if (userShard == destinationShard) {
            logger.info("User {} is already on shard {}. scheduling skipped", uid, userShard);
            return;
        }
        scheduleMigration(uid, userShard, destinationShard, Instant.now());
    }

    public void scheduleMigrationNow(DataApiUserId uid, int sourceShard, int destinationShard) {
        scheduleMigration(uid, sourceShard, destinationShard, Instant.now());
    }

    public void scheduleMigrationDelayed(DataApiUserId uid, int sourceShard, int destinationShard, Duration delay) {
        scheduleMigration(uid, sourceShard, destinationShard, Instant.now().plus(delay));
    }

    public void scheduleMigration(DataApiUserId uid, int sourceShard, int destinationShard, Instant scheduleTime) {
        bazingaTaskManager.schedule(
                new DatasyncMigrationTask(uid, sourceShard, destinationShard, sharpeiId), scheduleTime);
    }

    public MigrationStageStatus migrate(DataApiUserId uid, int sourceShard, int destinationShard) {
        logger.info("Setting lock to migrate user {} from shard {} to shard {}", uid, sourceShard, destinationShard);

        Option<OnetimeJob> migrationLockO = bazingaHelper
                .lockUser(uid, new DatasyncMigrationTask(this, sharpeiId), migrationControl.getLockDelay());

        if (!migrationLockO.isPresent()) {
            logger.info("User {} already locked for migration. New migration aborted.", uid);
            return MigrationStageStatus.ABORTED;
        }

        OnetimeJob migrationLock = migrationLockO.get();

        MigrationResult migrationResult;
        try {
            migrationResult = userMigrationManager.migrateUser(uid, sourceShard, destinationShard, migrationControl);

            if (migrationResult instanceof MigrationResult.Done) {
                String hash = ((MigrationResult.Done) migrationResult).getHash();
                if (migrationControl.isLogsCheckEnabled()) {
                    Duration logsCheckDelay = migrationControl.getLogsCheckDelay();
                    logger.info("User {} logs check will be performed in {}", uid, logsCheckDelay);
                    bazingaTaskManager.schedule(
                            new DatasyncMigrationLogsCheckTask(uid, sourceShard, Instant.now(), hash, sharpeiId),
                            Instant.now().plus(logsCheckDelay));
                } else {
                    Duration cleaningDelay = migrationControl.getCleaningDelay();
                    logger.info("User {} data from source shard ({}) will be cleaned in {}",
                            uid, sourceShard, cleaningDelay);
                    bazingaTaskManager.schedule(
                            new DatasyncMigrationCleanUpTask(uid, sourceShard, hash, sharpeiId),
                            Instant.now().plus(cleaningDelay)
                    );
                }
            }
        } finally {
            ThreadUtils.sleep(Duration.standardSeconds(2)); //make sure ro status is replicated
            if (userMigrationManager.isUserReadOnly(uid, migrationControl.getSharpeiSemaphore())) {
                logger.info("Migration failed to remove user {} readonly. Keep lock task.", uid);
            } else {
                logger.info("Removing migration lock for {}", uid);
                bazingaHelper.deleteOnetimeJob(migrationLock.getId());
            }
        }

        return MigrationStageStatus.fromMigrationResult(migrationResult);
    }

    public MigrationStageStatus removeReadOnly(DataApiUserId uid) {
        userMigrationManager.removeReadOnly(uid, migrationControl.getSharpeiSemaphore());
        return MigrationStageStatus.DONE;
    }

    public MigrationStageStatus checkLogs(DataApiUserId uid, int sourceShard, Instant migrationTime, String hash) {
        //TBD: check user logs
        logger.info("User {} logs check skipped as not yet implemented", uid);

        Duration cleaningDelay = migrationControl.getCleaningDelay();
        logger.info("User {} data from source shard ({}) will be cleaned in {}", uid, sourceShard, cleaningDelay);
        bazingaTaskManager.schedule(
                new DatasyncMigrationCleanUpTask(uid, sourceShard, hash, sharpeiId), Instant.now().plus(cleaningDelay)
        );
        return MigrationStageStatus.SKIPPED;
    }

    public MigrationStageStatus cleanUp(DataApiUserId uid, int shardId, Option<String> hash) {
        MigrationResult migrationResult;

        if (hash.isPresent()) {
            CompletableFuture<FullJobId> fallbackCleaningJobId = new CompletableFuture<>();

            Function0V onHashChecked = () -> {
                Instant fallbackCleaningTime = Instant.now().plus(
                        taskOverridesManager.getWithOverrides(new DatasyncMigrationCleanUpTask(this, "")).timeout());
                logger.info("Schedule fallback cleaning task for user {} on shard {}", uid, shardId);
                fallbackCleaningJobId.complete(bazingaTaskManager.schedule(
                        new DatasyncMigrationCleanUpTask(uid, shardId, sharpeiId), fallbackCleaningTime));
            };

            migrationResult = userMigrationManager.deleteUserData(
                    uid, shardId, hash.get(), onHashChecked, migrationControl.getLimiters());

            if (fallbackCleaningJobId.isDone()) {
                try {
                    logger.info("Delete fallback cleaning task for user {} on shard {}", uid, shardId);
                    bazingaHelper.deleteOnetimeJob(fallbackCleaningJobId.get());
                } catch (InterruptedException | ExecutionException e) {
                    logger.info("Failed to delete fallback cleaning task for user {} on shard {}", uid, shardId);
                }
            }
        } else {
            migrationResult = userMigrationManager.deleteUserData(uid, shardId, migrationControl.getLimiters());
        }

        return MigrationStageStatus.fromMigrationResult(migrationResult);
    }

    public boolean isMigrationPossible(ListF<Integer> shardIds) {
        return migrationControl.isMigrationPossible(shardIds);
    }

    public void addUsersFromYt(String path, long lowerIndex) {
        bazingaHelper.addUsersFromYt(path, lowerIndex, new DatasyncMigrationTask(this, sharpeiId).id(),
                (nextPath, nextIndex) -> new DatasyncMigrationYtSupplyTask(path, nextIndex, sharpeiId),
                MigrationSetup.class, (setup, counter) -> migrateUser(setup, Option.empty(), counter));
    }

    private void migrate(YPath path, long lowerIndex, long upperIndex, Option<Integer> destinationShard) {
        bazingaHelper.processYtTable(path, lowerIndex, upperIndex, MigrationSetup.class,
                (migrationSetup, counter) -> migrateUser(migrationSetup, destinationShard, counter));
    }

    private void migrateUser(MigrationSetup migrationSetup, Option<Integer> destinationShard, AtomicInteger counter) {
        int counterValue = counter.getAndIncrement();
        DataApiUserId uid = migrationSetup.getUid();
        int shard;
        if (destinationShard.isPresent()) {
            shard = destinationShard.get();
        } else if (migrationSetup.getShard().isPresent()) {
            shard = migrationSetup.getShard().get();
        } else {
            logger.warn("[{}] Failed to schedule user {} migration. No destination shard set", counterValue, uid);
            return;
        }
        logger.info("[{}] Schedule user {} migration to shard {}", counterValue, uid, shard);
        scheduleMigrationNow(uid, shard);
    }

    @AllArgsConstructor
    @Data
    @BenderBindAllFields
    private static class MigrationSetup {
        private final DataApiUserId uid;
        private final Option<Integer> shard;
    }
}
