package ru.yandex.chemodan.app.migrator.control;

import java.util.concurrent.ExecutorService;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;

import lombok.AllArgsConstructor;
import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function2;
import ru.yandex.bolts.function.Function2V;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.migrator.tasks.DatasyncMigrationLockTask;
import ru.yandex.chemodan.bazinga.PgOnetimeUtils;
import ru.yandex.chemodan.util.yt.YtHelper;
import ru.yandex.commune.bazinga.BazingaTaskManager;
import ru.yandex.commune.bazinga.impl.FullJobId;
import ru.yandex.commune.bazinga.impl.JobStatus;
import ru.yandex.commune.bazinga.impl.OnetimeJob;
import ru.yandex.commune.bazinga.impl.TaskId;
import ru.yandex.commune.bazinga.impl.TaskOverridesManager;
import ru.yandex.commune.bazinga.impl.storage.BazingaStorage;
import ru.yandex.commune.bazinga.scheduler.ActiveUidDuplicateBehavior;
import ru.yandex.commune.bazinga.scheduler.OnetimeTask;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.tables.YTableEntryTypes;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author yashunsky
 */
@AllArgsConstructor
public class BazingaHelper {
    private static final Logger logger = LoggerFactory.getLogger(BazingaHelper.class);

    private final BazingaStorage bazingaStorage;
    private final BazingaTaskManager bazingaTaskManager;
    private final TaskOverridesManager taskOverridesManager;
    private final YtHelper ytHelper;
    private final String sharpeiId;
    private final int threadsCount;
    private final int expectedReadyTasksCount;
    private final Duration ytSupplyInterval;


    public Option<OnetimeJob> lockUser(DataApiUserId uid, OnetimeTask targetTask, Duration lockDelay) {
        Duration lockOffset = taskOverridesManager.getWithOverrides(targetTask).timeout().plus(lockDelay);

        OnetimeJob migrationLock = PgOnetimeUtils.makeJob(
                new DatasyncMigrationLockTask(uid, sharpeiId), Instant.now().plus(lockOffset));
        FullJobId addedId = bazingaStorage.addOnetimeJob(migrationLock, ActiveUidDuplicateBehavior.DO_NOTHING);

        return Option.when(addedId.getJobId().equals(migrationLock.getJobId()), migrationLock);
    }

    public <T> void addUsersFromYt(
            String path, long lowerIndex, TaskId taskId,
            Function2<String, Long, OnetimeTask> createSupplyTask,
            Class<T> ytRowClass, Function2V<T, AtomicInteger> scheduleF)
    {
        YPath yPath = YPath.simple(path);
        long tableSize = getRowsCount(yPath);
        int readyTasks = bazingaStorage.findOnetimeJobCount(taskId, JobStatus.READY);

        if (readyTasks < expectedReadyTasksCount) {
            int rowsToImport = expectedReadyTasksCount - readyTasks;
            long upperIndex = lowerIndex + rowsToImport;
            logger.info("Going to import {} users from {} (offset {})", rowsToImport, path, lowerIndex);
            processYtTable(yPath, lowerIndex, upperIndex, ytRowClass, scheduleF);
            if (lowerIndex + rowsToImport < tableSize) {
                logger.info("Scheduling import from {} with offset {})", path, upperIndex);
                scheduleNextImportFromYt(path, upperIndex, createSupplyTask);
            } else {
                logger.info("All users from {} scheduled)", path);
            }
        } else {
            logger.info("There is enough ready {} tasks in bazinga {}. Will retry later", taskId, readyTasks);
            scheduleNextImportFromYt(path, lowerIndex, createSupplyTask);
        }
    }

    private void scheduleNextImportFromYt(String path, long offset, Function2<String, Long, OnetimeTask> createSupplyTask) {
        Instant nextTaskInstant = Instant.now().plus(ytSupplyInterval);
        bazingaTaskManager.schedule(createSupplyTask.apply(path, offset), nextTaskInstant);
    }

    private long getRowsCount(YPath path) {
        return ytHelper.getRowCount(path);
    }

    public <T> void processYtTable(
            YPath path, long lowerIndex, long upperIndex, Class<T> ytRowClass, Function2V<T, AtomicInteger> scheduleF)
    {
        YPath pathWithRange = path.withRange(lowerIndex, upperIndex);

        ExecutorService service = new ThreadPoolExecutor(threadsCount, threadsCount,
                0L, TimeUnit.MILLISECONDS, new LinkedBlockingQueue<>(threadsCount),
                new ThreadPoolExecutor.CallerRunsPolicy());

        AtomicInteger counter = new AtomicInteger(0);

        ytHelper.tables().read(pathWithRange, YTableEntryTypes.bender(ytRowClass), setup -> {
            service.submit(() -> scheduleF.apply(setup, counter));
        });

        service.shutdown();
    }

    public void deleteOnetimeJob(FullJobId id) {
        bazingaStorage.deleteOnetimeJob(id);
    }
}
