package ru.yandex.chemodan.app.migrator.control;

import java.util.concurrent.atomic.AtomicInteger;

import lombok.AllArgsConstructor;
import lombok.Data;
import org.joda.time.Duration;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.migrator.sharpei.cleanup.SharpeiCleanupControl;
import ru.yandex.chemodan.app.migrator.sharpei.cleanup.SharpeiCleanupManager;
import ru.yandex.chemodan.app.migrator.tasks.SharpeiCleanupTask;
import ru.yandex.chemodan.app.migrator.tasks.SharpeiCleanupYtSupplyTask;
import ru.yandex.commune.bazinga.BazingaStopExecuteException;
import ru.yandex.commune.bazinga.BazingaTaskManager;
import ru.yandex.commune.bazinga.impl.OnetimeJob;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.random.Random2;
import ru.yandex.misc.thread.ThreadUtils;

/**
 * @author yashunsky
 */
@AllArgsConstructor
public class BazingaSharpeiCleanupManager {
    private static final Logger logger = LoggerFactory.getLogger(BazingaSharpeiCleanupManager.class);

    private final SharpeiCleanupControl control;
    private final SharpeiCleanupManager cleanupManager;
    private final BazingaTaskManager bazingaTaskManager;

    private final BazingaHelper bazingaHelper;

    private final String sharpeiId;

    public boolean removeIfNoData(DataApiUserId uid) {
        logger.info("Setting lock for %s", uid);

        OnetimeJob migrationLock = bazingaHelper
                .lockUser(uid, new SharpeiCleanupTask(this, sharpeiId), control.getLockDelay())
                .getOrThrow(() -> new BazingaStopExecuteException("User already locked"));

        try {
            sleepOnStart();
            return cleanupManager.removeIfNoData(uid, true);
        } finally {
            ThreadUtils.sleep(Duration.standardSeconds(2)); //make sure sharpei data is replicated

            if (cleanupManager.isUserReadOnly(uid)) {
                logger.info("Cleanup failed to remove user {}. Keep lock task.", uid);
            } else {
                logger.info("Removing migration lock for {}", uid);
                bazingaHelper.deleteOnetimeJob(migrationLock.getId());
            }
        }
    }

    public boolean fastRemoveIfNoData(DataApiUserId uid) {
        sleepOnStart();
        return cleanupManager.removeIfNoData(uid, false);
    }

    public void addUsersFromYt(String path, long lowerIndex) {
        bazingaHelper.addUsersFromYt(path, lowerIndex, new SharpeiCleanupTask(this, sharpeiId).id(),
                (nextPath, nextIndex) -> new SharpeiCleanupYtSupplyTask(path, nextIndex, sharpeiId),
                CleanupSetup.class, this::scheduleCleanup);
    }

    private void sleepOnStart() {
        if (control.shuffledStart()) {
            ThreadUtils.sleep(Duration.millis(Random2.R.nextLong(control.getInitialPause().getMillis())));
        }
    }

    private void scheduleCleanup(CleanupSetup setup, AtomicInteger counter) {
        logger.info("[{}] Schedule user {} cleanup", counter.incrementAndGet(), setup.uid);
        bazingaTaskManager.schedule(new SharpeiCleanupTask(setup.uid, setup.ro, sharpeiId));
    }

    @AllArgsConstructor
    @Data
    @BenderBindAllFields
    private static class CleanupSetup {
        private final DataApiUserId uid;
        private final Option<Boolean> ro;
    }
}
