package ru.yandex.chemodan.app.migrator.control;

import java.util.concurrent.Semaphore;

import org.joda.time.Duration;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function0;
import ru.yandex.chemodan.app.dataapi.core.dao.usermeta.migration.MigrationControl;
import ru.yandex.chemodan.app.dataapi.core.ratelimiter.chunk.auto.AutoContextAwareRwRateLimiter;
import ru.yandex.chemodan.ratelimiter.chunk.ChunkRateLimiter;
import ru.yandex.chemodan.ratelimiter.chunk.auto.MetricsConfiguration;
import ru.yandex.chemodan.ratelimiter.chunk.auto.RateLimiterReference;
import ru.yandex.chemodan.ratelimiter.chunk.auto.RateLimitersMetrics;
import ru.yandex.chemodan.ratelimiter.chunk.auto.ReadWrite;
import ru.yandex.chemodan.util.yasm.monitor.YasmMonitor;
import ru.yandex.commune.db.shard2.ShardManager2;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.misc.worker.spring.DelayingWorkerServiceBeanSupport;

/**
 * @author yashunsky
 */
public class DynamicPropertiesMigrationControl extends DelayingWorkerServiceBeanSupport implements MigrationControl {
    private final DynamicProperty<Integer> lockDelay;
    private final DynamicProperty<Integer> roPause;
    private final DynamicProperty<Integer> replicationPause;
    private final DynamicProperty<Integer> logsCheckDelay;
    private final DynamicProperty<Integer> cleaningDelay;

    private final DynamicProperty<Double> initialRps;
    private final DynamicProperty<Double> incStepRps;
    private final DynamicProperty<Double> decStepRps;
    private final DynamicProperty<Double> readCoeff;
    private final DynamicProperty<Double> writeCoeff;
    private final DynamicProperty<Double> minUsage;
    private final DynamicProperty<Boolean> autoEnabled;
    private final DynamicProperty<Integer> chunkSize;
    private final DynamicProperty<Long> maxTimeSlot;
    private final DynamicProperty<Double> maxRps;
    private final DynamicProperty<ListF<String>> maxRpsByShard;
    private final DynamicProperty<Long> maxAwaitTime;
    private final DynamicProperty<ListF<String>> ignoredHosts;

    private final DynamicProperty<Boolean> allowedToCleanDestinationShard;
    private final DynamicProperty<Boolean> logsCheckEnabled;
    private final DynamicProperty<Boolean> awaitMigrationPossible;

    private final AutoContextAwareRwRateLimiter autoRateLimiter;
    private final ChunkRateLimiter readLimiterReference;
    private final ChunkRateLimiter writeLimiterReference;

    private final Semaphore sharpeiSemaphore;

    public DynamicPropertiesMigrationControl(ShardManager2 dataShardManager, YasmMonitor yasmMonitor,
            MetricsConfiguration metricsConfiguration, RateLimitersMetrics rateLimitersMetrics,
            Duration meterInterval, Duration averageInterval, Duration maintenancePeriod, int sharpeiMaxConnections,
            String sharpeiId)
    {

        String prefix = sharpeiId + "-user-migration-";

        String dPrefix = prefix + "delays--";
        this.lockDelay = DynamicProperty.cons(dPrefix + "lock[s]", 120);
        this.roPause = DynamicProperty.cons(dPrefix + "ro[s]", 90);
        this.replicationPause = DynamicProperty.cons(dPrefix + "replication[s]", 30);
        this.logsCheckDelay = DynamicProperty.cons(dPrefix + "logsCheck[h]", 1);
        this.cleaningDelay = DynamicProperty.cons(dPrefix + "cleaning[h]", 24 * 7 * 2);

        String rlPrefix = prefix + "rate-limit-";
        this.initialRps = DynamicProperty.cons(rlPrefix + "A--initial[rps]", 10.0);
        this.incStepRps = DynamicProperty.cons(rlPrefix + "B--inc-step[rps]", 0.08);
        this.decStepRps = DynamicProperty.cons(rlPrefix + "C--dec-step[rps]", 0.08);
        this.readCoeff = DynamicProperty.cons(rlPrefix + "D--read-coeff", 40.0);
        this.writeCoeff = DynamicProperty.cons(rlPrefix + "E--write-coeff", 1.0);
        this.minUsage = DynamicProperty.cons(rlPrefix + "F--min-usage", 0.8);
        this.autoEnabled = DynamicProperty.cons(rlPrefix + "G--auto-enabled", false);
        this.chunkSize = DynamicProperty.cons(rlPrefix + "I--chunkSize", 100);
        this.maxTimeSlot = DynamicProperty.cons(rlPrefix + "J--maxTimeSlot[ms]", 10000L);
        this.maxRps = DynamicProperty.cons(rlPrefix + "K--max[rps]", 100.0);
        this.maxRpsByShard = DynamicProperty.cons(rlPrefix + "L--max-by-shard[rps]", Cf.list());
        this.maxAwaitTime = DynamicProperty.cons(rlPrefix + "M--max-await-time[s]", 60L);
        this.ignoredHosts = DynamicProperty.cons(rlPrefix + "N--ignored-hosts", Cf.list());

        String fPrefix = prefix + "flags--";
        this.allowedToCleanDestinationShard = DynamicProperty.cons(fPrefix + "cleanDstShard", false);
        this.logsCheckEnabled = DynamicProperty.cons(fPrefix + "checkLogs", false);
        this.awaitMigrationPossible = DynamicProperty.cons(fPrefix + "await-migration-possible", true);

        this.autoRateLimiter = new AutoContextAwareRwRateLimiter(
                dataShardManager, yasmMonitor, metricsConfiguration, rateLimitersMetrics,
                rateSupplier(initialRps::get), this::getMaxRate,
                rateSupplier(incStepRps::get), rateSupplier(decStepRps::get),
                readCoeff::get, writeCoeff::get, minUsage::get, ignoredHosts::get, autoEnabled::get,
                maxTimeSlot::get, () -> Option.of(maxAwaitTime.get() * 1000), chunkSize::get,
                meterInterval, averageInterval);

        this.readLimiterReference = new RateLimiterReference(autoRateLimiter, ReadWrite.READ);
        this.writeLimiterReference = new RateLimiterReference(autoRateLimiter, ReadWrite.WRITE);
        this.sharpeiSemaphore = new Semaphore(sharpeiMaxConnections);

        setDelay(maintenancePeriod);
    }

    private double getMaxRate(int shardId) {
        return maxRpsByShard.get().map(element -> Cf.x(element.split(":"))).toMap(
                parts -> Integer.parseInt(parts.first()),
                parts -> Double.parseDouble(parts.last()))
                .getOrElse(shardId, maxRps.get()) * chunkSize.get() / 1000;
    }

    private Function0<Double> rateSupplier(Function0<Double> rpsSupplier) {
        return () -> rpsSupplier.apply() * chunkSize.get() / 1000;
    }

    @Override
    public ChunkRateLimiter getSelectLimiter() {
        return readLimiterReference;
    }

    @Override
    public ChunkRateLimiter getInsertLimiter() {
        return writeLimiterReference;
    }

    @Override
    public ChunkRateLimiter getDeleteLimiter() {
        return writeLimiterReference;
    }

    @Override
    public Semaphore getSharpeiSemaphore() {
        return sharpeiSemaphore;
    }

    @Override
    public Duration getRoPause() {
        return Duration.standardSeconds(roPause.get());
    }

    @Override
    public Duration getReplicationPause() {
        return Duration.standardSeconds(replicationPause.get());
    }

    @Override
    public boolean isAllowedToCleanDestinationShard() {
        return allowedToCleanDestinationShard.get();
    }

    @Override
    public Duration getLockDelay() {
        return Duration.standardSeconds(lockDelay.get());
    }

    @Override
    public boolean isLogsCheckEnabled() {
        return logsCheckEnabled.get();
    }

    @Override
    public Duration getLogsCheckDelay() {
        return Duration.standardHours(logsCheckDelay.get());
    }

    @Override
    public Duration getCleaningDelay() {
        return Duration.standardHours(cleaningDelay.get());
    }

    @Override
    protected void execute() throws Exception {
        autoRateLimiter.maintain();
    }

    @Override
    public boolean isMigrationPossible(ListF<Integer> shardIds) {
        return !awaitMigrationPossible.get() || autoRateLimiter.isMigrationPossible(shardIds);
    }

    public void updateMetrics() {
        autoRateLimiter.updateMetrics();
    }
}
