package ru.yandex.chemodan.app.migrator.migration;

import java.util.concurrent.TimeUnit;

import net.jodah.failsafe.RetryPolicy;
import org.joda.time.Duration;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.chemodan.app.dataapi.DataApiCoreContextConfiguration;
import ru.yandex.chemodan.app.dataapi.DataApiYtContextConfiguration;
import ru.yandex.chemodan.app.dataapi.api.deltas.DeltasJdbcDao;
import ru.yandex.chemodan.app.dataapi.core.dao.data.DataRecordsJdbcDao;
import ru.yandex.chemodan.app.dataapi.core.dao.data.DatabasesJdbcDao;
import ru.yandex.chemodan.app.dataapi.core.dao.data.DeletedDatabasesJdbcDao;
import ru.yandex.chemodan.app.dataapi.core.dao.usermeta.UserMetaManager;
import ru.yandex.chemodan.app.dataapi.core.dao.usermeta.UserMetaManagerWithSemaphore;
import ru.yandex.chemodan.app.dataapi.core.dao.usermeta.migration.MigrationControl;
import ru.yandex.chemodan.app.dataapi.core.mdssnapshot.MdsSnapshotReferenceJdbcDao;
import ru.yandex.chemodan.app.dataapi.web.admin.DataApiCoreAdminContextConfiguration;
import ru.yandex.chemodan.app.migrator.MigratorTaskQueueName;
import ru.yandex.chemodan.app.migrator.control.BazingaDatasyncUserMigrationManager;
import ru.yandex.chemodan.app.migrator.control.BazingaHelper;
import ru.yandex.chemodan.app.migrator.control.DynamicPropertiesMigrationControl;
import ru.yandex.chemodan.app.migrator.sharpei.cleanup.SharpeiCleanupContextConfiguration;
import ru.yandex.chemodan.app.migrator.tasks.DatasyncMigrationCleanUpTask;
import ru.yandex.chemodan.app.migrator.tasks.DatasyncMigrationLockTask;
import ru.yandex.chemodan.app.migrator.tasks.DatasyncMigrationLogsCheckTask;
import ru.yandex.chemodan.app.migrator.tasks.DatasyncMigrationTask;
import ru.yandex.chemodan.app.migrator.tasks.DatasyncMigrationYtSupplyTask;
import ru.yandex.chemodan.bazinga.BazingaWorkerTaskQueues;
import ru.yandex.chemodan.bazinga.ChemodanBazingaWorkerContextConfiguration;
import ru.yandex.chemodan.ratelimiter.chunk.auto.AutoRateLimiterSupportContextConfiguration;
import ru.yandex.chemodan.ratelimiter.chunk.auto.MetricsConfiguration;
import ru.yandex.chemodan.ratelimiter.chunk.auto.RateLimitersMetrics;
import ru.yandex.chemodan.util.yasm.monitor.YasmMonitor;
import ru.yandex.chemodan.util.yt.YtHelper;
import ru.yandex.commune.bazinga.BazingaTaskManager;
import ru.yandex.commune.bazinga.BazingaWorkerApp;
import ru.yandex.commune.bazinga.impl.storage.BazingaStorage;
import ru.yandex.commune.bazinga.pg.worker.PgBazingaWorkerConfiguration;
import ru.yandex.commune.bazinga.scheduler.TaskQueue;
import ru.yandex.commune.db.shard2.ShardManager2;
import ru.yandex.commune.zk2.ZkPath;
import ru.yandex.commune.zk2.primitives.observer.ZkPathObserver;
import ru.yandex.inside.yt.kosher.Yt;

/**
 * @author yashunsky
 */
@Configuration
@Import({
        ChemodanBazingaWorkerContextConfiguration.class,
        DataApiCoreContextConfiguration.class,
        DataApiCoreAdminContextConfiguration.class,
        DataApiYtContextConfiguration.class,
        AutoRateLimiterSupportContextConfiguration.class,
        SharpeiCleanupContextConfiguration.class,
})
public class MigratorContextConfiguration {

    @Bean
    public PgBazingaWorkerConfiguration pgBazingaWorkerConfiguration() {
        return new PgBazingaWorkerConfiguration(Cf.list(MigratorTaskQueueName.CRON));
    }

    @Bean
    public BazingaWorkerTaskQueues taskQueues(
            @Value("${migrator.migrate.tasks.count}")
                    int migrateThreadCount,
            @Value("${migrator.migrate.tasks.queue}")
                    int migrateQueueCount,
            @Value("${migrator.cleanup.tasks.count}")
                    int cleanUpThreadCount,
            @Value("${migrator.cleanup.tasks.queue}")
                    int cleanUpQueueCount,
            @Value("${migrator.sharpei-cleanup.tasks.count}")
                    int sharpeiCleanUpThreadCount,
            @Value("${migrator.sharpei-cleanup.tasks.queue}")
                    int sharpeiCleanUpQueueCount,
            @Value("${migrator.yt-supply.tasks.count}")
                    int supplyThreadCount,
            @Value("${migrator.yt-supply.tasks.queue}")
                    int supplyQueueCount)
    {
        TaskQueue migrateQueue = new TaskQueue(MigratorTaskQueueName.MIGRATION, migrateThreadCount, migrateQueueCount);
        TaskQueue cleanUpQueue = new TaskQueue(MigratorTaskQueueName.CLEAN_UP, cleanUpThreadCount, cleanUpQueueCount);
        TaskQueue sharpeiCleanup = new TaskQueue(MigratorTaskQueueName.SHARPEI_CLEAN_UP, sharpeiCleanUpThreadCount, sharpeiCleanUpQueueCount);
        TaskQueue ytSupplyQueue = new TaskQueue(MigratorTaskQueueName.YT_SUPPLY, supplyThreadCount, supplyQueueCount);

        return new BazingaWorkerTaskQueues(
                MigratorTaskQueueName.CRON, MigratorTaskQueueName.REGULAR, MigratorTaskQueueName.CPU_INTENSIVE,
                Cf.list(migrateQueue, cleanUpQueue, sharpeiCleanup, ytSupplyQueue));
    }

    @Bean
    public DynamicPropertiesMigrationControl migrationControl(
            ShardManager2 dataShardManager,
            YasmMonitor yasmMonitor,
            MetricsConfiguration metricsConfiguration,
            RateLimitersMetrics rateLimitersMetrics,
            @Value("${migrator.rate-limiter-meter-interval}") Duration meterInterval,
            @Value("${migrator.host-status-averaging-interval}") Duration averageInterval,
            @Value("${migrator.rate-limiter-maintenance-period}") Duration maintenancePeriod,
            @Value("${dataapi.sharpei.http.max.connections}") int sharpeiMaxConnections,
            @Value("${migrator.sharpei-host-identifier}") String sharpeiId)
    {
        return new DynamicPropertiesMigrationControl(
                dataShardManager, yasmMonitor, metricsConfiguration, rateLimitersMetrics,
                meterInterval, averageInterval, maintenancePeriod, sharpeiMaxConnections, sharpeiId);
    }

    @Bean
    public BazingaDatasyncUserMigrationManager bazingaUserMigrationManager(
            UserMigrationManager userMigrationManager,
            MigrationControl migrationControl,
            BazingaTaskManager bazingaTaskManager,
            BazingaWorkerApp bazingaWorkerApp,
            BazingaHelper bazingaHelper,
            @Value("${migrator.sharpei-host-identifier}") String sharpeiId)
    {
        return new BazingaDatasyncUserMigrationManager(
                userMigrationManager, migrationControl, bazingaTaskManager,
                bazingaWorkerApp.getTaskOverridesManager(), bazingaHelper, sharpeiId);
    }

    @Bean
    public DatasyncMigrationTask migrationTask(
            BazingaDatasyncUserMigrationManager migrationManager,
            @Value("${migrator.sharpei-host-identifier}") String sharpeiId)
    {
        return new DatasyncMigrationTask(migrationManager, sharpeiId);
    }

    @Bean
    public DatasyncMigrationLockTask migrationLockTask(
            BazingaDatasyncUserMigrationManager migrationManager,
            @Value("${migrator.sharpei-host-identifier}") String sharpeiId)
    {
        return new DatasyncMigrationLockTask(migrationManager, sharpeiId);
    }

    @Bean
    public DatasyncMigrationLogsCheckTask migrationLogsCheckTask(
            BazingaDatasyncUserMigrationManager migrationManager,
            @Value("${migrator.sharpei-host-identifier}") String sharpeiId)
    {
        return new DatasyncMigrationLogsCheckTask(migrationManager, sharpeiId);
    }

    @Bean
    public DatasyncMigrationCleanUpTask migrationCleanUpTask(
            BazingaDatasyncUserMigrationManager migrationManager,
            @Value("${migrator.sharpei-host-identifier}") String sharpeiId)
    {
        return new DatasyncMigrationCleanUpTask(migrationManager, sharpeiId);
    }

    @Bean
    public DatasyncMigrationYtSupplyTask migrationYtSupplyTask(
            BazingaDatasyncUserMigrationManager migrationManager,
            @Value("${migrator.sharpei-host-identifier}") String sharpeiId)
    {
        return new DatasyncMigrationYtSupplyTask(migrationManager, sharpeiId);
    }

    @Bean
    public UserMigrationManager userMigrationManager(
            UserMetaManager userMetaManager,
            DatabasesJdbcDao databasesDao,
            DeltasJdbcDao deltasDao,
            DataRecordsJdbcDao dataRecordsDao,
            DeletedDatabasesJdbcDao deletedDatabasesDao,
            MdsSnapshotReferenceJdbcDao mdsSnapshotReferenceJdbcDao,
            ZkPathObserver zkPathObserver,
            @Qualifier("zkRoot") ZkPath zkRoot,
            @Value("${sharpei.client-cache-ttl-zk-roots:-}") String sharpeiCacheTtlZkRoots,
            @Value("${dataapi.force-collate-c:-false}") boolean forceCollateC)
    {
        ListF<String> zkRoots = Cf.x(sharpeiCacheTtlZkRoots.split(","));
        return new UserMigrationManager(
                new UserMetaManagerWithSemaphore(userMetaManager), databasesDao, deltasDao, dataRecordsDao,
                deletedDatabasesDao, mdsSnapshotReferenceJdbcDao,
                zkPathObserver, zkRoot, zkRoots, forceCollateC);
    }

    @Bean
    public BazingaHelper bazingaHelper(BazingaStorage bazingaStorage, BazingaTaskManager bazingaTaskManager,
                                       BazingaWorkerApp bazingaWorkerApp,
                                       Yt ytClient,
                                       @Value("${migrator.sharpei-host-identifier}") String sharpeiId,
                                       @Value("${migrator.expected-ready-tasks-count}") int expectedReadyTasksCount,
                                       @Value("${migrator.yt-supply-interval}") Duration ytSupplyInterval,
                                       @Value("${migrator.yt-import.threads-count}") int threadsCount,
                                       @Value("${migrator.yt-import.retries-count}") int retiresCount,
                                       @Value("${migrator.yt-import.retries-interval}") Duration retiresInterval)
    {
        RetryPolicy retryPolicy = new RetryPolicy()
                .withMaxRetries(retiresCount)
                .withDelay(retiresInterval.getMillis(), TimeUnit.MILLISECONDS);
        YtHelper ytHelper = new YtHelper(ytClient, retryPolicy);

        return new BazingaHelper(bazingaStorage, bazingaTaskManager, bazingaWorkerApp.getTaskOverridesManager(),
                ytHelper, sharpeiId, threadsCount, expectedReadyTasksCount, ytSupplyInterval);
    }
}
