package ru.yandex.chemodan.app.migrator.migration;

import java.util.Comparator;
import java.util.concurrent.Semaphore;

import lombok.AllArgsConstructor;
import org.apache.commons.lang3.exception.ExceptionUtils;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.IteratorF;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.function.Function0;
import ru.yandex.bolts.function.Function0V;
import ru.yandex.bolts.function.Function1V;
import ru.yandex.chemodan.app.dataapi.api.data.filter.condition.CollectionIdCondition;
import ru.yandex.chemodan.app.dataapi.api.data.filter.condition.DataCondition;
import ru.yandex.chemodan.app.dataapi.api.data.filter.condition.RecordIdCondition;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataRecord;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataRecordId;
import ru.yandex.chemodan.app.dataapi.api.db.Database;
import ru.yandex.chemodan.app.dataapi.api.db.handle.DatabaseHandle;
import ru.yandex.chemodan.app.dataapi.api.deltas.Delta;
import ru.yandex.chemodan.app.dataapi.api.deltas.DeltasJdbcDao;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.dataapi.core.dao.data.DataRecordsJdbcDao;
import ru.yandex.chemodan.app.dataapi.core.dao.data.DatabasesJdbcDao;
import ru.yandex.chemodan.app.dataapi.core.dao.data.DeletedDatabasesJdbcDao;
import ru.yandex.chemodan.app.dataapi.core.dao.usermeta.MetaUser;
import ru.yandex.chemodan.app.dataapi.core.dao.usermeta.UserMetaManagerWithSemaphore;
import ru.yandex.chemodan.app.dataapi.core.dao.usermeta.migration.MigrationControl;
import ru.yandex.chemodan.app.dataapi.core.dao.usermeta.migration.MigrationSupport;
import ru.yandex.chemodan.app.dataapi.core.dao.usermeta.migration.RateLimiters;
import ru.yandex.chemodan.app.dataapi.core.mdssnapshot.MdsSnapshotReference;
import ru.yandex.chemodan.app.dataapi.core.mdssnapshot.MdsSnapshotReferenceJdbcDao;
import ru.yandex.chemodan.ratelimiter.chunk.ChunkRateLimiter;
import ru.yandex.chemodan.util.retry.RetryManager;
import ru.yandex.chemodan.util.sharpei.ShardUserInfo;
import ru.yandex.chemodan.util.sharpei.SharpeiCachingManager;
import ru.yandex.commune.util.RetryUtils;
import ru.yandex.commune.zk2.ZkPath;
import ru.yandex.commune.zk2.client.Zk;
import ru.yandex.commune.zk2.primitives.observer.ZkPathObserver;
import ru.yandex.misc.db.masterSlave.MasterSlaveContextHolder;
import ru.yandex.misc.db.masterSlave.MasterSlavePolicy;
import ru.yandex.misc.lang.Check;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.thread.ThreadUtils;

/**
 * @author yashunsky
 */

@AllArgsConstructor
public class UserMigrationManager {
    public static final long DELTAS_TO_COPY_LIMIT = 1000;
    private static final int REMOVE_RO_RETRIES = 50;
    private static final int REMOVE_RO_RETRIES_DELAY = 1000;
    private static final int DB_RETRIES = 3;

    private static final Logger logger = LoggerFactory.getLogger(UserMigrationManager.class);

    private static final Comparator<Database> databaseComparator =
            (first, last) -> first.handleValue().compareTo(last.handleValue());
    private static final Comparator<MdsSnapshotReference> snapshotComparator =
            (first, last) -> first.databaseHandle.compareTo(last.databaseHandle);

    private final UserMetaManagerWithSemaphore userMetaManager;

    private final DatabasesJdbcDao databasesDao;
    private final DeltasJdbcDao deltasDao;
    private final DataRecordsJdbcDao dataRecordsDao;
    private final DeletedDatabasesJdbcDao deletedDatabasesDao;
    private final MdsSnapshotReferenceJdbcDao mdsSnapshotReferenceDao;
    private final ZkPathObserver zkPathObserver;
    private final ZkPath zkRoot;
    private final ListF<String> sharpeiCacheTtlZkRoots;

    private final boolean forceCollateC;

    public MigrationResult migrateUser(DataApiUserId uid, int fromShardId, int toShardId, MigrationControl control) {
        checkSarpeiCacheSettings();

        Semaphore sharpeiSemaphore = control.getSharpeiSemaphore();
        int userShard = getUserShard(uid, sharpeiSemaphore);

        if (userShard == toShardId) {
            return new MigrationResult.Skipped();
        }
        if (fromShardId == toShardId) {
            logger.info("Source and destination shard should not match ({}). Migration aborted for user {}",
                    fromShardId, uid);
            return new MigrationResult.Aborted();
        }
        if (fromShardId != userShard) {
            logger.info("User {} data is actually located at shard {}, not at shard {}. Migration aborted",
                    uid, userShard, fromShardId);
            return new MigrationResult.Aborted();
        }

        logger.info("Migrating user {} data from shard {} to shard {}", uid, fromShardId, toShardId);

        boolean migrationStarted = false;
        String hash;

        try {
            doStep(uid, "Enable user read-only", () -> setReadOnly(uid, sharpeiSemaphore));
            doStep(uid, "Wait in read-only", () -> ThreadUtils.sleep(control.getRoPause()));

            int recheckedUserShard = getUserShard(uid, sharpeiSemaphore);
            if (fromShardId != recheckedUserShard) {
                logger.info("User {} shard changed from {} to {} during ro-pause. Migration aborted",
                        uid, userShard, recheckedUserShard);
                doStep(uid, "Disable user read-only due aborted migration", () -> removeReadOnly(uid, sharpeiSemaphore));
                return new MigrationResult.Aborted();
            }

            boolean anyDataOnDestinationShard = doStep(uid, "Check for data on destination shard",
                    () -> isAnyUserDataOnShard(uid, toShardId));
            if (anyDataOnDestinationShard) {
                if (control.isAllowedToCleanDestinationShard()) {
                    doStep(uid, "Clean destination shard",
                            () -> deleteUserData(uid, toShardId, control.getLimiters()));
                } else {
                    logger.info("User {} has data on destination shard ({}). Migration aborted", uid, toShardId);
                    doStep(uid, "Disable user read-only due aborted migration",
                            () -> removeReadOnly(uid, sharpeiSemaphore));
                    return new MigrationResult.Aborted();
                }
            }

            migrationStarted = true;
            doStep(uid, "Copy user data to destination shard",
                    () -> copyUserData(uid, fromShardId, toShardId, control.getLimiters()));

            ThreadUtils.sleep(control.getReplicationPause());

            hash = doStep(uid, "Check user data consistency",
                    () -> compareUserData(uid, fromShardId, toShardId, control.getSelectLimiter()));
            doStep(uid, "Assign new shard to user and disable read-only",
                    () -> updateShardAndRemoveRoSafe(uid, toShardId, sharpeiSemaphore));
        } catch (Throwable t) {
            try {
                doStep(uid, "Rollback shard and disable user read-only due to error",
                        () -> updateShardAndRemoveRoSafe(uid, fromShardId, sharpeiSemaphore));
            } catch (Throwable tr) {
                tr.addSuppressed(t);
                throw tr;
            }

            if (migrationStarted) {
                try {
                    doStep(uid, "Clean destination shard after unsuccessful migration attempt",
                            () -> deleteUserData(uid, toShardId, control.getLimiters()));
                } catch (Throwable tr) {
                    tr.addSuppressed(t);
                    throw tr;
                }
            }

            throw t;
        }
        logger.info("User {} data successfully migrated from shard {} to shard {}", uid, fromShardId, toShardId);
        return new MigrationResult.Done(hash);
    }

    public void checkSarpeiCacheSettings() {
        String dynPropertyName = SharpeiCachingManager.ttl.name();
        Zk zk = zkPathObserver.zk();
        sharpeiCacheTtlZkRoots.forEach(root -> {
            ZkPath path = new ZkPath("/" + root)
                    .child(zkRoot.getName()).child("dyn-properties").child(dynPropertyName + "-int");
            boolean longTllOverrideExists =
                    zk.getDataIfExists(path).map(String::new).map(Integer::valueOf).exists(ttl -> ttl > 1);
            Check.isFalse(longTllOverrideExists, dynPropertyName + " > 1 min for " + root);
        });
    }

    public void removeReadOnly(DataApiUserId uid, Semaphore semaphore) {
        updateShardAndRemoveRoSafe(uid, Option.empty(), semaphore);
    }

    public MigrationResult deleteUserData(DataApiUserId uid, int shardId, RateLimiters limiters) {
        return deleteUserData(uid, shardId, Option.empty(), () -> {}, limiters);
    }

    public MigrationResult deleteUserData(
            DataApiUserId uid, int shardId, String expectedHash, Function0V onHashChecked, RateLimiters limiters)
    {
        return deleteUserData(uid, shardId, Option.of(expectedHash), onHashChecked, limiters);
    }

    private MigrationResult deleteUserData(
            DataApiUserId uid, int shardId, Option<String> expectedHash, Function0V onHashChecked,
            RateLimiters limiters)
    {
        ChunkRateLimiter selectLimiter = limiters.forSelect();
        ChunkRateLimiter deleteLimiter = limiters.forDelete();

        int userShard = getUserShard(uid, limiters.forSharpei());

        if (userShard == shardId) {
            logger.info("User {} is actually located at shard {}. Cleaning aborted.", uid, shardId);
            return new MigrationResult.CleaningAborted();
        }

        if (expectedHash.isPresent()) {
            if (!isHashOkInAnyPossibleWay(uid, shardId, expectedHash.get(), selectLimiter)) {
                return new MigrationResult.CleaningAborted();
            }
            logger.info("User {} data hash on shard {} successfully checked", uid, shardId);
            onHashChecked.apply();
        }

        logger.info("Deleting user {} data from shard {}", uid, shardId);

        MigrationSupport.doWithShard(uid, shardId, () -> {
            doDeleteDatabases(uid, databasesDao.find(uid, true), databasesDao::delete, deleteLimiter);
            doDeleteDatabases(uid, deletedDatabasesDao.find(uid, true),
                    deletedDatabasesDao::removeFromDeleted, deleteLimiter);
        });

        logger.info("User {} data successfully deleted from shard {}", uid, shardId);

        return new MigrationResult.CleaningDone();
    }

    private boolean isHashOkInAnyPossibleWay(
            DataApiUserId uid, int shardId, String expectedHash, ChunkRateLimiter selectLimiter)
    {
        ListF<Function0<String>> hashResolvers = Cf.list(
                // default
                () -> getUserDataHash(uid, shardId, selectLimiter, true, false),
                // backward compatibility for users migrated before forced record changes
                () -> getUserDataHash(uid, shardId, selectLimiter, true, true),
                // backward compatibility for users, migrated with more then 1000 deltas
                () -> getUserDataHash(uid, shardId, selectLimiter, false, true),
                // this case shouldn't exist, but let it be
                () -> getUserDataHash(uid, shardId, selectLimiter, false, true)
        );

        for (Function0<String> hashResolver : hashResolvers) {
            String hash = hashResolver.apply();
            if (expectedHash.equals(hash)) {
                return true;
            } else {
                logger.info("User {} data from shard {} hash differs from expected ({} vs {}).",
                        uid, shardId, expectedHash, hash);
            }
        }
        logger.info("User {} data from shard {} hash differs from expected. Cleaning aborted", uid, shardId);
        return false;
    }

    private void updateShardAndRemoveRoSafe(DataApiUserId uid, int shard, Semaphore semaphore) {
        updateShardAndRemoveRoSafe(uid, Option.of(shard), semaphore);
    }

    private void updateShardAndRemoveRoSafe(DataApiUserId uid, Option<Integer> shardO, Semaphore semaphore) {
        RetryManager retryManager = new RetryManager().withRetryPolicy(REMOVE_RO_RETRIES, REMOVE_RO_RETRIES_DELAY);
        Function0V action = shardO.isPresent()
                ? () -> userMetaManager.updateShardIdAndReadOnly(uid, shardO.get(), false, semaphore)
                : () -> userMetaManager.updateReadOnly(uid, false, semaphore);

        try {
            retryManager.run(action);
        } catch (RuntimeException e) {
            try {
                MetaUser user = getUserMeta(uid, semaphore);
                if (shardO.exists(shard -> shard != user.getShardId()) || user.isRo()) {
                    throw e;
                }
            } catch (Throwable t) {
                t.addSuppressed(e);
                throw t;
            }
        }
    }

    public int getUserShard(DataApiUserId uid, Semaphore semaphore) {
        return getUserMeta(uid, semaphore).getShardId();
    }

    public Option<Integer> getUserShardO(DataApiUserId uid, Semaphore semaphore) {
        return userMetaManager.findMetaUser(uid, semaphore).map(MetaUser::getShardId);
    }

    public boolean isUserReadOnly(DataApiUserId uid, Semaphore semaphore) {
        return getUserMeta(uid, semaphore).isRo();
    }

    public boolean isUserReadOnlySafe(DataApiUserId uid, Semaphore semaphore) {
        return userMetaManager.findMetaUser(uid, semaphore).map(ShardUserInfo::isRo).orElse(false);
    }

    public void setReadOnly(DataApiUserId uid, Semaphore semaphore) {
        userMetaManager.updateReadOnly(uid, true, semaphore);
    }

    private MetaUser getUserMeta(DataApiUserId uid, Semaphore semaphore) {
        Option<MetaUser> user = userMetaManager.findMetaUser(uid, semaphore);
        Validate.some(user, "User " + uid + " not found in meta storage");
        return user.get();
    }

    private void doStep(DataApiUserId uid, String what, Function0V action) {
        doStep(uid, what, action.asFunction0ReturnNull());
    }

    private <T> T doStep(DataApiUserId uid, String what, Function0<T> action) {
        String message = what + " for " + uid;
        logger.info(message);
        try {
            return action.apply();
        } catch (Throwable t) {
            Throwable cause = ExceptionUtils.getCause(t);
            throw new UserMigrationException(
                    "Failed to " + StringUtils.decapitalize(message) + ": " + t + ", cause: " + cause, t);
        }
    }

    public boolean isAnyUserDataOnShard(DataApiUserId uid, int shardId) {
        return MigrationSupport.doWithShard(uid, shardId, () -> {
            if (databasesDao.findDatabasesCount(uid) > 0) {
                return true;
            } else if (deletedDatabasesDao.findDatabasesCount(uid) > 0) {
                return true;
            }
            return false;
        });
    }

    private void doDeleteDatabases(
            DataApiUserId uid, ListF<Database> databases, Function1V<Database> deleteDbF, ChunkRateLimiter limiter)
    {
        databases.paginate(limiter.getDefaultChunkSize()).forEach(dbs -> {
            ListF<String> handles = dbs.map(Database::handleValue);
            dataRecordsDao.deleteAllRecordFromDatabases(uid, handles, limiter);
            deltasDao.deleteAllForDatabases(uid, handles, limiter);
            mdsSnapshotReferenceDao.delete(uid, handles, limiter);
            dbs.forEach(db -> limiter.acquirePermitAndExecuteV(1, () -> deleteDbF.apply(db)));
        });
    }

    private void doCopyDatabases(DataApiUserId uid, ListF<Database> databases, int fromShardId, int toShardId,
            Function1V<ListF<Database>> insertDbsF, RateLimiters limiters)
    {
        ChunkRateLimiter selectLimiter = limiters.forSelect();
        ChunkRateLimiter insertLimiter = limiters.forInsert();

        databases.paginate(selectLimiter.getDefaultChunkSize()).forEach(dbs -> {

            insertWithRateLimit(uid, toShardId, insertLimiter, dbs.iterator().paginate(dbs.size()), insertDbsF);

            dbs.forEach((database) -> {
                logger.info("Migrating user {} database {}, rev {}, records count {}",
                        uid, database.handleValue(), database.rev, database.meta.recordsCount);
                copyDatabaseDeltas(uid, database, fromShardId, toShardId, limiters);
                copyDatabaseRecords(uid, database, fromShardId, toShardId, limiters);
            });

            copySnapshotReferences(uid, dbs, fromShardId, toShardId, limiters);
        });
    }

    private void copyUserData(DataApiUserId uid, int fromShardId, int toShardId, RateLimiters limiters) {
        MigrationSupport.doWithShard(uid, fromShardId, () -> {
            ChunkRateLimiter selectLimiter = limiters.forSelect();

            ListF<Database> realDatabases = MasterSlaveContextHolder.withPolicy(MasterSlavePolicy.R_SM,
                    () -> getAndThenAcquirePermit(() -> withRetry(() -> databasesDao.find(uid, false)), selectLimiter));
            ListF<Database> deletedDatabases = MasterSlaveContextHolder.withPolicy(MasterSlavePolicy.R_SM,
                    () -> getAndThenAcquirePermit(() -> withRetry(() -> deletedDatabasesDao.find(uid, false)), selectLimiter));

            doCopyDatabases(uid, realDatabases, fromShardId, toShardId,
                    dbs -> withRetry(() -> databasesDao.insertBatch(uid, dbs)), limiters);
            doCopyDatabases(uid, deletedDatabases, fromShardId, toShardId,
                    dbs -> withRetry(() -> deletedDatabasesDao.saveAsDeletedBatch(uid, dbs)), limiters);

            Check.equals(realDatabases.size(), databasesDao.findDatabasesCount(uid),
                    "New database created during copying");
        });
    }

    private String compareUserData(DataApiUserId uid, int fromShardId, int toShardId, ChunkRateLimiter limiter) {
        return getUserDataHash(uid, fromShardId, Option.of(toShardId), limiter, true, false);
    }

    public String getUserDataHash(DataApiUserId uid, int shardId, ChunkRateLimiter limiter,
            boolean onlyLatestDeltas, boolean oldStyleDelta) {
        return getUserDataHash(uid, shardId, Option.empty(), limiter, onlyLatestDeltas, oldStyleDelta);
    }

    private String getUserDataHash(
            DataApiUserId uid, int fromShardId, Option<Integer> toShardId,
            ChunkRateLimiter limiter, boolean onlyLatestDeltas, boolean oldStyleDelta)
    {
        MigrationDigester digester = new MigrationDigester();

        ListF<Database> databases = getCompareAndThenAcquirePermit(uid, fromShardId, toShardId,
                () -> databasesDao.find(uid, false), databaseComparator, limiter);
        databases.forEach(digester::append);

        ListF<DatabaseHandle> handles = databases.map(Database::getDbHandle);

        getCompareAndThenAcquirePermit(uid, fromShardId, toShardId,
                () -> deletedDatabasesDao.find(uid, false), databaseComparator, limiter).forEach(digester::append);

        getCompareAndThenAcquirePermit(uid, fromShardId, toShardId,
                () -> mdsSnapshotReferenceDao.find(uid, handles.map(DatabaseHandle::handleValue)),
                snapshotComparator, limiter).forEach(digester::append);

        databases.forEach(database -> {
            new RecordsIterator(uid, fromShardId, toShardId, limiter, database.dbHandle)
                    .forEachRemaining(records -> {
                        /* comparision completed within resolving if toShardId is present*/
                        records.forEach(digester::append);
                    });
            new DeltasIterator(
                    uid, fromShardId, toShardId, limiter, database, onlyLatestDeltas)
                    .forEachRemaining(deltas -> {
                        /* comparision completed within resolving if toShardId is present*/
                        if (oldStyleDelta) {
                            deltas.map(OldStyleDelta::new).forEach(digester::append);
                        } else {
                            deltas.forEach(digester::append);
                        }
                    });
        });

        return digester.end().base64();
    }

    private void copyDatabaseDeltas(
            DataApiUserId uid, Database db, int fromShardId, int toShardId, RateLimiters limiters)
    {
        DeltasIterator dataSource =
                new DeltasIterator(uid, fromShardId, Option.empty(), limiters.forSelect(), db, true);
        Function1V<ListF<Delta>> insertDeltasF =
                deltas -> withRetry(() -> deltasDao.insertBatch(uid, deltas.toTuple2List((d) -> Tuple2.tuple(db.handleValue(), d))));

        insertWithRateLimit(uid, toShardId, limiters.forInsert(), dataSource, insertDeltasF);
    }

    private void copyDatabaseRecords(
            DataApiUserId uid, Database db, int fromShardId, int toShardId, RateLimiters limiters)
    {
        RecordsIterator dataSource =
                new RecordsIterator(uid, fromShardId, Option.empty(), limiters.forSelect(), db.dbHandle);
        Function1V<ListF<DataRecord>> insertRecordsF =
                records -> withRetry(() -> dataRecordsDao.insertBatched(uid, db.dbRef(), records)
                );

        insertWithRateLimit(uid, toShardId, limiters.forInsert(), dataSource, insertRecordsF);
    }

    private void copySnapshotReferences(
            DataApiUserId uid, ListF<Database> databases, int fromShardId, int toShardId, RateLimiters limiters)
    {
        ChunkRateLimiter selectLimiter = limiters.forSelect();
        ChunkRateLimiter insertLimiter = limiters.forInsert();

        Function0<ListF<MdsSnapshotReference>> getF =
                () -> withRetry(() -> mdsSnapshotReferenceDao.find(uid, databases.map(Database::handleValue)));

        IteratorF<ListF<MdsSnapshotReference>> datasource = MigrationSupport.doWithShard(
                uid, fromShardId, () -> getAndThenAcquirePermit(getF, selectLimiter)
                .iterator().paginate(selectLimiter.getDefaultChunkSize()));
        Function1V<ListF<MdsSnapshotReference>> insertSnapshotsF =
                (references) -> withRetry(() -> mdsSnapshotReferenceDao.insertBatch(uid, references));

        insertWithRateLimit(uid, toShardId, insertLimiter, datasource, insertSnapshotsF);
    }

    private <T> ListF<T> getCompareAndThenAcquirePermit(
            DataApiUserId uid, int fromShardId, Option<Integer> toShardId,
            Function0<ListF<T>> getF, Comparator<T> comparator, ChunkRateLimiter limiter)
    {
        return MigrationSupport.getWithOptionalConsistencyCheck(uid, fromShardId, toShardId,
                () -> getAndThenAcquirePermit(() -> getF.apply().sorted(comparator), limiter));
    }

    private <T> ListF<T> getAndThenAcquirePermit(Function0<ListF<T>> getF, ChunkRateLimiter limiter) {
        ListF<T> data = limiter.acquirePermitAndExecute(0, getF);
        limiter.acquirePermitAndExecuteV(data.size(), () -> {});
        return data;
    }

    private <T> void insertWithRateLimit(
            DataApiUserId uid, int toShardId, ChunkRateLimiter limiter,
            IteratorF<ListF<T>> dataSource, Function1V<ListF<T>> insertF)
    {
        Function1V<ListF<T>> insertWithRateLimitF =
                items -> MigrationSupport.doWithShardPolicy(uid, toShardId, MasterSlavePolicy.RW_M,
                        () -> limiter.acquirePermitAndExecuteV(items.size(), () -> insertF.apply(items)));
        dataSource.forEachRemaining(insertWithRateLimitF);
    };

    private <T> T withRetry(Function0<T> action) {
        return RetryUtils.retryOrThrow(logger, DB_RETRIES, action);
    }

    private void withRetry(Function0V action) {
        withRetry(action.asFunction0ReturnNull());
    }

    private class RecordsIterator extends MigrationShardIterator<DataRecord> {
        private final DatabaseHandle dbHandle;
        private volatile Option<DataRecordId> lastRecordId = Option.empty();

        private RecordsIterator(DataApiUserId uid, int primaryShardId,
                Option<Integer> secondaryShardId, ChunkRateLimiter rateLimiter, DatabaseHandle dbHandle)
        {
            super(uid, primaryShardId, secondaryShardId, rateLimiter);
            this.dbHandle = dbHandle;
        }

        @Override
        ListF<DataRecord> getDataInner() {
            // records' handle is overwritten because source and destination record dbId may vary
            // e.g. .ext.maps_common@ymapspointshistory1 -> ymapspointshistory1
            // first format is wrong but exists in db

            return rateLimiter.acquirePermitAndExecute(chunkSize -> withRetry(
                () -> dataRecordsDao.findNext(uid, dbHandle,
                    CollectionIdCondition.all(), RecordIdCondition.all(), DataCondition.all(),
                    lastRecordId, chunkSize, forceCollateC).map(record -> record.withHandle(dbHandle))));
        }

        @Override
        void updateLimits(ListF<DataRecord> data) {
            if (data.isNotEmpty()) {
                lastRecordId = Option.of(data.last().id);
            }
        }
    }

    private class DeltasIterator extends MigrationShardIterator<Delta> {
        private final DatabaseHandle dbHandle;
        private volatile long nextRev;

        private DeltasIterator(DataApiUserId uid, int primaryShardId,
                Option<Integer> secondaryShardId, ChunkRateLimiter rateLimiter, Database db, boolean onlyLatestDeltas)
        {
            super(uid, primaryShardId, secondaryShardId, rateLimiter);
            this.dbHandle = db.dbHandle;
            this.nextRev = onlyLatestDeltas ? db.rev - DELTAS_TO_COPY_LIMIT : 0;
        }

        @Override
        ListF<Delta> getDataInner() {
            return rateLimiter.acquirePermitAndExecute(
                    chunkSize -> withRetry(() -> deltasDao.findAfterRevision(uid, dbHandle, nextRev, chunkSize)));
        }

        @Override
        void updateLimits(ListF<Delta> data) {
            if (data.isNotEmpty()) {
                nextRev = data.last().rev.get() + 1;
            }
        }
    }
}
