package ru.yandex.chemodan.app.migrator.migration;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Semaphore;

import org.joda.time.Duration;
import org.joda.time.Instant;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;
import org.mockito.Mockito;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function0;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataRecord;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataRecordId;
import ru.yandex.chemodan.app.dataapi.api.db.Database;
import ru.yandex.chemodan.app.dataapi.api.db.DatabaseDeletionMode;
import ru.yandex.chemodan.app.dataapi.api.db.handle.DatabaseHandle;
import ru.yandex.chemodan.app.dataapi.api.db.ref.GlobalDatabaseRef;
import ru.yandex.chemodan.app.dataapi.api.db.ref.SpecialDatabases;
import ru.yandex.chemodan.app.dataapi.api.db.ref.UserDatabaseSpec;
import ru.yandex.chemodan.app.dataapi.api.deltas.Delta;
import ru.yandex.chemodan.app.dataapi.api.deltas.DeltaUtilsTest;
import ru.yandex.chemodan.app.dataapi.api.deltas.DeltasJdbcDao;
import ru.yandex.chemodan.app.dataapi.api.deltas.RecordChangeType;
import ru.yandex.chemodan.app.dataapi.api.deltas.RevisionCheckMode;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.dataapi.core.dao.ForcedUserShardInfoHolder;
import ru.yandex.chemodan.app.dataapi.core.dao.ShardPartitionDataSource;
import ru.yandex.chemodan.app.dataapi.core.dao.UserShardId;
import ru.yandex.chemodan.app.dataapi.core.dao.UserShardInfo;
import ru.yandex.chemodan.app.dataapi.core.dao.data.DataRecordsJdbcDao;
import ru.yandex.chemodan.app.dataapi.core.dao.data.DatabasesJdbcDao;
import ru.yandex.chemodan.app.dataapi.core.dao.data.DeletedDatabasesJdbcDao;
import ru.yandex.chemodan.app.dataapi.core.dao.support.TransactionUserShardIdHolder;
import ru.yandex.chemodan.app.dataapi.core.dao.test.ActivateDataApiEmbeddedPg;
import ru.yandex.chemodan.app.dataapi.core.dao.usermeta.UserMetaManager;
import ru.yandex.chemodan.app.dataapi.core.dao.usermeta.UserMetaManagerWithSemaphore;
import ru.yandex.chemodan.app.dataapi.core.dao.usermeta.migration.MigrationControl;
import ru.yandex.chemodan.app.dataapi.core.dao.usermeta.migration.RateLimiters;
import ru.yandex.chemodan.app.dataapi.core.datasources.disk.DiskDataSource;
import ru.yandex.chemodan.app.dataapi.core.manager.DataApiManager;
import ru.yandex.chemodan.app.dataapi.core.mdssnapshot.MdsSnapshotReference;
import ru.yandex.chemodan.app.dataapi.core.mdssnapshot.MdsSnapshotReferenceJdbcDao;
import ru.yandex.chemodan.app.dataapi.test.DataApiTestSupport;
import ru.yandex.chemodan.ratelimiter.chunk.ChunkRateLimiter;
import ru.yandex.commune.db.shard2.Shard2;
import ru.yandex.commune.zk2.ZkPath;
import ru.yandex.commune.zk2.primitives.observer.ZkPathObserver;
import ru.yandex.misc.lang.DefaultObject;
import ru.yandex.misc.test.Assert;
import ru.yandex.misc.thread.ThreadUtils;


/**
 * @author yashunsky
 */

@ActivateDataApiEmbeddedPg
public class UserMigrationManagerTest extends DataApiTestSupport {
    @Autowired
    private DataApiManager dataApiManager;
    @Autowired
    private DiskDataSource diskDataSource;
    @Autowired
    private UserMetaManager userMetaManager;
    @Autowired
    private ShardPartitionDataSource dataSource;
    @Autowired
    private DatabasesJdbcDao databasesDao;
    @Autowired
    private DeletedDatabasesJdbcDao deletedDatabasesDao;
    @Autowired
    private DeltasJdbcDao deltasDao;
    @Autowired
    private DataRecordsJdbcDao dataRecordsDao;
    @Autowired
    private MdsSnapshotReferenceJdbcDao mdsSnapshotReferenceDao;
    @Autowired
    private ZkPathObserver zkPathObserver;
    @Autowired
    private ZkPath zkRoot;

    private UserMigrationManager userMigrationManager;

    private MigrationControl control;
    private int fromShardId;
    private int toShardId;
    private DataApiUserId uid;

    @Before
    public void setup() {
        SpecialDatabases.setMdsSnapshotableAnyForTest(true);

        ListF<String> zkRoots = Cf.list();

        userMigrationManager = new UserMigrationManager(
                new UserMetaManagerWithSemaphore(userMetaManager), databasesDao, deltasDao, dataRecordsDao,
                deletedDatabasesDao, mdsSnapshotReferenceDao,
                zkPathObserver, zkRoot, zkRoots, false);

        control = createMockMigrationControl();
        ListF<Shard2> shards = dataSource.shardManager.shards().toList();
        Assert.gt(shards.size(), 1);

        fromShardId = shards.first().getShardInfo().getId();
        toShardId = shards.last().getShardInfo().getId();

        uid = createRandomCleanUser();
    }

    @After
    public void teardown() {
        SpecialDatabases.setMdsSnapshotableAnyForTest(false);
    }

    @Test
    public void migrateUserData() {
        saveUserToShard(uid, fromShardId);

        int deltasToCopy = (int) UserMigrationManager.DELTAS_TO_COPY_LIMIT;
        int recordsCount = deltasToCopy + 10;

        DatabaseHandle handle = createDatabaseWithSequentialCreatedRecords(uid, "db_1", createRecords(1, recordsCount));
        createDatabaseWithSequentialCreatedRecords(uid, "db_2", createRecords(2, 10));
        diskDataSource.deleteDatabases(uid, new GlobalDatabaseRef("db_2"), DatabaseDeletionMode.MARK_DELETED);

        Option<String> key = Option.of("key");
        MdsSnapshotReference mdsSnapshotReference =
                new MdsSnapshotReference(handle.handleValue(), 1, Instant.now(), key, uid);
        mdsSnapshotReferenceDao.insert(mdsSnapshotReference);

        DatabasesData databasesBefore = loadDatabases(uid);

        userMigrationManager.migrateUser(uid, fromShardId, toShardId, control);

        DatabasesData databasesAfter = getFromShard(uid, toShardId, () -> loadDatabases(uid));
        Assert.equals(databasesBefore.deletedDbs, databasesAfter.deletedDbs);

        Assert.hasSize(1, databasesBefore.realDbs);
        Assert.hasSize(1, databasesAfter.realDbs);

        DatabaseData dbBefore = databasesBefore.realDbs.single();
        DatabaseData dbAfter = databasesAfter.realDbs.single();

        Assert.equals(dbBefore.db, dbAfter.db);
        Assert.equals(dbBefore.records, dbAfter.records);
        Assert.equals(dbBefore.mdsSnapshots, dbAfter.mdsSnapshots);

        Assert.hasSize(recordsCount, dbBefore.deltas);
        Assert.hasSize(deltasToCopy, dbAfter.deltas);

        long lastDelta = dbAfter.deltas.filterMap(delta -> delta.newRev).max();

        Assert.equals(lastDelta, (long) recordsCount);

        Assert.equals(dbBefore.deltas.drop(10), dbAfter.deltas);
    }

    @Test
    public void deleteUserData() {
        saveUserToShard(uid, fromShardId);

        DatabaseHandle db1 = createDatabaseWithSequentialCreatedRecords(uid, "db_1", createRecords(1, 10));
        DatabaseHandle db2 = createDatabaseWithSequentialCreatedRecords(uid, "db_2", createRecords(2, 10));
        diskDataSource.deleteDatabases(uid, new GlobalDatabaseRef("db_2"), DatabaseDeletionMode.MARK_DELETED);

        String hash = userMigrationManager.getUserDataHash(
                uid, fromShardId, control.getSelectLimiter(), true, false);

        saveUserToShard(uid, toShardId);

        CompletableFuture<Boolean> hashChecked = new CompletableFuture<>();

        userMigrationManager.deleteUserData(
                uid, fromShardId, hash, () -> hashChecked.complete(true), control.getLimiters());

        Assert.isTrue(hashChecked.isDone());

        ForcedUserShardInfoHolder.set(new UserShardInfo(uid, false, fromShardId));
        try {
            Assert.isEmpty(diskDataSource.listDatabases(uid));
            Assert.isEmpty(diskDataSource.listDeletedDatabases(uid));
            Assert.isEmpty(dataRecordsDao.find(uid, db1));
            Assert.isEmpty(dataRecordsDao.find(uid, db2));
            Assert.isEmpty(deltasDao.findAfterRevision(uid, db1, 0, 100500));
            Assert.isEmpty(deltasDao.findAfterRevision(uid, db2, 0, 100500));
            Assert.isEmpty(mdsSnapshotReferenceDao.find(uid, Cf.list(db1.handle, db2.handle)));
        } finally {
            ForcedUserShardInfoHolder.remove();
        }
    }

    @Test
    public void abortMigrationIfAnyDataOnDestinationShard() {
        saveUserToShard(uid, toShardId);
        createDatabaseWithSequentialCreatedRecords(uid, "db_1", createRecords(1, 10));
        saveUserToShard(uid, fromShardId);

        MigrationResult result = userMigrationManager.migrateUser(uid, fromShardId, toShardId, control);
        Assert.isInstance(result, MigrationResult.Aborted.class);
    }

    @Test
    public void abortMigrationIfUserShardChanged() throws Exception {
        saveUserToShard(uid, fromShardId);

        Mockito.when(control.getRoPause()).thenReturn(Duration.standardSeconds(1));

        CompletableFuture<MigrationResult> result = CompletableFuture.supplyAsync(
                () -> userMigrationManager.migrateUser(uid, fromShardId, toShardId, control));

        ThreadUtils.sleep(500);
        saveUserToShard(uid, toShardId);
        ThreadUtils.sleep(Duration.standardSeconds(1));

        Assert.isInstance(result.get(), MigrationResult.Aborted.class);
    }

    @Test
    public void abortCleaningIfDataChanged() {
        saveUserToShard(uid, fromShardId);

        createDatabaseWithSequentialCreatedRecords(uid, "db_1", createRecords(1, 10));
        createDatabaseWithSequentialCreatedRecords(uid, "db_2", createRecords(2, 10));
        diskDataSource.deleteDatabases(uid, new GlobalDatabaseRef("db_2"), DatabaseDeletionMode.MARK_DELETED);

        MigrationResult migrationResult =
                userMigrationManager.migrateUser(uid, fromShardId, toShardId, control);

        String hash = ((MigrationResult.Done) migrationResult).getHash();

        saveUserToShard(uid, fromShardId);
        createDatabaseWithSequentialCreatedRecords(uid, "db_3", createRecords(3, 1));
        saveUserToShard(uid, toShardId);

        MigrationResult cleaningResult = userMigrationManager.deleteUserData(
                uid, fromShardId, hash, () -> Assert.fail("onHashChecked called"), control.getLimiters());

        Assert.isInstance(cleaningResult, MigrationResult.CleaningAborted.class);
    }

    private MigrationControl createMockMigrationControl() {
        ChunkRateLimiter rateLimiter = new ChunkRateLimiter() {
            @Override
            public int getDefaultChunkSize() {
                return 2;
            }

            @Override
            public <T> T acquirePermitAndExecute(int chunkSize, Function<Integer, T> action) {
                return action.apply(chunkSize);
            }
        };

        Semaphore sharpeiSemaphore = new Semaphore(1);

        MigrationControl migrationControl = Mockito.mock(MigrationControl.class);
        Mockito.when(migrationControl.getSelectLimiter()).thenReturn(rateLimiter);
        Mockito.when(migrationControl.getInsertLimiter()).thenReturn(rateLimiter);
        Mockito.when(migrationControl.getDeleteLimiter()).thenReturn(rateLimiter);
        Mockito.when(migrationControl.getSharpeiSemaphore()).thenReturn(sharpeiSemaphore);
        Mockito.when(migrationControl.getRoPause()).thenReturn(Duration.ZERO);
        Mockito.when(migrationControl.getReplicationPause()).thenReturn(Duration.ZERO);

        Mockito.when(migrationControl.getLimiters())
                .thenReturn(new RateLimiters(rateLimiter, rateLimiter, rateLimiter, sharpeiSemaphore));

        Mockito.when(migrationControl.isAllowedToCleanDestinationShard()).thenReturn(false);

        return migrationControl;
    }

    private <T> T getFromShard(DataApiUserId uid, int shardId, Function0<T> getData) {
        TransactionUserShardIdHolder.set(new UserShardId(uid, shardId));
        try {
            return getData.apply();
        } finally {
            TransactionUserShardIdHolder.remove();
        }
    }

    private ListF<String> createRecords(int dbId, int count) {
        return Cf.range(0, count).map(rId -> "db_" + dbId + "_rec_" + rId);
    }

    private void saveUserToShard(DataApiUserId uid, int shardId) {
        userMetaManager.registerIfNotExists(uid);
        userMetaManager.updateShardIdAndReadOnly(uid, shardId, false);
    }

    private DatabaseHandle createDatabaseWithSequentialCreatedRecords(DataApiUserId uid, String dbId,
                                                                        ListF<String> recordIds)
    {
        GlobalDatabaseRef dbRef = new GlobalDatabaseRef(dbId);
        UserDatabaseSpec databaseSpec = new UserDatabaseSpec(uid, dbRef);

        dataApiManager.createDatabase(databaseSpec);

        ListF<Delta> deltas = recordIds.map(id ->
                new Delta(DeltaUtilsTest.createEmptyChange(
                        new DataRecordId(dbRef.consHandle(dbId), "col", id), RecordChangeType.INSERT)
                )
        );
        dataApiManager.applyDeltas(databaseSpec, 0, RevisionCheckMode.PER_RECORD, deltas);

        return dataApiManager.getAndInitCopyOnWrite(databaseSpec).get().dbHandle;
    }

    private DatabasesData loadDatabases(DataApiUserId uid) {
        return new DatabasesData(
                diskDataSource.listDatabases(uid).map(this::loadDatabase),
                diskDataSource.listDeletedDatabases(uid).map(this::loadDatabase));
    }

    private DatabaseData loadDatabase(Database database) {
        return new DatabaseData(database,
                deltasDao.findAfterRevision(database.uid, database.dbHandle, 0, 100500),
                dataRecordsDao.find(database.uid, database.dbHandle),
                mdsSnapshotReferenceDao.find(database.uid, Cf.list(database.handleValue())));
    }

    private static class DatabaseData extends DefaultObject {
        public final Database db;
        public final ListF<Delta> deltas;
        public final ListF<DataRecord> records;
        public final ListF<MdsSnapshotReference> mdsSnapshots;

        public DatabaseData(Database db,
                            ListF<Delta> deltas, ListF<DataRecord> records,
                            ListF<MdsSnapshotReference> mdsSnapshots)
        {
            this.db = db;
            this.deltas = deltas;
            this.records = records;
            this.mdsSnapshots = mdsSnapshots;
        }
    }

    protected static class DatabasesData extends DefaultObject {
        public final ListF<DatabaseData> realDbs;
        public final ListF<DatabaseData> deletedDbs;

        public DatabasesData(ListF<DatabaseData> realDbs, ListF<DatabaseData> deletedDbs) {
            this.realDbs = realDbs;
            this.deletedDbs = deletedDbs;
        }
    }

}
