package ru.yandex.chemodan.app.migrator.sharpei.cleanup;

import java.util.concurrent.Semaphore;

import org.joda.time.Duration;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.ratelimiter.chunk.ChunkRateLimiter;
import ru.yandex.chemodan.ratelimiter.chunk.ChunkRateLimiterWithMeter;
import ru.yandex.chemodan.ratelimiter.chunk.auto.MasterSlave;
import ru.yandex.chemodan.ratelimiter.chunk.auto.MetricsConfiguration;
import ru.yandex.chemodan.ratelimiter.chunk.auto.RateLimitersMetrics;
import ru.yandex.chemodan.util.yasm.monitor.YasmMetric;
import ru.yandex.chemodan.util.yasm.monitor.YasmMetricRanges;
import ru.yandex.chemodan.util.yasm.monitor.YasmMonitor;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.misc.worker.spring.DelayingWorkerServiceBeanSupport;

/**
 * @author yashunsky
 */
public class BazingaSharpeiCleanupControl extends DelayingWorkerServiceBeanSupport implements SharpeiCleanupControl {
    private final YasmMonitor yasmMonitor;
    private final MetricsConfiguration metricsConfiguration;
    private final RateLimitersMetrics rateLimitersMetrics;

    private final DynamicProperty<Boolean> cleanupEnabled;
    private final DynamicProperty<Boolean> shuffledStart;
    private final DynamicProperty<Integer> roPause;
    private final DynamicProperty<Integer> initialPause;
    private final DynamicProperty<Integer> lockDelay;
    private final DynamicProperty<Double> nominalRps;
    private final DynamicProperty<Long> maxTimeSlot;
    private final DynamicProperty<Long> maxAwaitTime;
    private final DynamicProperty<ListF<String>> ignoredHosts;

    private final Semaphore sharpeiSemaphore;
    private final ChunkRateLimiterWithMeter rateLimiter;

    private final String sharpeiId;
    private final ListF<String> sharpeiHosts;
    private final Duration averageInterval;

    private volatile boolean sharpeiIsOk;

    public BazingaSharpeiCleanupControl(YasmMonitor yasmMonitor,
                                        MetricsConfiguration metricsConfiguration,
                                        RateLimitersMetrics rateLimitersMetrics,
                                        int sharpeiMaxConnections,
                                        String sharpeiId,
                                        ListF<String> sharpeiHosts,
                                        Duration meterInterval,
                                        Duration averageInterval,
                                        Duration maintenancePeriod)
    {
        this.yasmMonitor = yasmMonitor;
        this.metricsConfiguration = metricsConfiguration;
        this.rateLimitersMetrics = rateLimitersMetrics;

        this.sharpeiId = sharpeiId;
        this.sharpeiHosts = sharpeiHosts;
        this.averageInterval = averageInterval;

        this.sharpeiIsOk = false;

        String prefix = this.sharpeiId + "-sharpei-cleanup-";

        this.cleanupEnabled = DynamicProperty.cons(prefix + "enabled", false);
        this.shuffledStart = DynamicProperty.cons(prefix + "shuffeled-start", true);

        String dPrefix = prefix + "delays-";
        this.lockDelay = DynamicProperty.cons(dPrefix + "lock[s]", 120);
        this.roPause = DynamicProperty.cons(dPrefix + "ro[s]", 90);
        this.initialPause = DynamicProperty.cons(dPrefix + "initial[s]", 90);

        String rlPrefix = prefix + "rate-limit-";
        this.nominalRps = DynamicProperty.cons(rlPrefix + "-initial[rps]", 10.0);
        this.maxTimeSlot = DynamicProperty.cons(rlPrefix + "-maxTimeSlot[ms]", 10000L);
        this.maxAwaitTime = DynamicProperty.cons(rlPrefix + "-max-await-time[s]", 60L);
        this.ignoredHosts = DynamicProperty.cons(rlPrefix + "-ignored-hosts", Cf.list());

        this.sharpeiSemaphore = new Semaphore(sharpeiMaxConnections);
        this.rateLimiter = new ChunkRateLimiterWithMeter(
                this::getRate, maxTimeSlot::get, () -> Option.of(this.maxAwaitTime.get()), () -> 1, meterInterval);

        setDelay(maintenancePeriod);
    }

    private double getRate() {
        return sharpeiIsOk ? (nominalRps.get() / 1000) : 0;
    }

    @Override
    protected void execute() throws Exception {
        sharpeiIsOk = checkIsSharpeiOk();
        rateLimitersMetrics.rates.set(getRate(), sharpeiId + "-cleanup-target-rate");
        rateLimitersMetrics.rates.set(rateLimiter.getRealRate(), sharpeiId + "-cleanup-real-rate");
    }

    private boolean checkIsSharpeiOk() {
        ListF<String> hosts = sharpeiHosts.filter(host -> !ignoredHosts.get().containsTs(host));
        for (String host: hosts) {
            if (yasmMonitor.getHostStatus(host, getMetricRanges(host), averageInterval).isCrit()) {
                return false;
            }
        }
        return true;
    }

    // вбил этот костыль, потому что базки шарпея не умеют отдавать метрику dead tup в нужном формате
    // ожидается что в дальнейшем этот код просто выпилят из репы
    private MapF<YasmMetric, YasmMetricRanges> getMetricRanges(String host) {
        return metricsConfiguration.getRanges(host, MasterSlave.MASTER).filterKeys(metric -> metric != YasmMetric.PG_DEAD_TUP);
    }

    @Override
    public Semaphore getSharpeiSemaphore() {
        return sharpeiSemaphore;
    }

    @Override
    public Duration getRoPause() {
        return Duration.standardSeconds(roPause.get());
    }

    @Override
    public Duration getInitialPause() {
        return Duration.standardSeconds(initialPause.get());
    }

    @Override
    public Duration getLockDelay() {
        return Duration.standardSeconds(lockDelay.get());
    }

    @Override
    public ChunkRateLimiter getRateLimiter() {
        return rateLimiter;
    }

    @Override
    public boolean shuffledStart() {
        return shuffledStart.get();
    }

    @Override
    public boolean cleanupEnable() {
        return cleanupEnabled.get();
    }
}
