package ru.yandex.chemodan.app.migrator.sharpei.cleanup;

import org.joda.time.Duration;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.migrator.control.BazingaHelper;
import ru.yandex.chemodan.app.migrator.control.BazingaSharpeiCleanupManager;
import ru.yandex.chemodan.app.migrator.migration.UserMigrationManager;
import ru.yandex.chemodan.app.migrator.tasks.SharpeiCleanupTask;
import ru.yandex.chemodan.app.migrator.tasks.SharpeiCleanupYtSupplyTask;
import ru.yandex.chemodan.queller.utils.ConductorUtils;
import ru.yandex.chemodan.ratelimiter.chunk.auto.MetricsConfiguration;
import ru.yandex.chemodan.ratelimiter.chunk.auto.RateLimitersMetrics;
import ru.yandex.chemodan.util.http.HttpClientConfigurator;
import ru.yandex.chemodan.util.yasm.monitor.YasmAggrClient;
import ru.yandex.chemodan.util.yasm.monitor.YasmMetricExtractor;
import ru.yandex.chemodan.util.yasm.monitor.YasmMonitor;
import ru.yandex.chemodan.util.yasm.monitor.YasmMonitorHostsSupplier;
import ru.yandex.commune.alive2.location.LocationResolver;
import ru.yandex.commune.bazinga.BazingaTaskManager;

/**
 * @author yashunsky
 */
@Configuration
@Import({
        SharpeiJdbcContextConfiguration.class,
})
public class SharpeiCleanupContextConfiguration {

    @Bean
    public YasmMonitorHostsSupplier sharpeiHostsSupplier(LocationResolver locationResolver,
                                                         @Value("${sharpei.pg.hosts}") String groupsOrHosts)
    {
        ListF<String> hosts = ConductorUtils.resolveHostsFromString(locationResolver, groupsOrHosts, Option.empty());
        return () -> hosts;
    }

    @Bean
    public YasmMonitor sharpeiCleanupYasmMonitor(
            @Qualifier("yasmMonitorHttpClientConfigurator") HttpClientConfigurator httpClientConfigurator,
            YasmMonitorHostsSupplier sharpeiHostsSupplier,
            @Value("${host-status-maintenance-period}") Duration maintenancePeriod,
            @Value("${host-status-averaging-interval}") Duration averageInterval)
    {
        YasmAggrClient client = new YasmAggrClient(httpClientConfigurator.configure());
        return new YasmMonitor(sharpeiHostsSupplier, client, YasmMetricExtractor.SHARPERI_CLEANING_EXTRACTORS, maintenancePeriod, averageInterval);
    }

    @Bean
    public SharpeiCleanupControl sharpeiCleanupControl(@Qualifier("sharpeiCleanupYasmMonitor") YasmMonitor yasmMonitor,
                                                       MetricsConfiguration metricsConfiguration,
                                                       RateLimitersMetrics rateLimitersMetrics,
                                                       YasmMonitorHostsSupplier sharpeiHostsSupplier,
                                                       @Value("${migrator.rate-limiter-meter-interval}") Duration meterInterval,
                                                       @Value("${migrator.host-status-averaging-interval}") Duration averageInterval,
                                                       @Value("${migrator.rate-limiter-maintenance-period}") Duration maintenancePeriod,
                                                       @Value("${dataapi.sharpei.http.max.connections}") int sharpeiMaxConnections,
                                                       @Value("${migrator.sharpei-host-identifier}") String sharpeiId)
    {
        return new BazingaSharpeiCleanupControl(
                yasmMonitor, metricsConfiguration, rateLimitersMetrics, sharpeiMaxConnections, sharpeiId,
                sharpeiHostsSupplier.getHosts(), meterInterval, averageInterval, maintenancePeriod);
    }

    @Bean
    public SharpeiCleanupManager sharpeiCleanupManager(SharpeiCleanupDao sharpeiCleanupDao,
                                                       UserMigrationManager userMigrationManager,
                                                       SharpeiCleanupControl sharpeiCleanupControl)
    {
        return new SharpeiCleanupManager(sharpeiCleanupDao, userMigrationManager, sharpeiCleanupControl);
    }

    @Bean
    public BazingaSharpeiCleanupManager bazingaSharpeiCleanupManager(SharpeiCleanupControl sharpeiCleanupControl,
                                                                     SharpeiCleanupManager sharpeiCleanupManager,
                                                                     BazingaTaskManager bazingaTaskManager,
                                                                     BazingaHelper bazingaHelper,
                                                                     @Value("${migrator.sharpei-host-identifier}") String sharpeiId)
    {
        return new BazingaSharpeiCleanupManager(sharpeiCleanupControl, sharpeiCleanupManager,
                bazingaTaskManager, bazingaHelper, sharpeiId);
    }

    @Bean
    public SharpeiCleanupTask sharpeiCleanupTask(BazingaSharpeiCleanupManager bazingaSharpeiCleanupManager,
                                                 @Value("${migrator.sharpei-host-identifier}") String sharpeiId)
    {
        return new SharpeiCleanupTask(bazingaSharpeiCleanupManager, sharpeiId);
    }

    @Bean
    public SharpeiCleanupYtSupplyTask sharpeiCleanupYtSupplyTask(BazingaSharpeiCleanupManager bazingaSharpeiCleanupManager,
                                                                 @Value("${migrator.sharpei-host-identifier}") String sharpeiId)
    {
        return new SharpeiCleanupYtSupplyTask(bazingaSharpeiCleanupManager, sharpeiId);
    }
}
